#pragma once

#include "abstract.h"
#include "snapshot.h"

#include <drive/backend/device_snapshot/config/config.h>

#include <library/cpp/object_factory/object_factory.h>

#include <rtline/library/rect_hash/rect_hash.h>
#include <rtline/util/auto_actualization.h>
#include <rtline/util/algorithm/iterator.h>

class TConstDevicesSnapshot {
protected:
    using TSnapshots = TMap<TString, TRTDeviceSnapshot>;
    using TCarsCount = TMap<TString, ui32>;

protected:
    TAtomicSharedPtr<TSnapshots> SnapshotsByIMEI = new TSnapshots;
    TAtomicSharedPtr<TSnapshots> SnapshotsById = new TSnapshots;
    TAtomicSharedPtr<TSnapshots> SignalqOnlySnapshotsById = new TSnapshots;
    TAtomicSharedPtr<TCarsCount> AreaCarsCount = new TCarsCount;
    TAtomicSharedPtr<TCarsCount> FutureAreaCarsCount = new TCarsCount;

public:
    TConstDevicesSnapshot() = default;

    TSet<TString> GetNearestIds(const TGeoCoord& c, const ui32 count, const bool withIncorrectLocation, const bool withClusters, const TSet<TString>* preparedIds = nullptr) const;

    ui32 GetAreaCarsCount(const TString& areaId) const {
        auto it = AreaCarsCount->find(areaId);
        if (it == AreaCarsCount->end()) {
            return 0;
        }
        return it->second;
    }

    ui32 GetFutureAreaCarsCount(const TString& areaId) const {
        auto it = FutureAreaCarsCount->find(areaId);
        if (it == FutureAreaCarsCount->end()) {
            return 0;
        }
        return it->second;
    }

    const TSnapshots& ByIMEI() const {
        return *SnapshotsByIMEI;
    }

    const TSnapshots& ById() const {
        return *SnapshotsById;
    }

    const TSnapshots& SignalqOnlyById() const {
        return *SignalqOnlySnapshotsById;
    }

    template <class TAction>
    void ApplyForSnapshots(const TSet<TString>& carIds, TAction& action) const {
        if (carIds.size() > ById().size() / 10) {
            auto itIds = carIds.begin();
            auto itSnapshot = begin();
            for (; itIds != carIds.end(); ++itIds) {
                if (Advance(itSnapshot, end(), *itIds)) {
                    action(*itIds, itSnapshot->second);
                }
            }
        } else {
            for (auto&& i : carIds) {
                auto it = find(i);
                if (it != end()) {
                    action(i, it->second);
                }
            }
        }
    }

    TMap<TString, TRTDeviceSnapshot>::const_iterator find(const TString& key) const {
        return SnapshotsById->find(key);
    }

    TMap<TString, TRTDeviceSnapshot>::const_iterator begin() const {
        return SnapshotsById->begin();
    }

    TMap<TString, TRTDeviceSnapshot>::const_iterator end() const {
        return SnapshotsById->end();
    }

    NDrive::IObjectSnapshot::TPtr GetSnapshotPtr(const TString& carId) const {
        auto it = SnapshotsById->find(carId);
        if (it == SnapshotsById->end()) {
            return nullptr;
        }
        return new TRTDeviceSnapshot(it->second);
    }

    const TRTDeviceSnapshot* GetSnapshot(const TString& carId) const {
        auto it = SnapshotsById->find(carId);
        if (it == SnapshotsById->end()) {
            return nullptr;
        }
        return &it->second;
    }

    const TSnapshots& GetSnapshots() const {
        return *SnapshotsById;
    }
};

class TDriveAPI;

class TDevicesSnapshot: public TConstDevicesSnapshot {
public:
    TMap<TString, ui32>& MutableAreaCarsCount() {
        return *AreaCarsCount;
    }

    TMap<TString, TRTDeviceSnapshot>& ByIMEI() {
        return *SnapshotsByIMEI;
    }

    TSnapshots& ById() {
        return *SnapshotsById;
    }

    TSnapshots& SignalqOnlyById() const {
        return *SignalqOnlySnapshotsById;
    }

    void BuildById(const TMap<TString, TString>& imeiToCarId, const TVector<TString>& carIdsOnlySignalq, const NDrive::IServer& server);
    void RemoveDuplicatingSignalqOnlySnapshotsById(const TVector<TString>& signalqWithNewTelematicsCarIds);

    TDevicesSnapshot Clone() const;
};

class TDevicesSnapshotManager
    : public IAutoActualization
    , public IMessageProcessor
{
public:
    struct TState {
        ui64 InsertedAreasEventsCount = Max<ui64>();
        TInstant AreasLastUpdate;
        TInstant HashLastUpdate;
        TMap<TString, TInstant> HeartbeatLastUpdate;
        TMap<TString, TInstant> LocationLastUpdate;
        TMap<NDrive::TSensorId, TInstant> SensorLastUpdate;
    };

    struct THashedDevice {
        THashedDevice(const TString& carId, const TString& modelId, NDrive::TLocation&& location)
            : CarId(carId)
            , ModelId(modelId)
            , Location(std::move(location))
        {
        }

        bool operator<(const THashedDevice& item) const {
            return CarId < item.CarId;
        };

        bool operator==(const THashedDevice& item) const {
            return CarId == item.CarId;
        };

        TString CarId;
        TString ModelId;
        NDrive::TLocation Location;

        TRect<TGeoCoord> GetRect() const {
            return { Location.GetCoord() };
        };
    };

    using THash = TRectHash<TGeoCoord, THashedDevice>;
    using THashPtr = TAtomicSharedPtr<THash>;
    using TOptions = NDevicesSnapshotManager::TOptions;

private:
    const NDrive::IServer& Server;
    TOptions Options;
    TState State;

    static const ui64 BucketsNumber = 3;

    std::atomic<ui64> CurrentIdx = 0;

    ui64 GetCurrentIdx(const ui64 delta = 0) const {
        return (CurrentIdx.load() + delta) % BucketsNumber;
    }

    void SetCurrentIdx(const i64 idx) {
        CurrentIdx.store(idx);
    }

    std::array<TDevicesSnapshot, BucketsNumber> CurrentSnapshots;
    std::array<TRWMutex, BucketsNumber> CurrentSnapshotsLock;

    std::array<THashPtr, BucketsNumber> GeoHash;
    std::array<TRWMutex, BucketsNumber> GeoHashLock;

private:
    void DropCache();

private:
    TMaybe<TInstant> LastRefreshInstant;

protected:
    virtual bool MetricSignal() override;

public:
    TDevicesSnapshotManager(const NDrive::IServer& server, const TOptions& options);
    ~TDevicesSnapshotManager();

    virtual TString Name() const override;
    virtual bool Process(IMessage* message) override;

    virtual bool Refresh() override;
    bool UpdatedTo(TInstant threshold = TInstant::Zero()) const;
    bool Wait(TInstant deadline = TInstant::Max(), TInstant threshold = TInstant::Zero()) const;

    THashPtr GetGeoHash() const;
    TDeviceLocationOptions GetLocationOptions() const;

    TDevicesSnapshot GetSnapshots(const TSet<TString>& carIds) const;
    TConstDevicesSnapshot GetSnapshots() const;

    TRTDeviceSnapshot GetSnapshot(const TString& carId) const;
    TAtomicSharedPtr<TRTDeviceSnapshot> GetSnapshotPtr(const TString& carId) const;
    TAtomicSharedPtr<TRTDeviceSnapshot> FetchSnapshot(const TString& carId, TInstant deadline) const;

public:
    class TDevicesSnapshotSelection {
    private:
        using TLinks = TVector<std::pair<TString, const TRTDeviceSnapshot&>>;
        using TObjects = TVector<std::pair<TString, TRTDeviceSnapshot>>;

    private:
        TConstDevicesSnapshot Snapshot;
        TLinks Selection;

    public:
        class TConstSelectionIterator {
        private:
            TLinks::const_iterator ItA;
            TLinks::const_iterator FinishA;

        public:
            TConstSelectionIterator(TLinks::const_iterator itStartA, TLinks::const_iterator itFinishA)
                : ItA(itStartA)
                , FinishA(itFinishA)
            {
            }

            const TRTDeviceSnapshot* SkipTo(const TString& id);
        };

    public:
        TDevicesSnapshotSelection() = default;
        TDevicesSnapshotSelection(TConstDevicesSnapshot&& snapshot, const TSet<TString>& carIds)
            : Snapshot(snapshot)
        {
            const auto inserter = [this](const TString& id, const TRTDeviceSnapshot& ds) {
                Selection.emplace_back(id, ds);
            };
            Snapshot.ApplyForSnapshots(carIds, inserter);
        }

        TConstSelectionIterator GetIterator() const {
            return TConstSelectionIterator(Selection.begin(), Selection.end());
        }

        const TLinks& GetSelection() const {
            return Selection;
        }
    };

public:
    TDevicesSnapshotSelection GetSnapshotsVector(const TSet<TString>& carIds) const;
};
