#include "abstract.h"

#include <drive/backend/proto/snapshot.pb.h>

#include <library/cpp/json/json_reader.h>

#include <rtline/util/blob_with_header.h>

TBlob NDrive::IObjectSnapshot::SerializeToBlob() const {
    NDrive::NProto::TSnapshotHeader header;
    header.SetType(GetTypeName());
    TBlobWithHeader<NDrive::NProto::TSnapshotHeader> bwh(header, DoSerializeToBlob());
    return bwh.Save();
}

NJson::TJsonValue NDrive::IObjectSnapshot::SerializeToJson() const {
    NJson::TJsonValue jsonResult;
    jsonResult.InsertValue("type", GetTypeName());
    jsonResult.InsertValue("data", DoSerializeToJson());
    return jsonResult;
}

bool NDrive::IObjectSnapshot::DeserializeFromJson(const NJson::TJsonValue& value) {
    return DoDeserializeFromJson(value["data"]);
}

NDrive::IObjectSnapshot::TPtr NDrive::IObjectSnapshot::ConstructFromBlob(const TBlob& data) {
    TBlobWithHeader<NDrive::NProto::TSnapshotHeader> bwh;
    if (!bwh.Load(data)) {
        return nullptr;
    }

    IObjectSnapshot::TPtr result = IObjectSnapshot::TFactory::Construct(bwh.GetHeader().GetType());
    if (!result) {
        return result;
    }
    if (!result->DoDeserializeFromBlob(bwh.GetData())) {
        return nullptr;
    }
    return result;
}

NDrive::IObjectSnapshot::TPtr NDrive::IObjectSnapshot::ConstructFromJson(const NJson::TJsonValue& value) {
    const TString& type = value["type"].GetString();
    IObjectSnapshot::TPtr result = IObjectSnapshot::TFactory::Construct(type);
    if (!result) {
        return result;
    }
    if (!result->DeserializeFromJson(value)) {
        return nullptr;
    }
    return result;
}

NDrive::IObjectSnapshot::TPtr NDrive::IObjectSnapshot::ConstructFromJsonString(const TString& value) {
    NJson::TJsonValue json;
    if (!NJson::ReadJsonFastTree(value, &json)) {
        return nullptr;
    }
    return ConstructFromJson(json);
}
