#include "image.h"

#include <drive/backend/proto/snapshot.pb.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/blob_with_header.h>

#include <util/digest/fnv.h>

namespace NJson {
    template <>
    TJsonValue ToJson(const TImagesSnapshot::TImage& object) {
        NJson::TJsonValue r;

        NJson::InsertField(r, "image_id", object.ImageId);

        NJson::InsertField(r, "marker", object.Marker);
        NJson::InsertField(r, "path", object.Path);
        NJson::InsertField(r, "preview_path", NJson::ToJson(NJson::Nullable(object.PreviewPath)));

        NJson::InsertField(r, "origin", NJson::ToJson(NJson::Nullable(object.Origin)));

        NJson::InsertField(r, "external_id", object.ExternalId);
        NJson::InsertField(r, "md5", object.MD5);

        return r;
    }

    template <>
    bool TryFromJson(const TJsonValue& data, TImagesSnapshot::TImage& result) {
        return NJson::ParseField(data["external_id"], result.ExternalId, /* required = */ true) &&
               NJson::ParseField(data["path"], result.Path, /* required = */ true) &&
               NJson::ParseField(data["preview_path"], result.PreviewPath) &&
               NJson::ParseField(data["marker"], result.Marker, /* required = */ true) &&
               NJson::ParseField(data["md5"], result.MD5, /* required = */ true) &&
               NJson::ParseField(data["image_id"], result.ImageId) &&
               NJson::ParseField(data["origin"], result.Origin);
    }
}

TBlob TImagesSnapshot::DoSerializeToBlob() const {
    NDrive::NProto::TImageSnapshot protoSnapshot;
    if (Comment) {
        protoSnapshot.SetComment(*Comment);
    }
    for (size_t i = 0; i < Images.size(); ++i) {
        const TImage& image = Images[i];
        auto protoImage = protoSnapshot.AddImage();
        protoImage->SetExternalId(image.ExternalId);
        protoImage->SetPath(image.Path);
        if (image.PreviewPath) {
            protoImage->SetPreviewPath(image.PreviewPath);
        }
        protoImage->SetMarker(image.Marker);
        protoImage->SetMD5(image.MD5);
        if (image.ImageId) {
            protoImage->SetImageId(*image.ImageId);
        }
        if (image.Origin) {
            protoImage->SetOrigin(image.Origin);
        }
    }
    return TBlob::FromString(protoSnapshot.SerializeAsString());
}

bool TImagesSnapshot::DoDeserializeFromBlob(const TBlob& data) {
    NDrive::NProto::TImageSnapshot protoSnapshot;
    if (!protoSnapshot.ParseFromArray(data.AsCharPtr(), data.Size())) {
        return false;
    }
    if (protoSnapshot.HasComment()) {
        Comment = protoSnapshot.GetComment();
    }
    if (protoSnapshot.HasExternalId() || protoSnapshot.HasPath() || protoSnapshot.HasMarker() || protoSnapshot.HasMD5()) {
        TImage image;
        image.ExternalId = protoSnapshot.GetExternalId();
        image.Path = protoSnapshot.GetPath();
        image.Marker = protoSnapshot.GetMarker();
        image.MD5 = protoSnapshot.GetMD5();
        Images.push_back(image);
    }
    for (auto&& protoImage : protoSnapshot.GetImage()) {
        TImage image;
        image.ExternalId = protoImage.GetExternalId();
        image.Path = protoImage.GetPath();
        if (protoImage.HasPreviewPath()) {
            image.PreviewPath = protoImage.GetPreviewPath();
        }
        image.Marker = protoImage.GetMarker();
        image.MD5 = protoImage.GetMD5();
        if (protoImage.HasImageId()) {
            image.ImageId = protoImage.GetImageId();
        }
        if (protoImage.HasOrigin()) {
            image.Origin = protoImage.GetOrigin();
        }
        Images.push_back(image);
    }
    return true;
}

NJson::TJsonValue TImagesSnapshot::DoSerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "comment", Comment);
    NJson::TJsonValue& rawImages = result.InsertValue("images", NJson::JSON_ARRAY);
    for (const TImage& image: Images) {
        rawImages.AppendValue(NJson::ToJson(image));
    }
    return result;
}

bool TImagesSnapshot::DoDeserializeFromJson(const NJson::TJsonValue& jsonValue) {
    if (!NJson::ParseField(jsonValue["comment"], Comment)) {
        return false;
    }
    if (jsonValue.Has("external_id") || jsonValue.Has("path") || jsonValue.Has("marker") || jsonValue.Has("md5")) {
        TImage image;
        if (!NJson::TryFromJson(jsonValue, image)) {
            return false;
        }
        Images.push_back(std::move(image));
    }

    const NJson::TJsonValue& images = jsonValue["images"];
    if (images.IsDefined()) {
        return (images.IsArray()) ? NJson::TryFromJson(images, Images) : false;
    }

    return true;
}

ui64 TImagesSnapshot::CalcHash() const {
    ui64 result = 0;
    for (auto&& image : Images) {
        result ^= FnvHash<ui64>(image.Path);
    }
    return result;
}

void TImagesSnapshot::AddImage(TImage&& image) {
    Images.push_back(std::move(image));
}
