#include "snapshot.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/proto/snapshot.pb.h>

#include <drive/library/cpp/threading/future.h>
#include <drive/telematics/protocol/vega.h>

#include <rtline/util/types/string_pool.h>

#include <util/memory/blob.h>
#include <util/random/random.h>

#include <tuple>

static Y_THREAD(TStringPool) TagsPool;

NDrive::IObjectSnapshot::TFactory::TRegistrator<THistoryDeviceSnapshot> THistoryDeviceSnapshot::Registrator("device_snapshot");

namespace {
    template <class T>
    NDrive::TSensor MakeSensor(NDrive::TSensorId id, T value, TTimestamp timestamp) {
        NDrive::TSensor result(id);
        result.Value = std::move(value);
        result.Timestamp = timestamp;
        result.Since = timestamp;
        return result;
    }

    template <class T, class V>
    bool UpdateSensor(const NDrive::TSensor& sensor, V& value, TTimestamp& timestamp) {
        auto optionalValue = sensor.TryConvertTo<T>();
        if (timestamp < sensor.Timestamp && optionalValue) {
            timestamp = sensor.Timestamp;
            value = *optionalValue;
            return true;
        } else {
            return false;
        }
    }
}

void TSensorSnapshot::Iterate(std::function<void(NDrive::TSensor&&)> f) const {
    if (!f) {
        return;
    }
    if (EngineOnTimestamp) {
        f(MakeSensor(CAN_ENGINE_IS_ON, ui64(EngineOn), EngineOnTimestamp));
    }
    if (MileageTimestamp) {
        f(MakeSensor(CAN_ODOMETER_KM, double(Mileage), MileageTimestamp));
    }
    if (FuelLevelTimestamp) {
        f(MakeSensor(CAN_FUEL_LEVEL_P, ui64(FuelLevel), FuelLevelTimestamp));
    }
    if (FuelDistanceTimestamp) {
        f(MakeSensor(CAN_FUEL_DISTANCE_KM, double(FuelDistance), FuelDistanceTimestamp));
    }
    if (FuelVolumeTimestamp) {
        f(MakeSensor(CAN_CUSTOM_FUEL_VOLUME, double(FuelVolume), FuelVolumeTimestamp));
    }
}

bool TSensorSnapshot::Update(const NDrive::TMultiSensor& sensors, const TInstant futureDeadline) {
    bool result = false;
    for (auto&& sensor : sensors) {
        result |= Update(sensor, futureDeadline);
    }
    return result;
}

bool TSensorSnapshot::Update(const NDrive::TSensor& sensor, const TInstant futureDeadline) {
    if (sensor.Timestamp > futureDeadline) {
        return false;
    }
    switch (sensor.Id) {
        case CAN_ENGINE_IS_ON:
            return UpdateSensor<ui64>(sensor, EngineOn, EngineOnTimestamp);
        case CAN_ODOMETER_KM:
            return UpdateSensor<double>(sensor, Mileage, MileageTimestamp);
        case CAN_FUEL_LEVEL_P:
            return UpdateSensor<ui64>(sensor, FuelLevel, FuelLevelTimestamp);
        case CAN_FUEL_DISTANCE_KM:
            return UpdateSensor<double>(sensor, FuelDistance, FuelDistanceTimestamp);
        case CAN_CUSTOM_FUEL_VOLUME:
            return UpdateSensor<double>(sensor, FuelVolume, FuelVolumeTimestamp);
        default:
            return false;
    }
}

TBlob THistoryDeviceSnapshot::DoSerializeToBlob() const {
    NDrive::NProto::THistoryDeviceSnapshot protoSnapshot;
    SensorSnapshot.Iterate([&protoSnapshot](NDrive::TSensor&& i) {
        *protoSnapshot.AddSensor() = i.ToProto();
    });
    if (!!Location) {
        *protoSnapshot.MutableLocation() = NDrive::TLocation(Location).ToProto();
    }
    TSet<TString> localtionTagsSet;
    TSet<TString> localtionTagPrefixesSet;
    if (NDrive::HasServer()) {
        const auto& server = NDrive::GetServer();
        const auto& settings = server.GetSettings();
        const TString locationTags = settings.GetValueDef<TString>("location_tags.store_history_list", "no_price,allow_drop_car");
        StringSplitter(locationTags).SplitBySet(", \n\r").SkipEmpty().Collect(&localtionTagsSet);
        const TString locationTagPrefixes = settings.GetValueDef<TString>("location_tags.store_history_prefixes", "DRIVEANALYTICS-330");
        StringSplitter(locationTagPrefixes).SplitBySet(", \n\r").SkipEmpty().Collect(&localtionTagPrefixesSet);
    }
    for (auto&& i : LocationTagsArray) {
        if (localtionTagsSet.contains(i)) {
            protoSnapshot.AddTagInPoint(i);
        } else {
            for (auto&& p : localtionTagPrefixesSet) {
                if (i.StartsWith(p)) {
                    protoSnapshot.AddTagInPoint(i);
                    break;
                }
            }
        }
    }
    return TBlob::FromString(protoSnapshot.SerializeAsString());
}

bool THistoryDeviceSnapshot::DoDeserializeFromBlob(const TBlob& data) {
    NDrive::NProto::THistoryDeviceSnapshot protoSnapshot;
    if (!protoSnapshot.ParseFromArray(data.AsCharPtr(), data.Size())) {
        return false;
    }

    for (auto&& i : protoSnapshot.GetSensor()) {
        NDrive::TSensor sensor;
        if (!sensor.FromProto(i)) {
            return false;
        }
        SensorSnapshot.Update(sensor, TInstant::Max());
    }

    if (protoSnapshot.HasLocation()) {
        NDrive::TLocation location;
        if (!location.FromProto(protoSnapshot.GetLocation())) {
            return false;
        }
        Location = std::move(location);
    }

    LocationTagsArray.clear();
    LocationTagsArray.reserve(protoSnapshot.TagInPointSize());
    for (auto&& i : protoSnapshot.GetTagInPoint()) {
        LocationTagsArray.push_back(TlsRef(TagsPool).Get(i));
    }

    return true;
}

NJson::TJsonValue THistoryDeviceSnapshot::DoSerializeToJson() const {
    NJson::TJsonValue result;
    NJson::TJsonValue& sensors = result.InsertValue("sensors", NJson::JSON_ARRAY);
    SensorSnapshot.Iterate([&sensors](NDrive::TSensor&& i) {
        sensors.AppendValue(i.ToJson());
    });
    if (!!Location) {
        result.InsertValue("location", NDrive::TLocation(Location).ToJson());
    }
    if (LocationTagsArray.size()) {
        NJson::TJsonValue& tags = result.InsertValue("ptags", NJson::JSON_ARRAY);
        for (auto&& i : LocationTagsArray) {
            tags.AppendValue(i);
        }
    }
    return result;
}

bool THistoryDeviceSnapshot::DoDeserializeFromJson(const NJson::TJsonValue& jsonValue) {
    if (jsonValue["sensors"].IsArray()) {
        for (auto&& i : jsonValue["sensors"].GetArraySafe()) {
            NDrive::TSensor sensor;
            if (!sensor.TryFromJson(i)) {
                return false;
            }
            SensorSnapshot.Update(sensor, TInstant::Max());
        }
    }
    if (jsonValue.Has("location")) {
        NDrive::TLocation location;
        if (!location.TryFromJson(jsonValue["location"])) {
            return false;
        }
        Location = location;
    }
    if (jsonValue.Has("ptags")) {
        if (!jsonValue["ptags"].IsArray()) {
            return false;
        }
        LocationTagsArray.clear();
        LocationTagsArray.reserve(jsonValue["ptags"].GetArraySafe().size());
        for (auto&& i : jsonValue["ptags"].GetArraySafe()) {
            if (!i.IsString()) {
                return false;
            }
            LocationTagsArray.push_back(TlsRef(TagsPool).Get(i.GetString()));
        }
    }
    return true;
}

namespace {
    template <class T>
    const T& SelectLatest(const T& first, const T& second) {
        if (first.Timestamp >= second.Timestamp) {
            return first;
        } else {
            return second;
        }
    }

    template <class T>
    const T& SelectLatest(const T* first, const T& second) {
        if (first) {
            return SelectLatest(*first, second);
        } else {
            return second;
        }
    }
}

void TRTDeviceSnapshot::UpdateHeartbeats(const NDrive::THeartbeat& heartbeat, const TInstant futureDeadline) {
    if (heartbeat.Timestamp < futureDeadline) {
        Heartbeat = SelectLatest(Heartbeat.Get(), heartbeat);
    }
}

void TRTDeviceSnapshot::UpdateConfiguratorHeartbeat(const NDrive::THeartbeat& heartbeat, const TInstant futureDeadline) {
    if (heartbeat.Timestamp < futureDeadline) {
        ConfiguratorHeartbeat = SelectLatest(ConfiguratorHeartbeat.Get(), heartbeat);
    }
}

void TRTDeviceSnapshot::UpdateSensors(NDrive::TMultiSensor& sensors, const TInstant futureDeadline) {
    Y_ASSERT(std::is_sorted(Sensors.begin(), Sensors.end()));
    std::sort(sensors.begin(), sensors.end());

    NDrive::TMultiSensor result;
    auto incoming = sensors.begin();
    for (auto&& existing : Sensors) {
        for (; incoming != sensors.end() && *incoming < existing; ++incoming) {
            result.emplace_back(*incoming);
        }
        NDrive::TSensor current = existing;
        for (; incoming != sensors.end() && *incoming == existing; ++incoming) {
            if (incoming->Timestamp < futureDeadline) {
                current = SelectLatest(current, *incoming);
            }
        }
        result.emplace_back(current);
    }
    for (; incoming != sensors.end(); ++incoming) {
        result.emplace_back(*incoming);
    }
    Sensors = std::move(result);
    SensorSnapshot.Update(sensors, futureDeadline);
}

bool TRTDeviceSnapshot::UpdateLinkedLocation(const NDrive::TLocation& location, const TInstant futureDeadline) {
    return UpdateLocationImpl(LinkedLocation, location, futureDeadline);
}

bool TRTDeviceSnapshot::UpdateLBSLocation(const NDrive::TLocation& location, const TInstant futureDeadline) {
    return UpdateLocationImpl(LBSLocation, location, futureDeadline);
}

bool TRTDeviceSnapshot::UpdateHeadLocation(const NDrive::TLocation& location, const TInstant futureDeadline) {
    return UpdateLocationImpl(HeadLocation, location, futureDeadline);
}

bool TRTDeviceSnapshot::UpdateBeaconsLocation(const NDrive::TLocation& location, const TInstant futureDeadline) {
    if (location.Timestamp > futureDeadline) {
        return false;
    }
    if (!BeaconsLocation) {
        BeaconsLocation = location;
        return true;
    }
    if (BeaconsLocation->Timestamp >= location.Timestamp) {
        return false;
    }

    BeaconsLocation = location;
    return true;
}

bool TRTDeviceSnapshot::UpdateGeocodedLocation(const NDrive::TLocation& location, const TInstant futureDeadline) {
    return UpdateLocationImpl(GeocodedLocation, location, futureDeadline);
}

bool TRTDeviceSnapshot::UpdateRawLocation(const NDrive::TLocation& location, const TInstant futureDeadline) {
    if (location.Type != NDrive::TLocation::EType::GPSCurrent && location.Type != NDrive::TLocation::EType::GPSPrevious) {
        return false;
    }
    UpdateHistoryLocation(location, futureDeadline);
    return UpdateLocationImpl(RawLocation, location, futureDeadline);
}

void TRTDeviceSnapshot::UpdateSignalqStatusAndLocation(const NDrive::NSignalq::TV1StatusesRetrieveResponse::TSerialNumberData &serialNumberData, const TInstant futureDeadline, bool needUpdateHistoryLocation /*= false */) {
    if (serialNumberData.HasLastLocation()) {
        const auto& lastLocation = serialNumberData.GetLastLocationUnsafe();
        const auto& updateAt = lastLocation.GetUpdateAt();
        if (updateAt <= futureDeadline
                && (!SignalqLocationData || updateAt > SignalqLocationData->Location.Timestamp)) {
            SignalqLocationData = NDrive::TSignalqLocation(
                lastLocation.GetLat(),
                lastLocation.GetLon(),
                lastLocation.GetAccuracyM(),
                lastLocation.GetDirectionDeg(),
                NDrive::TLocation::GPSCurrent,
                lastLocation.GetSpeedKmph(),
                lastLocation.GetUpdateAt()
            );
        }
        if (needUpdateHistoryLocation) {
            UpdateHistoryLocation(SignalqLocationData->Location, futureDeadline);
        }
    }

    if (serialNumberData.HasStatus()) {
        const auto& status = serialNumberData.GetStatusUnsafe();
        const auto& statusAt = status.GetStatusAt();
        if (!SignalqStatus
                || (statusAt > SignalqStatus->GetStatusAt())) {
            SignalqStatus = std::move(status);
        }
    }
}

void TRTDeviceSnapshot::UpdateLastSignalqEvent(const NDrive::NSignalq::TV1EventsRetrieveResponse::TSerialNumberData &serialNumberData) {
    if (!serialNumberData.HasEvent()) {
        return;
    }
    auto event = serialNumberData.GetEventUnsafe();
    const auto eventAt = event.GetAt();
    if (!LastSignalqEvent || (eventAt > LastSignalqEvent->GetAt())) {
        LastSignalqEvent = std::move(event);
    }
}

bool THistoryDeviceSnapshot::UpdateHistoryLocation(const NDrive::TLocation& location, const TInstant futureDeadline) {
    if (location.Timestamp > futureDeadline || !location) {
        return false;
    }
    if (Location.Timestamp > location.Timestamp) {
        return false;
    }
    Location = location;
    return true;
}

bool THistoryDeviceSnapshot::UpdateLocationImpl(TMaybe<NDrive::TLocation>& l, const NDrive::TLocation& location, const TInstant futureDeadline) const {
    if (location.Timestamp > futureDeadline || location.IsZero()) {
        return false;
    }
    if (!l) {
        l = location;
        return true;
    }
    if (l->Timestamp > location.Timestamp) {
        return false;
    }
    const TGeoCoord previous = l->GetCoord();
    l = location;
    return previous.GetLengthTo(location.GetCoord()) > 1;
}

void THistoryDeviceSnapshot::SetTagsInPoint(TVector<TString>&& value) {
    Y_ASSERT(std::is_sorted(value.begin(), value.end()));
    LocationTagsArray = std::move(value);
}

NDrive::TLocations TRTDeviceSnapshot::GetLocations() const {
    NDrive::TLocations result;
    if (RawLocation) {
        result.push_back(*RawLocation);
    }
    if (LinkedLocation) {
        result.push_back(*LinkedLocation);
    }
    if (LBSLocation) {
        result.push_back(*LBSLocation);
    }
    if (HeadLocation) {
        result.push_back(*HeadLocation);
    }
    if (BeaconsLocation) {
        result.push_back(*BeaconsLocation);
    }
    if (SignalqLocationData) {
        result.push_back(SignalqLocationData->Location);
    }
    return result;
}

TMaybe<double> THistoryDeviceSnapshot::GetFuelDistance(const TDuration maxAge /*= TDuration::Max()*/) const {
    TInstant timestamp = SensorSnapshot.FuelDistanceTimestamp;
    if (timestamp && timestamp + maxAge > Now()) {
        return SensorSnapshot.FuelDistance;
    }
    if (auto fuelLevel = GetFuelLevel(maxAge)) {
        return *fuelLevel * 5;
    }
    return {};
}

bool THistoryDeviceSnapshot::GetFuelLevel(double& value, const TDuration maxAge) const {
    TInstant timestamp = SensorSnapshot.FuelLevelTimestamp;
    if (timestamp && timestamp + maxAge > Now()) {
        value = SensorSnapshot.FuelLevel;
        return true;
    } else {
        return false;
    }
}

TMaybe<double> THistoryDeviceSnapshot::GetFuelLevel(const TDuration maxAge /*= TDuration::Max()*/) const {
    double result;
    if (GetFuelLevel(result, maxAge)) {
        return result;
    } else {
        return {};
    }
}

TMaybe<double> THistoryDeviceSnapshot::GetFuelVolume(const TDuration maxAge /*= TDuration::Max()*/) const {
    double result;
    if (GetFuelVolume(result, maxAge)) {
        return result;
    } else {
        return {};
    }
}

bool THistoryDeviceSnapshot::GetFuelVolume(double& value, const TDuration maxAge) const {
    TInstant timestamp = SensorSnapshot.FuelLevelTimestamp;
    if (timestamp && timestamp + maxAge > Now()) {
        value = SensorSnapshot.FuelVolume;
        return true;
    } else {
        return false;
    }
}

TMaybe<double> THistoryDeviceSnapshot::GetMileage(const TDuration maxAge /*= TDuration::Max()*/) const {
    double mileage;
    if (GetMileage(mileage, maxAge)) {
        return mileage;
    } else {
        return {};
    }
}

bool THistoryDeviceSnapshot::GetMileage(double& value, const TDuration maxAge) const {
    TInstant timestamp = SensorSnapshot.MileageTimestamp;
    if (timestamp && timestamp + maxAge > Now()) {
        value = SensorSnapshot.Mileage;
        return true;
    } else {
        return false;
    }
}

bool THistoryDeviceSnapshot::GetMileage(double& value, const TInstant since) const {
    TInstant timestamp = SensorSnapshot.MileageTimestamp;
    if (timestamp && timestamp >= since) {
        value = SensorSnapshot.Mileage;
        return true;
    } else {
        return false;
    }
}

TMaybe<NDrive::TLocation> TRTDeviceSnapshot::GetLocation(TStringBuf name, TDuration maxAge, const TLocationOptions& options) const {
    if (name == NDrive::RawLocationName) {
        return GetLocation(maxAge, options);
    }
    if (name == NDrive::BeaconsLocationName) {
        return GetBeaconsLocation(maxAge, options);
    }
    if (name == NDrive::LBSLocationName) {
        return GetLBSLocation(maxAge);
    }
    return {};
}

TMaybe<NDrive::TLocation> TRTDeviceSnapshot::GetLocation(const TDuration maxAge /*= TDuration::Max()*/, const TLocationOptions& options /*= Default<TLocationOptions>()*/) const {
    NDrive::TLocation result;
    if (GetLocation(result, maxAge, options)) {
        return std::move(result);
    } else {
        return {};
    }
}

bool TRTDeviceSnapshot::GetLocation(NDrive::TLocation& value, const TDuration maxAge, const TLocationOptions& options) const {
    const auto now = Now();
    const auto rawLocationTimestamp = RawLocation ? RawLocation->Timestamp : TInstant::Zero();

    const bool hasLocation = HasRawLocation(maxAge, now);
    const bool hasNamedLocation = HasLinkedLocation(maxAge, now);
    const bool hasSignalqLocation = HasSignalqLocation(maxAge, now);

    const auto lbsMaxAge = options.UseLbsFallback ? now - rawLocationTimestamp : maxAge;
    const bool hasLBSLocation = HasLBSLocation(lbsMaxAge, now);
    if (hasLocation && hasNamedLocation) {
        bool mergeThresholdSatisfied = LinkedLocation->GetCoord().GetLengthTo(RawLocation->GetCoord()) < options.MergeThreshold;
        bool tagsSatisfied = true;
        for (auto&& tag : options.BannedTags) {
            if (LocationTags.contains(tag)) {
                tagsSatisfied = false;
                break;
            }
        }
        if (mergeThresholdSatisfied && tagsSatisfied) {
            value = *LinkedLocation;
            value.Timestamp = std::max(value.Timestamp, RawLocation->Timestamp);
            value.Type = RawLocation->IsRealtime() ? value.Type : NDrive::TLocation::LinkedPrevious;
        } else {
            value = *RawLocation;
        }
        return true;
    }
    if (hasLocation) {
        value = *RawLocation;
        return true;
    }
    if (hasNamedLocation) {
        value = *LinkedLocation;
        return true;
    }
    if (hasSignalqLocation) {
        value = SignalqLocationData->Location;
        return true;
    }
    if (hasLBSLocation) {
        value = *LBSLocation;
        return true;
    }
    return false;
}

TMaybe<NDrive::TLocation> THistoryDeviceSnapshot::GetHistoryLocation(const TDuration maxAge) const {
    NDrive::TLocation result;
    if (GetHistoryLocation(result, maxAge)) {
        return result;
    } else {
        return {};
    }
}

bool THistoryDeviceSnapshot::GetHistoryLocation(NDrive::TLocation& value, const TDuration maxAge) const {
    if (!HasHistoryLocation(maxAge)) {
        return false;
    }
    value = Location;
    return true;
}

bool THistoryDeviceSnapshot::HasHistoryLocation(const TDuration maxAge) const {
    return Location && (Location.Timestamp + maxAge > Now());
}

TMaybe<NDrive::TLocation> TRTDeviceSnapshot::GetGeocoded(const TDuration maxAge /*= TDuration::Max()*/, const TLocationOptions& options /*= Default<TLocationOptions>()*/) const {
    auto now = Now();
    bool hasGeocodedLocation = GeocodedLocation && (maxAge == TDuration::Max() || GeocodedLocation->Timestamp + maxAge > now);
    if (!hasGeocodedLocation) {
        return {};
    }
    auto location = GetLocation(maxAge, options);
    if (!location) {
        return GeocodedLocation;
    }
    bool mergeThresholdSatisfied = GeocodedLocation->GetCoord().GetLengthTo(location->GetCoord()) < std::max(options.MergeThreshold, options.GeocodedMergeThreshold);
    if (!mergeThresholdSatisfied) {
        return {};
    }
    return GeocodedLocation;
}

TMaybe<NDrive::THeartbeat> TRTDeviceSnapshot::GetHeartbeat(const TDuration maxAge /*= TDuration::Max()*/) const {
    NDrive::THeartbeat result;
    if (GetHeartbeat(result, maxAge)) {
        return std::move(result);
    } else {
        return {};
    }
}

bool TRTDeviceSnapshot::GetHeartbeat(NDrive::THeartbeat& value, const TDuration maxAge /*= TDuration::Max()*/) const {
    if (Heartbeat) {
        if (maxAge == TDuration::Max() || Heartbeat->Timestamp + maxAge > Now()) {
            value = *Heartbeat;
            return true;
        }
    }
    return false;
}

TMaybe<NDrive::THeartbeat> TRTDeviceSnapshot::GetConfiguratorHeartbeat(const TDuration maxAge /*= TDuration::Max()*/) const {
    if (ConfiguratorHeartbeat && ConfiguratorHeartbeat->Timestamp + maxAge > Now()) {
        return ConfiguratorHeartbeat;
    } else {
        return {};
    }
}

TMaybe<NDrive::TLocation> TRTDeviceSnapshot::GetLBSLocation(const TDuration maxAge) const {
    NDrive::TLocation result;
    if (GetLBSLocation(result, maxAge)) {
        return result;
    } else {
        return{};
    }
}

bool TRTDeviceSnapshot::GetLBSLocation(NDrive::TLocation& value, const TDuration maxAge) const {
    if (!HasLBSLocation(maxAge)) {
        return false;
    }
    value = *LBSLocation;
    return true;
}

bool TRTDeviceSnapshot::HasBeaconsLocation(const TDuration maxAge) const {
    return BeaconsLocation && (maxAge == TDuration::Max() || BeaconsLocation->Timestamp + maxAge > Now());
}

TMaybe<NDrive::TLocation> TRTDeviceSnapshot::GetBeaconsLocation(const TDuration maxAge, const TLocationOptions& options /*= Default<TLocationOptions>()*/) const {
    if (!HasBeaconsLocation(maxAge)) {
        return {};
    }
    auto location = GetLocation(maxAge, options);
    if (!location || !options.MaybeBeaconsMergeThreshold) {
        return BeaconsLocation;
    }
    bool mergeThresholdSatisfied = location->GetCoord().GetLengthTo(BeaconsLocation->GetCoord()) < *options.MaybeBeaconsMergeThreshold;
    if (!mergeThresholdSatisfied) {
        return {};
    }
    return BeaconsLocation;
}

TMaybe<NDrive::TLocation> TRTDeviceSnapshot::GetPreciseLocation(const TDuration maxAge /*= TDuration::Max()*/, const TLocationOptions& options /*= Default<TLocationOptions>()*/) const {
    auto maybeBeaconLocation = GetBeaconsLocation(maxAge, options);
    auto maybeGeocodedLocation = GetGeocoded(maxAge, options);

    if (!maybeBeaconLocation) {
        return maybeGeocodedLocation;
    }
    return maybeBeaconLocation;
}

TMaybe<NDrive::TLocation> TRTDeviceSnapshot::GetBeaconsOrRawLocation(const TDuration maxAge /*= TDuration::Max()*/, const TLocationOptions& options /*= Default<TLocationOptions>()*/) const {
    auto maybeBeaconsLocation = GetBeaconsLocation(maxAge, options);
    auto maybeRawLocation = GetRawLocation(maxAge);
    if (maybeBeaconsLocation) {
        return maybeBeaconsLocation;
    }
    return maybeRawLocation;
}

TMaybe<NDrive::TLocation> TRTDeviceSnapshot::GetRawLocation(const TDuration maxAge /*= TDuration::Max()*/) const {
    NDrive::TLocation result;
    if (GetRawLocation(result, maxAge)) {
        return result;
    } else {
        return {};
    }
}

bool TRTDeviceSnapshot::GetRawLocation(NDrive::TLocation& value, const TDuration maxAge /*= TDuration::Max()*/) const {
    if (!HasRawLocation(maxAge)) {
        return false;
    }
    value = *RawLocation;
    return true;
}

TMaybe<NDrive::TSensor> TRTDeviceSnapshot::GetSensor(NDrive::TSensorId id, const TDuration maxAge /*= TDuration::Max()*/) const {
    Y_ASSERT(std::is_sorted(Sensors.begin(), Sensors.end()));
    auto p = std::lower_bound(Sensors.begin(), Sensors.end(), id);
    if (p == Sensors.end() || !(*p == id)) {
        return {};
    }
    if (p->Timestamp + maxAge < Now()) {
        return {};
    }
    return *p;
}

bool TRTDeviceSnapshot::HasLBSLocation(const TDuration maxAge, const TInstant now) const {
    return LBSLocation && (LBSLocation->Timestamp + maxAge > now);
}

bool TRTDeviceSnapshot::HasRawLocation(const TDuration maxAge, const TInstant now) const {
    return RawLocation && (RawLocation->Timestamp + maxAge > now);
}

bool TRTDeviceSnapshot::HasLinkedLocation(const TDuration maxAge, const TInstant now) const {
    return LinkedLocation && (LinkedLocation->Timestamp + maxAge > now);
}

bool TRTDeviceSnapshot::HasSignalqLocation(const TDuration maxAge, const TInstant now) const {
    return SignalqLocationData && (SignalqLocationData->Location.Timestamp + maxAge > now);
}

TMaybe<TString> TRTDeviceSnapshot::GetBeaconParkingPlaceNumber() const {
    auto beaconsLocation = GetBeaconsLocation();
    if (NJson::TJsonValue jsonInfo; beaconsLocation && NJson::ReadJsonFastTree(beaconsLocation->Content, &jsonInfo) && jsonInfo["parking_place"].IsString()) {
        TVector<TString> parts = SplitString(jsonInfo["parking_place"].GetString(), "|");
        if (parts.size() == 2) {
            return parts[1];
        }
    }
    return {};
}

bool TRTDeviceSnapshot::IsGettingSignalqLocation(const TDuration maxAge, const TInstant now) const {
    return !HasRawLocation(maxAge, now)
           && !HasLinkedLocation(maxAge, now)
           && HasSignalqLocation(maxAge, now);
}

TMaybe<double> TRTDeviceSnapshot::GetSpeed(const TDuration maxAge, const TInstant now) const {
    if (IsGettingSignalqLocation(maxAge, now)) {
        return SignalqLocationData->Speed;
    }

    if (auto speed = GetSensor(VEGA_SPEED, maxAge)) {
        return speed->ConvertTo<double>();
    }

    return {};
}
