#pragma once

#include "abstract.h"

#include <drive/backend/areas/location.h>

#include <drive/library/cpp/taxi/signalq_drivematics_api/definitions.h>
#include <drive/telematics/protocol/sensor.h>
#include <drive/telematics/server/location/heartbeat.h>
#include <drive/telematics/server/location/location.h>
#include <drive/telematics/server/location/names.h>

#include <rtline/library/geometry/coord.h>
#include <rtline/util/algorithm/iterator.h>

namespace NDrive {
    class IServer;
}

struct TDeviceLocationOptions {
    TVector<TString> BannedTags;
    double GeocodedMergeThreshold = 1;
    double MergeThreshold = 0;
    bool UseLbsFallback = false;
    TMaybe<double> MaybeBeaconsMergeThreshold = 200;
};

struct TSensorSnapshot {
public:
    float Mileage = 0;
    float FuelDistance = 0;
    ui8 FuelLevel = 0;
    ui8 FuelVolume = 0;
    bool EngineOn = false;

    TTimestamp MileageTimestamp;
    TTimestamp FuelDistanceTimestamp;
    TTimestamp FuelLevelTimestamp;
    TTimestamp FuelVolumeTimestamp;
    TTimestamp EngineOnTimestamp;

public:
    void Iterate(std::function<void(NDrive::TSensor&&)> f) const;
    bool Update(const NDrive::TMultiSensor& sensors, const TInstant futureDeadline);
    bool Update(const NDrive::TSensor& sensor, const TInstant futureDeadline);
};

class THistoryDeviceSnapshot: public NDrive::IObjectSnapshot {
private:
    static TFactory::TRegistrator<THistoryDeviceSnapshot> Registrator;

protected:
    NDrive::TSimpleLocation Location;
    NDrive::TLocationTagsArray LocationTagsArray;
    TSensorSnapshot SensorSnapshot;

protected:
    virtual NJson::TJsonValue DoSerializeToJson() const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual TBlob DoSerializeToBlob() const override;
    virtual bool DoDeserializeFromBlob(const TBlob& data) override;

public:
    THistoryDeviceSnapshot& SetLocation(const NDrive::TLocation& value) {
        Location = value;
        return *this;
    }

    NDrive::TSimpleLocation GetSimpleLocation() const {
        return Location;
    }

    const NDrive::TLocationTagsArray& GetLocationTagsArray() const {
        return LocationTagsArray;
    }
    const TSensorSnapshot& GetSnapshot() const {
        return SensorSnapshot;
    }

    virtual TString GetTypeName() const override {
        return "device_snapshot";
    }

    TMaybe<double> GetFuelDistance(const TDuration maxAge = TDuration::Max()) const;
    TMaybe<double> GetFuelLevel(const TDuration maxAge = TDuration::Max()) const;
    TMaybe<double> GetFuelVolume(const TDuration maxAge = TDuration::Max()) const;
    TMaybe<double> GetMileage(const TDuration maxAge = TDuration::Max()) const;
    TMaybe<NDrive::TLocation> GetHistoryLocation(const TDuration maxAge = TDuration::Max()) const;

    bool GetFuelLevel(double& value, const TDuration maxAge = TDuration::Max()) const;
    bool GetFuelVolume(double& value, const TDuration maxAge = TDuration::Max()) const;
    bool GetMileage(double& value, const TDuration maxAge = TDuration::Max()) const;
    bool GetMileage(double& value, const TInstant since) const;
    bool GetHistoryLocation(NDrive::TLocation& value, const TDuration maxAge = TDuration::Max()) const;
    bool HasHistoryLocation(const TDuration maxAge = TDuration::Max()) const;

protected:
    void SetTagsInPoint(NDrive::TLocationTagsArray&& value);
    bool UpdateHistoryLocation(const NDrive::TLocation& location, const TInstant futureDeadline);
    bool UpdateLocationImpl(TMaybe<NDrive::TLocation>& l, const NDrive::TLocation& location, const TInstant futureDeadline) const;
};

class TDBTag;

class TRTDeviceSnapshot: public THistoryDeviceSnapshot {
public:
    using TBase = THistoryDeviceSnapshot;
    using TLocationOptions = TDeviceLocationOptions;

private:
    TString Imei;
    NDrive::TMultiSensor Sensors;
    TMaybe<NDrive::THeartbeat> Heartbeat;
    TMaybe<NDrive::THeartbeat> ConfiguratorHeartbeat;
    TMaybe<NDrive::TLocation> LinkedLocation;
    TMaybe<NDrive::TLocation> LBSLocation;
    TMaybe<NDrive::TLocation> HeadLocation;
    TMaybe<NDrive::TLocation> BeaconsLocation;
    TMaybe<NDrive::TLocation> GeocodedLocation;
    TMaybe<NDrive::TLocation> RawLocation;
    TMaybe<NDrive::TSignalqLocation> SignalqLocationData;
    NDrive::TLocationTags LocationTags;
    NDrive::TLocationTags LBSLocationTags;
    TSet<TString> AreaIds;
    TSet<TString> AreaIdsGuarantee;
    R_FIELD(NDrive::TLocationTags, LocationTagsGuarantee);
    R_FIELD(NDrive::TLocationTags, LocationTagsPotentially);
    TVector<TDBTag> HardTagsInPoint;
    TMaybe<TGeoCoord> FutureLocation;
    R_FIELD(TSet<TString>, FutureAreaIds);
    R_FIELD(TSet<TString>, ClusterIds);
    R_OPTIONAL(NDrive::NSignalq::TStatus, SignalqStatus);
    R_OPTIONAL(NDrive::NSignalq::TEvent, LastSignalqEvent);

public:
    using THistoryDeviceSnapshot::THistoryDeviceSnapshot;

    void DropFutureLocation() {
        FutureLocation = {};
        FutureAreaIds.clear();
    }

    bool UpdateFutureLocation(const TGeoCoord& c) {
        if (!FutureLocation || c.GetLengthTo(*FutureLocation) > 1) {
            FutureLocation = c;
            FutureAreaIds.clear();
            return true;
        }
        return false;
    }

    const TVector<TDBTag>& GetHardTagsInPoint() const {
        return HardTagsInPoint;
    }
    void SetHardTags(TVector<TDBTag>&& value) {
        HardTagsInPoint = std::move(value);
    }

    const TSet<TString>& GetAreaIdsGuarantee() const {
        return AreaIdsGuarantee;
    }
    const TSet<TString>& GetAreaIds() const {
        return AreaIds;
    }
    void SetAreaIds(TSet<TString>&& areaIds) {
        AreaIds = std::move(areaIds);
    }
    void SetAreaIdsGuarantee(TSet<TString>&& areaIdsGuarantee) {
        AreaIdsGuarantee = std::move(areaIdsGuarantee);
    }

    const TString& GetImei() const {
        return Imei;
    }
    void SetImei(const TString& imei) {
        Imei = imei;
    }

    const NDrive::TLocationTags& GetTagsInPoint() const {
        return LocationTags;
    }
    const NDrive::TLocationTags& GetLocationTags() const {
        return LocationTags;
    }
    void SetLocationTags(NDrive::TLocationTags&& value) {
        LocationTags = std::move(value);
        SetTagsInPoint({LocationTags.begin(), LocationTags.end()});
    }

    const NDrive::TLocationTags& GetLBSLocationTags() const {
        return LBSLocationTags;
    }
    void SetLBSLocationTags(NDrive::TLocationTags&& value) {
        LBSLocationTags = std::move(value);
    }

    const NDrive::TMultiSensor& GetSensors() const {
        return Sensors;
    }

    void UpdateHeartbeats(const NDrive::THeartbeat& heartbeat, const TInstant futureDeadline);
    void UpdateConfiguratorHeartbeat(const NDrive::THeartbeat& heartbeat, const TInstant futureDeadline);
    void UpdateSensors(NDrive::TMultiSensor& sensors, const TInstant futureDeadline);
    bool UpdateLinkedLocation(const NDrive::TLocation& location, const TInstant futureDeadline);
    bool UpdateLBSLocation(const NDrive::TLocation& location, const TInstant futureDeadline);
    bool UpdateHeadLocation(const NDrive::TLocation& location, const TInstant futureDeadline);
    bool UpdateBeaconsLocation(const NDrive::TLocation& location, const TInstant futureDeadline);
    bool UpdateGeocodedLocation(const NDrive::TLocation& location, const TInstant futureDeadline);
    bool UpdateRawLocation(const NDrive::TLocation& location, const TInstant futureDeadline);

    void UpdateSignalqStatusAndLocation(const NDrive::NSignalq::TV1StatusesRetrieveResponse::TSerialNumberData &serialNumberData, const TInstant futureDeadline, bool needUpdateHistoryLocation = false);
    void UpdateLastSignalqEvent(const NDrive::NSignalq::TV1EventsRetrieveResponse::TSerialNumberData &serialNumberData);

    NDrive::TLocations GetLocations() const;
    TMaybe<NDrive::TLocation> GetLocation(TStringBuf name, TDuration maxAge = TDuration::Max(), const TLocationOptions& options = Default<TLocationOptions>()) const;
    TMaybe<NDrive::TLocation> GetLocation(const TDuration maxAge = TDuration::Max(), const TLocationOptions& options = Default<TLocationOptions>()) const;
    TMaybe<NDrive::TLocation> GetGeocoded(const TDuration maxAge = TDuration::Max(), const TLocationOptions& options = Default<TLocationOptions>()) const;
    TMaybe<NDrive::THeartbeat> GetHeartbeat(const TDuration maxAge = TDuration::Max()) const;
    TMaybe<NDrive::THeartbeat> GetConfiguratorHeartbeat(const TDuration maxAge = TDuration::Max()) const;
    TMaybe<NDrive::TLocation> GetLBSLocation(const TDuration maxAge = TDuration::Max()) const;
    TMaybe<NDrive::TLocation> GetBeaconsLocation(const TDuration maxAge = TDuration::Max(), const TLocationOptions& options = Default<TLocationOptions>()) const;
    TMaybe<NDrive::TLocation> GetRawLocation(const TDuration maxAge = TDuration::Max()) const;
    TMaybe<NDrive::TLocation> GetPreciseLocation(const TDuration maxAge = TDuration::Max(), const TLocationOptions& options = Default<TLocationOptions>()) const;
    TMaybe<NDrive::TLocation> GetBeaconsOrRawLocation(const TDuration maxAge = TDuration::Max(), const TLocationOptions& options = Default<TLocationOptions>()) const;
    TMaybe<NDrive::TSensor> GetSensor(NDrive::TSensorId id, const TDuration maxAge = TDuration::Max()) const;

    bool GetLocation(NDrive::TLocation& value, const TDuration maxAge = TDuration::Max(), const TLocationOptions& options = Default<TLocationOptions>()) const;
    bool GetHeartbeat(NDrive::THeartbeat& value, const TDuration maxAge = TDuration::Max()) const;
    bool GetLBSLocation(NDrive::TLocation& value, const TDuration maxAge = TDuration::Max()) const;
    bool GetRawLocation(NDrive::TLocation& value, const TDuration maxAge = TDuration::Max()) const;
    bool HasLBSLocation(const TDuration maxAge = TDuration::Max(), const TInstant now = Now()) const;
    bool HasRawLocation(const TDuration maxAge = TDuration::Max(), const TInstant now = Now()) const;
    bool HasLinkedLocation(const TDuration maxAge = TDuration::Max(), const TInstant now = Now()) const;
    bool HasSignalqLocation(const TDuration maxAge = TDuration::Max(), const TInstant now = Now()) const;
    bool HasBeaconsLocation(const TDuration maxAge = TDuration::Max()) const;
    TMaybe<TString> GetBeaconParkingPlaceNumber() const;
    bool IsGettingSignalqLocation(const TDuration maxAge = TDuration::Max(), const TInstant now = Now()) const;
    TMaybe<double> GetSpeed(const TDuration maxAge = TDuration::Max(), const TInstant now = Now()) const;
};
