#include "database.h"
#include <drive/backend/database/transaction/tx.h>

namespace NDrive {
    TDistributingBlockEventStats& TDistributingBlockEventStats::AddShowsCount(ui32 toAdd) {
        auto maxShows = static_cast<ui32>(Max<i32>());
        if (ShowsCount > maxShows) {
            ShowsCount = maxShows;
        } else if (toAdd > maxShows - ShowsCount) {
            ShowsCount = maxShows;
        } else {
            ShowsCount += toAdd;
        }
        return *this;
    }

    TDistributingBlockEventStats::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
        Type = GetFieldDecodeIndex("type", decoderBase);
        UserId = GetFieldDecodeIndex("user_id", decoderBase);
        ShowsCount = GetFieldDecodeIndex("shows_count", decoderBase);
    }

    bool TDistributingBlockEventStats::Parse(const NStorage::TTableRecord& row) {
        return TBaseDecoder::DeserializeFromTableRecord(*this, row);
    }

    bool TDistributingBlockEventStats::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, Type);
        READ_DECODER_VALUE(decoder, values, UserId);
        READ_DECODER_VALUE(decoder, values, ShowsCount);
        return true;
    }

    NStorage::TTableRecord TDistributingBlockEventStats::SerializeToTableRecord() const {
        NStorage::TTableRecord row;
        row.Set("type", Type);
        row.Set("user_id", UserId);
        row.Set("shows_count", ShowsCount);
        return row;
    }

    bool TDistributingBlockEventStatsDB::AddShow(const TDistributingBlockEvent& event, NDrive::TEntitySession& session) const {
        auto table = Database->GetTable(GetTableName());
        const auto& type = event.GetType();
        const auto& userId = event.GetUserId();
        TBaseEntityManager::TQueryOptions queryOptions;
        queryOptions.AddGenericCondition("user_id", userId);
        queryOptions.AddGenericCondition("type", type);
        auto result = FetchInfo(session, queryOptions);
        TEntity entity;
        if (!result.empty()) {
            entity = result.begin()->second;
        } else {
            entity.SetUserId(userId);
            entity.SetType(type);
        }
        entity.AddShowsCount(event.GetShowsCount());
        NStorage::TObjectRecordsSet<TEntity> upsertedData;
        if (!Upsert(entity, session, &upsertedData, nullptr)) {
            return false;
        }
        TRecordsSet addedHistory;
        if (!HistoryWriter.AddHistory<NStorage::TObjectRecordsSet<TEntity>>(upsertedData, userId, EObjectHistoryAction::UpdateData, session, &addedHistory)) {
            session.AddErrorMessage(HistoryWriter.GetTableName(), "unable to add history");
            return false;
        }

        if (addedHistory.size() != 1) {
            session.AddErrorMessage(HistoryWriter.GetTableName(), TStringBuilder() << "expected to add 1 row, got " << addedHistory.size());
            return false;
        }
        return true;
    }

    TVector<TDistributingBlockEventStats> TDistributingBlockEventStatsDB::GetShows(const TString& userId) const {
        NDrive::TEntitySession session{Database->CreateTransaction(true)};
        TBaseEntityManager::TQueryOptions queryOptions;
        queryOptions.AddGenericCondition("user_id", userId);
        auto retrievedData = Fetch(session, queryOptions);
        Y_ENSURE(retrievedData, TStringBuilder() << "can't get events: " << session.GetStringReport());
        return *retrievedData;
    }

    IDistributingBlockEventsStorageConfig::TFactory::TRegistrator<TDBDistributingBlockEventsStorageConfig> TDBDistributingBlockEventsStorageConfig::Registrator("db");

    void TDBDistributingBlockEventsStorageConfig::Init(const TYandexConfig::Section& section) {
        TDBEntitiesManagerConfig::Init(&section);
    }

    THolder<IDistributingBlockEventsStorage> TDBDistributingBlockEventsStorageConfig::BuildStorage(const NDrive::IServer& server) const {
        auto db = server.GetDatabase(GetDBName());
        AssertCorrectConfig(db.Get(), TStringBuilder() << "Inexistent DBName for distributing block events storage: " << GetDBName());
        auto historyContext = MakeHolder<THistoryContext>(db);
        return MakeHolder<TDBDistributingBlockEventsStorage>(std::move(historyContext));
    }

    TDBDistributingBlockEventsStorage::TDBDistributingBlockEventsStorage(THolder<IHistoryContext> historyContext)
        : HistoryContext(std::move(historyContext))
        , StatsDB(*HistoryContext)
    {
    }

    NThreading::TFuture<TDistributingBlockEvent::TPtr> TDBDistributingBlockEventsStorage::Store(TDistributingBlockEvent::TPtr event, NDrive::TEntitySession* session) const {
        Y_ENSURE(event);
        Y_ENSURE(session);
        Y_ENSURE(StatsDB.AddShow(*event, *session), TStringBuilder() << "can't store event: " << session->GetStringReport());
        return NThreading::MakeFuture(event);
    }

    NThreading::TFuture<TVector<TDistributingBlockEvent::TPtr>> TDBDistributingBlockEventsStorage::Retrieve(const TString& userId) const {
        TVector<TDistributingBlockEventStats> eventsStats;
        try {
            eventsStats = StatsDB.GetShows(userId);
        } catch (...) {
            return NThreading::MakeErrorFuture<TVector<TDistributingBlockEvent::TPtr>>(std::current_exception());
        }

        TVector<TDistributingBlockEvent::TPtr> result;
        for (const auto& eventStats : eventsStats) {
            auto event = MakeAtomicShared<TDistributingBlockEvent>();
            event->SetType(eventStats.GetType());
            event->SetUserId(eventStats.GetUserId());
            event->SetTimestamp(TInstant::Zero());
            event->SetShowsCount(eventStats.GetShowsCount());
            result.push_back(std::move(event));
        }
        return NThreading::MakeFuture(std::move(result));
    }
}
