#pragma once

#include "common.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/db_entities.h>

#include <rtline/library/storage/structured.h>


namespace NDrive {
    class TDistributingBlockEventStats {
    public:
        TDistributingBlockEventStats& AddShowsCount(ui32 toAdd);
        ui32 GetShowsCount() const {
            return ShowsCount;
        }

    public:
        R_FIELD(TString, Type);
        R_FIELD(TString, UserId);

    private:
        ui32 ShowsCount = 0;

    public:
        class TDecoder: public TBaseDecoder {
        public:
            R_FIELD(i32, Type, -1);
            R_FIELD(i32, UserId, -1);
            R_FIELD(i32, ShowsCount, -1);

        public:
            TDecoder() = default;
            TDecoder(const TMap<TString, ui32>& decoderBase);
        };

        bool Parse(const NStorage::TTableRecord& row);
        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

        NStorage::TTableRecord SerializeToTableRecord() const;
    };


    class TDBDistributingBlockEventsStorageConfig : public IDistributingBlockEventsStorageConfig, public TDBEntitiesManagerConfig {
    public:
        using TDBEntitiesManagerConfig::TDBEntitiesManagerConfig;

    private:
        virtual void Init(const TYandexConfig::Section& section) override;
        virtual THolder<IDistributingBlockEventsStorage> BuildStorage(const NDrive::IServer& server) const override;

    private:
        static TFactory::TRegistrator<TDBDistributingBlockEventsStorageConfig> Registrator;
    };


    class TDistributingBlockEventStatsHistoryManager : public TDatabaseHistoryManager<TDistributingBlockEventStats> {
    private:
        using TBase = TDatabaseHistoryManager<TDistributingBlockEventStats>;

    public:
        TDistributingBlockEventStatsHistoryManager(const IHistoryContext& context)
            : TBase(context, "distributing_block_event_stats_history")
        {
        }
    };

    class TDistributingBlockEventStatsDB : public TDBEntities<TDistributingBlockEventStats> {
    private:
        using TEntity = TDistributingBlockEventStats;
        using TBase = TDBEntities<TEntity>;

    public:
        TDistributingBlockEventStatsDB(const IHistoryContext& context)
            : TBase(context.GetDatabase())
            , HistoryWriter(context)
        {
        }

        bool AddShow(const TDistributingBlockEvent& event, NDrive::TEntitySession& session) const;
        TVector<TDistributingBlockEventStats> GetShows(const TString& userId) const;

    private:
        TString GetTableName() const override {
            return "distributing_block_event_stats";
        }

        TString GetColumnName() const override {
            return "user_id";
        }

        TString GetMainId(const TEntity& entity) const override {
            return entity.GetUserId();
        }

        virtual void UpdateUniqueCondition(NStorage::TTableRecord& unique, const TEntity& info) const override {
            unique.Set("type", info.GetType());
        }

    private:
        TDistributingBlockEventStatsHistoryManager HistoryWriter;
    };

    class TDBDistributingBlockEventsStorage : public IDistributingBlockEventsStorage {
    public:
        TDBDistributingBlockEventsStorage(THolder<IHistoryContext> historyContext);

    private:
        virtual NThreading::TFuture<TDistributingBlockEvent::TPtr> Store(TDistributingBlockEvent::TPtr event, NDrive::TEntitySession* session) const override;
        virtual NThreading::TFuture<TVector<TDistributingBlockEvent::TPtr>> Retrieve(const TString& userId) const override;

    private:
        THolder<IHistoryContext> HistoryContext;
        TDistributingBlockEventStatsDB StatsDB;
    };
}
