#pragma once

#include <drive/library/cpp/scheme/scheme.h>
#include <drive/library/cpp/tex_builder/tex_builder.h>

#include <library/cpp/charset/wide.h>
#include <library/cpp/object_factory/object_factory.h>
#include <library/cpp/string_utils/base64/base64.h>

#include <rtline/util/types/messages_collector.h>

#include <util/memory/blob.h>
#include <util/string/builder.h>

enum class EBlockType {
    Text,
    Image,
    Binary,
    Pdf,
};

struct TAddBlockContext {
    TString ImagePosition = "p";

    bool Parse(const NJson::TJsonValue& json) {
        if (json.Has("image_position")) {
            ImagePosition = json["image_position"].GetString();
        }
        return true;
    }

    NJson::TJsonValue SerializeToJson() const {
        NJson::TJsonValue json;
        json.InsertValue("image_position", ImagePosition);
        return json;
    }

    static NDrive::TScheme GetScheme() {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("image_position", "Положение картинки").SetDefault("p");
        return scheme;
    }
};

class IDocumentAssembler {
public:
    virtual ~IDocumentAssembler() {}
    virtual TBlob BuildFinalDocument(TMessagesCollector& errors) const = 0;
    virtual bool AddBlock(const TString& content, TMessagesCollector& messages, EBlockType type = EBlockType::Text, const TAddBlockContext& context = Default<TAddBlockContext>()) = 0;

private:
   R_FIELD(TString, AuthorId);
};

class TRawDocBuilder: public IDocumentAssembler {
public:
    virtual TBlob BuildFinalDocument(TMessagesCollector& /*errors*/) const override {
        return TBlob::FromString(Content);
    }

    virtual bool AddBlock(const TString& content, TMessagesCollector& /*messages*/, EBlockType type, const TAddBlockContext& /*context*/) override {
        switch (type) {
        case EBlockType::Image:
        case EBlockType::Pdf:
        case EBlockType::Binary:
            Content << Base64Encode(content);
            break;
        case EBlockType::Text:
            Content << content;
            break;
        default:
            return false;
        }
        return true;
    }

private:
    TStringBuilder Content;
};

class TCsvDocBuilder : public IDocumentAssembler {
public:
    virtual TBlob BuildFinalDocument(TMessagesCollector& /*errors*/) const override {
        return TBlob::FromString(WideToChar(UTF8ToWide(Content), CODES_WIN));
    }

    virtual bool AddBlock(const TString& content, TMessagesCollector& /*messages*/, EBlockType /*type*/, const TAddBlockContext& /*context*/) override {
        Content << content;
        return true;
    }

private:
    TStringBuilder Content;
};

class TTexServerDocBuilder : public IDocumentAssembler, public NTexBuilder::TTexBuilder {
public:
    TTexServerDocBuilder(const NTexBuilder::TTexBuilderConfig& config)
        : IDocumentAssembler()
        , NTexBuilder::TTexBuilder(config)
    {}

    virtual TBlob BuildFinalDocument(TMessagesCollector& errors) const override {
        return NTexBuilder::TTexBuilder::BuildFinalDocument(errors);
    }

    virtual bool AddBlock(const TString& content, TMessagesCollector& messages, EBlockType type, const TAddBlockContext& context) override {
        switch (type) {
        case EBlockType::Image: {
            NTexBuilder::EPosition position = NTexBuilder::EPosition::Page;
            if (TryFromString(context.ImagePosition, position)) {
                return AddImage(content, messages, position);
            }
            return AddImage(content, messages);
        }
        case EBlockType::Text:
            return AddText(content, messages);
        case EBlockType::Pdf:
            return AddPDF(content, messages);
        default:
            return false;
        }
        return false;
    }
};
