#include "document.h"

#include <drive/backend/abstract/frontend.h>

#include <rtline/library/json/cast.h>

#include <library/cpp/string_utils/relaxed_escaper/relaxed_escaper.h>

TDocumentDescription::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Name = GetFieldDecodeIndex("document_name", decoderBase);
    ContentType = GetFieldDecodeIndex("content_type", decoderBase);
    Comment = GetFieldDecodeIndex("comment", decoderBase);
    IsActive = GetFieldDecodeIndex("active", decoderBase);
    DocumentMeta = GetFieldDecodeIndex("document_meta", decoderBase);
}

bool TDocumentDescription::DeserializeWithDecoder(const TDocumentDescription::TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Name);
    READ_DECODER_VALUE(decoder, values, Comment);
    READ_DECODER_VALUE(decoder, values, ContentType);
    READ_DECODER_VALUE(decoder, values, IsActive);

    if (!Name) {
        return false;
    }

    NJson::TJsonValue meta;
    READ_DECODER_VALUE_JSON(decoder, values, meta, DocumentMeta);
    return ParseMeta(meta);
}

NStorage::TTableRecord TDocumentDescription::SerializeToTableRecord() const {
    NStorage::TRecordBuilder builder;
    builder("content_type", ContentType)("document_name", Name)("comment", Comment)("active", IsActive);
    builder("document_meta", SaveMeta());
    return builder;
}

bool TDocumentDescription::ParseMeta(const NJson::TJsonValue& meta) {
    JREAD_CONTAINER_OPT(meta, "templates", Templates);
    JREAD_CONTAINER_OPT(meta, "additional_parameters", AdditionalParameters);

    if (meta.Has("override_settings")) {
        const NJson::TJsonValue& parameters = meta["override_settings"];
        if (!parameters.IsArray()) {
            return false;
        }
        for (auto&& t : parameters.GetArray()) {
            const auto& key = t["template"].GetString();
            TSet<TString> fields;
            for (const auto& field : t["fields"].GetArray()) {
                fields.emplace(field.GetStringRobust());
            }
            if (key && fields) {
                OverrideParameters.emplace(key, std::move(fields));
            }
        }
    }
    JREAD_BOOL_OPT(meta, "use_user_watermark", UseUserWatermark);
    JREAD_STRING_OPT(meta, "document_format", DocumentFormat);
    JREAD_CONTAINER_OPT(meta, "groupping_tags", GrouppingTags);
    JREAD_STRING_OPT(meta, "queue", Queue);
    JREAD_INSTANT_OPT(meta, "deadline", Deadline);
    return DoParseMeta(meta);
}

NJson::TJsonValue TDocumentDescription::SaveMeta() const {
    NJson::TJsonValue meta = DoSaveMeta();
    TJsonProcessor::WriteContainerArray(meta, "templates", Templates);
    TJsonProcessor::WriteContainerArray(meta, "additional_parameters", AdditionalParameters);

    NJson::TJsonValue& overrideParameters = meta.InsertValue("override_settings", NJson::TJsonValue(NJson::JSON_ARRAY));
    for (auto&& t : OverrideParameters) {
        NJson::TJsonValue json;
        json.InsertValue("template", t.first);
        NJson::TJsonValue::TArray& fieldsArray = json["fields"].SetType(NJson::JSON_ARRAY).GetArraySafe();
        fieldsArray = NJson::TJsonValue::TArray(t.second.begin(), t.second.end());
        overrideParameters.AppendValue(std::move(json));
    }

    JWRITE(meta, "use_user_watermark", UseUserWatermark);
    JWRITE(meta, "document_format", DocumentFormat);
    JWRITE(meta, "queue", Queue);
    TJsonProcessor::WriteInstant(meta, "deadline", Deadline, TInstant::Max());
    TJsonProcessor::WriteContainerString(meta, "groupping_tags", GrouppingTags);
    return meta;
}

NDrive::TScheme TDocumentDescription::GetScheme(const IServerBase& server) const {
    TSet<TString> keys;
    ITemplateData::TFactory::GetRegisteredKeys(keys);

    NDrive::TScheme meta = DoGetScheme(server);
    meta.Add<TFSVariants>("templates", "Необходимые наборы данных").SetMultiSelect(true).SetVariants(keys);
    meta.Add<TFSArray>("additional_parameters", "Дополнительные параметры").SetElement<TFSString>();
    meta.Add<TFSBoolean>("use_user_watermark", "Добавить водяной знак").SetDefault(false);
    meta.Add<TFSString>("document_format", "Формат построения документа").SetDefault("pdf");
    meta.Add<TFSString>("queue", "Очередь построения").SetDefault("default");

    NDrive::TScheme overrideSetting;
    overrideSetting.Add<TFSVariants>("template", "Шаблон").SetVariants(keys).SetRequired(true);
    overrideSetting.Add<TFSArray>("fields", "Поля").SetElement<TFSString>().SetRequired(true);
    meta.Add<TFSArray>("override_settings", "Переопределяемые параметры").SetElement(overrideSetting);

    meta.Add<TFSString>("groupping_tags", "Атрибуты группировки");
    meta.Add<TFSNumeric>("deadline", "Доступен до").SetVisual(TFSNumeric::EVisualType::DateTime);

    NDrive::TScheme result;
    result.Add<TFSStructure>("document_meta", "Структура документа").SetStructure(meta);
    result.Add<TFSString>("comment", "Название");
    result.Add<TFSString>("document_name", "Уникальный идентификатор документа");
    result.Add<TFSBoolean>("active", "Доступен для построения").SetDefault(false);
    return result;
}

TString TDocumentDescription::GetDocumentOriginFormat() const {
    return DocumentFormat == "epdf" ? "pdf" : DocumentFormat;
}

bool TDocumentDescription::CreateContent(const NJson::TJsonValue& postData, const TMap<TString, ITemplateData::TPtr>& parameters, const NDrive::IServer& server, IDocumentAssembler& builder, TMessagesCollector& errors) const {
    if (Now() > Deadline) {
        errors.AddMessage("ui_errors", "Срок годности документа \"" + Comment + "\" подошел к концу");
        return false;
    }

    if (UseUserWatermark) {
        TString watermarkString = "\\newwallpaper*[page=\\thepage,textalign=center, textangle=45, tilexoffset=-5cm, tileyoffset=3cm, boxalign=center, tileysize=20pt, wpxoffset=-2cm, wpyoffset=-2cm]{\\usebox\\userwatermark}";
        if (!builder.AddBlock(watermarkString, errors)) {
            errors.AddMessage(__LOCATION__, "cannot add watermark " + errors.GetStringReport());
            return false;
        }
    }
    return DoCreateContent(postData, parameters, server, builder, errors);
}

NDrive::TScheme TDocumentDescription::GetFullTemplatesSchemeResult(const IServerBase& server) const {
    NDrive::TScheme scheme = GetFullTemplatesScheme(server);
    if (!scheme.HasField("notifiers")) {
        scheme.Add<TFSArray>("notifiers", "Нотификация").SetElement(TQueuedDocument::TNotifier::GetScheme(server));
    }
    if (!scheme.HasField("document_format")) {
        scheme.Add<TFSString>("document_format", "Формат документа").SetDefault(GetDocumentOriginFormat());
    }
    if (!scheme.HasField("queue")) {
        scheme.Add<TFSString>("queue", "Очередь обработки").SetDefault(Queue);
    }
    return scheme;
}

NDrive::TScheme TDocumentDescription::GetFullTemplatesScheme(const IServerBase& server) const {
    NDrive::TScheme scheme;
    AddFullTemplatesScheme(scheme, server);
    return scheme;
}

void TDocumentDescription::AddFullTemplatesScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    AddTemplatesToScheme(scheme, server);
    AddPostParametersToScheme(scheme, server);
    AddOverrideParametersToScheme(scheme, server);
}

void TDocumentDescription::AddTemplatesToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    for (const auto& t : Templates) {
        THolder<ITemplateData> templatePtr(ITemplateData::TFactory::Construct(t));
        if (templatePtr) {
            templatePtr->AddInputsToScheme(server, scheme);
        }
    }
}

void TDocumentDescription::AddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& /*server*/) const {
    for (const auto& t : AdditionalParameters) {
        if (!scheme.HasField(t)) {
            scheme.Add<TFSString>(t).SetRequired(true);
        }
    }
}

void TDocumentDescription::AddOverrideParametersToScheme(NDrive::TScheme& scheme, const IServerBase& /*server*/) const {
    if (OverrideParameters.empty()) {
        return;
    }
    NDrive::TScheme overrideScheme;
    for (const auto& t : OverrideParameters) {
        if (!overrideScheme.HasField(t.first)) {
            NDrive::TScheme parametersScheme;
            for (const auto& param : t.second) {
                parametersScheme.Add<TFSString>(param);
            }
            overrideScheme.Add<TFSStructure>(t.first).SetStructure(parametersScheme);
        }
    }
    if (!scheme.HasField("override_parameters")) {
        scheme.Add<TFSStructure>("override_parameters").SetStructure(overrideScheme);
    }
}

TDocumentDescription::TPtr TDocumentDescription::BuildFromJson(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector* errors /*= nullptr*/) {
    TString type = json["content_type"].GetString();
    TPtr description = TFactory::Construct(type);
    if (!description) {
        if (errors) {
            errors->AddMessage("parsing", "Unknown document type: '" + type + "'");
        }
        return nullptr;
    }

    if (!description->FromJson(json, server, errors)) {
        return nullptr;
    }
    return description;
}

bool TDocumentDescription::FromJson(const NJson::TJsonValue& json, const NDrive::IServer& /*server*/, TMessagesCollector* /*errors*/) {
    return TBaseDecoder::DeserializeFromJson(*this, json);
}

NJson::TJsonValue TDocumentDescription::GetReport() const {
    NJson::TJsonValue result;
    result["content_type"] = ContentType;
    result["document_name"] = Name;
    result["comment"] = Comment;
    result["document_meta"] = SaveMeta();
    result["active"] = IsActive;
    return result;
};

bool TDocumentDescription::AddTemplatesFromPostData(TMap<TString, ITemplateData::TPtr>& result, const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) const {
    TSet<TString> additionalTemplates;
    for (const auto& currentTemplate : Templates) {
        if (!result.contains(currentTemplate) || currentTemplate == "post") {
            additionalTemplates.insert(currentTemplate);
        }
    }
    return ITemplateData::BuildFromPostData(result, json, additionalTemplates, server, errors);
}

bool TDocumentDescriptionPtr::DeserializeWithDecoder(const TDocumentDescriptionPtr::TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
    TString type;
    READ_DECODER_VALUE_TEMP(decoder, values, type, ContentType);

    Document = TDocumentDescription::TFactory::Construct(type);
    if (!Document) {
        return false;
    }

    return Document->DeserializeWithDecoder(decoder, values, hContext);
}

TString EscapeJson(const TString& value) {
    return NEscJ::EscapeJ<false>(value);
}

ITemplateData::TDataEscape TDocumentDescription::GetDataEscape() const {
    if (DocumentFormat == "json") {
        return EscapeJson;
    } else if (DocumentFormat == "epdf") {
        return NTexBuilder::TTextStyle::Quote;
    }
    return {};
}

NStorage::TTableRecord TDocumentDescriptionPtr::SerializeToTableRecord() const {
    return Document->SerializeToTableRecord();
}

NStorage::TTableRecord TDocumentDescriptionPtr::SerializeUniqueToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("document_name", Document->GetName());
    return result;
}

bool TDocumentsDescriptionDB::Upsert(const TDocumentDescriptionPtr& description, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(TDocumentDescriptionPtr::GetTableName());
    const NStorage::TTableRecord record = description.SerializeToTableRecord();
    const NStorage::TTableRecord unique = description.SerializeUniqueToTableRecord();
    bool isUpdate;
    if (!table->Upsert(record, transaction, unique, &isUpdate)) {
        session.SetErrorInfo("set_document_description", "Upsert failed", EDriveSessionResult::TransactionProblem);
        return false;
    }
    if (!HistoryManager->AddHistory(description, userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
        return false;
    }
    return true;
}

bool TDocumentsDescriptionDB::Remove(const TVector<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const {
    if (ids.empty())
        return true;
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(TDocumentDescriptionPtr::GetTableName());
    const TString condition = "document_name IN (" + session->Quote(ids) + ")";
    NStorage::TObjectRecordsSet<TDocumentDescriptionPtr> records;
    if (!table->RemoveRow(condition, transaction, &records)) {
        session.SetErrorInfo("remove_document_description", "RemoveRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }

    if (!HistoryManager->AddHistory(records.GetObjects(), userId, EObjectHistoryAction::Remove, session)) {
        return false;
    }
    return true;
}

NDrive::TScheme TQueuedDocument::TNotifier::GetScheme(const IServerBase& server) {
    NDrive::TScheme scheme;
    scheme.Add<TFSVariants>("notifier", "Нотифаер").SetVariants(server.GetNotifierNames()).SetRequired(true);
    scheme.Add<TFSString>("name", "Имя файла").SetRequired(true);
    scheme.Add<TFSString>("title", "Заголовок");
    scheme.Add<TFSText>("description", "Описание");
    return scheme;
}

bool TQueuedDocument::TNotifier::DeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_STRING(json, "notifier", Notifier);
    JREAD_STRING(json, "name", Name);
    JREAD_STRING_OPT(json, "title", Title);
    if (!Title) {
        Title = Name;
    }
    JREAD_STRING_OPT(json, "description", Description);
    return true;
}

NJson::TJsonValue TQueuedDocument::TNotifier::SerializeToJson() const {
    NJson::TJsonValue json;
    JWRITE(json, "notifier", Notifier);
    JWRITE(json, "name", Name);
    JWRITE(json, "title", Title);
    JWRITE(json, "description", Description);
    return json;
}

template <>
bool NJson::TryFromJson<TQueuedDocument::TNotifier>(const TJsonValue& value, TQueuedDocument::TNotifier& result) {
    return result.DeserializeFromJson(value);
}

template <>
NJson::TJsonValue NJson::ToJson<TQueuedDocument::TNotifier>(const TQueuedDocument::TNotifier& object) {
    return object.SerializeToJson();
}

TQueuedDocument::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Id = GetFieldDecodeIndex("id", decoderBase);
    Error = GetFieldDecodeIndex("error", decoderBase);
    Author = GetFieldDecodeIndex("author", decoderBase);
    CreateTime = GetFieldDecodeIndex("create_time", decoderBase);
    LastUpdate = GetFieldDecodeIndex("last_update", decoderBase);
    Status = GetFieldDecodeIndex("status", decoderBase);
    InputParameters = GetFieldDecodeIndex("inputs", decoderBase);
    ResultOutput = GetFieldDecodeIndex("result_output", decoderBase);
}

bool TQueuedDocument::DeserializeWithDecoder(const TQueuedDocument::TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Id);
    READ_DECODER_VALUE(decoder, values, Error);
    READ_DECODER_VALUE(decoder, values, Author);
    READ_DECODER_VALUE_INSTANT(decoder, values, CreateTime);
    READ_DECODER_VALUE_INSTANT(decoder, values, LastUpdate);

    READ_DECODER_VALUE(decoder, values, Status);
    READ_DECODER_VALUE_JSON(decoder, values, InputParameters, InputParameters);
    READ_DECODER_VALUE_JSON(decoder, values, ResultOutput, ResultOutput);

    if (InputParameters.Has("notifiers")) {
        for (const auto& notifier : InputParameters["notifiers"].GetArray()) {
            TNotifier notifierConfig;
            if (!notifierConfig.DeserializeFromJson(notifier)) {
                return false;
            }
            Notifiers.emplace_back(std::move(notifierConfig));
        }
    }
    return SetInputParameters(InputParameters);
}

NStorage::TTableRecord TQueuedDocument::SerializeToTableRecord() const {
    NStorage::TRecordBuilder builder;
    builder("id", Id)("status", ToString(Status))("result_output", ResultOutput.GetStringRobust())("inputs", InputParameters.GetStringRobust())("error", Error)("author", Author);
    return builder("create_time", CreateTime.Seconds())("last_update", LastUpdate.Seconds());
}

bool TQueuedDocumentDB::Upsert(const TQueuedDocument& document, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(TQueuedDocument::GetTableName());
    const NStorage::TTableRecord record = document.SerializeToTableRecord();
    NStorage::TTableRecord unique;
    unique.Set("id", document.GetId());
    bool isUpdate;
    if (!table->Upsert(record, transaction, unique, &isUpdate)) {
        session.SetErrorInfo("upsert_queued_document", "Upsert failed", EDriveSessionResult::TransactionProblem);
        return false;
    }
    if (!HistoryManager->AddHistory(document, userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
        return false;
    }
    return true;
}

bool TQueuedDocumentDB::Remove(const TVector<TString>& ids, const TString& userId, NDrive::TEntitySession& session) const {
    if (ids.empty())
        return true;
    NStorage::ITransaction::TPtr transaction = session.GetTransaction();
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(TQueuedDocument::GetTableName());
    const TString condition = "id IN (" + session->Quote(ids) + ")";
    NStorage::TObjectRecordsSet<TQueuedDocument> records;
    if (!table->RemoveRow(condition, transaction, &records)) {
        session.SetErrorInfo("remove_queued_document", "RemoveRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }

    if (!HistoryManager->AddHistory(records.GetObjects(), userId, EObjectHistoryAction::Remove, session)) {
        return false;
    }
    return true;
}
