#pragma once

#include "builder.h"
#include "template.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/db_entities.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/json/writer/json_value.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

class TDocumentsManager;

class TDocumentDescription {
    using TOverrideParameters = TMap<TString, TSet<TString>>;
    R_READONLY(TString, Name);
    R_READONLY(TString, Comment);
    R_READONLY(TString, ContentType);
    R_READONLY(TSet<TString>, Templates, {});
    R_READONLY(bool, IsActive, false);
    R_READONLY(TSet<TString>, AdditionalParameters, {});
    R_READONLY(TOverrideParameters, OverrideParameters, {});
    R_READONLY(bool, UseUserWatermark, false);
    R_READONLY(TString, DocumentFormat, "pdf");
    R_READONLY(TSet<TString>, GrouppingTags);
    R_READONLY(TString, Queue, "default");
    R_READONLY(TInstant, Deadline, TInstant::Max());

public:
    class TDecoder : public TBaseDecoder {
        R_FIELD(i32, Name, -1);
        R_FIELD(i32, Comment, -1);
        R_FIELD(i32, ContentType, -1);
        R_FIELD(i32, IsActive, -1);
        R_FIELD(i32, DocumentMeta, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

protected:
    ITemplateData::TDataEscape GetDataEscape() const;

public:
    using TPtr = TAtomicSharedPtr<TDocumentDescription>;

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    virtual bool ParseMeta(const NJson::TJsonValue& meta);
    virtual bool DoParseMeta(const NJson::TJsonValue& meta) = 0;

    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue SaveMeta() const;
    virtual NJson::TJsonValue DoSaveMeta() const = 0;

    virtual NDrive::TScheme GetScheme(const IServerBase& server) const;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const = 0;
    virtual TString GetDocumentOriginFormat() const;

    NDrive::TScheme GetFullTemplatesScheme(const IServerBase& server) const;
    NDrive::TScheme GetFullTemplatesSchemeResult(const IServerBase& server) const;
    void AddFullTemplatesScheme(NDrive::TScheme& scheme, const IServerBase& server) const;
    virtual void AddTemplatesToScheme(NDrive::TScheme& scheme, const IServerBase& server) const;
    virtual void AddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const;
    virtual void AddOverrideParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const;

    static TDocumentDescription::TPtr BuildFromJson(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector* errors = nullptr);
    virtual bool FromJson(const NJson::TJsonValue& json, const NDrive::IServer& /*server*/, TMessagesCollector* errors = nullptr);

    NJson::TJsonValue GetReport() const;

    bool CreateContentFromJson(const NJson::TJsonValue& postData, const NDrive::IServer& server, IDocumentAssembler& builder, TMessagesCollector& errors) const {
        TMap<TString, ITemplateData::TPtr> parameters;
        return CreateContentFromJson(postData, parameters, server, builder, errors);
    }

    bool CreateContentFromJson(const NJson::TJsonValue& postData, TMap<TString, ITemplateData::TPtr> parameters, const NDrive::IServer& server, IDocumentAssembler& builder, TMessagesCollector& errors) const {
        return AddTemplatesFromPostData(parameters, postData, server, errors) && CreateContent(postData, parameters, server, builder, errors);
    }

    bool CreateContent(const NJson::TJsonValue& postData, const TMap<TString, ITemplateData::TPtr>& parameters, const NDrive::IServer& server, IDocumentAssembler& builder, TMessagesCollector& errors) const;

    virtual bool DoCreateContent(const NJson::TJsonValue& postData, const TMap<TString, ITemplateData::TPtr>& parameters, const NDrive::IServer& server, IDocumentAssembler& builder, TMessagesCollector& errors) const = 0;

    virtual bool AddTemplatesFromPostData(TMap<TString, ITemplateData::TPtr>& result, const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) const;

    virtual ~TDocumentDescription() {}

public:
    using TFactory = NObjectFactory::TObjectFactory<TDocumentDescription, TString>;
};

class TDocumentDescriptionPtr {
public:
    using TDecoder = TDocumentDescription::TDecoder;
    using TId = TString;

public:
    TDocumentDescriptionPtr() = default;
    TDocumentDescriptionPtr(TDocumentDescription::TPtr document)
        : Document(document)
    {}

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext);

    NStorage::TTableRecord SerializeToTableRecord() const;
    NStorage::TTableRecord SerializeUniqueToTableRecord() const;

    const TDocumentDescription::TPtr operator->() const {
        return Document;
    }

    bool operator!() const {
        return !Document;
    }

    const TString& GetInternalId() const {
        return Document->GetName();
    }

    static TString GetTableName() {
        return "service_documents";
    }

private:
    TDocumentDescription::TPtr Document;
};

class TDocumentsDescriptionHistoryManager : public TIndexedAbstractHistoryManager<TDocumentDescriptionPtr> {
private:
    using TBase = TIndexedAbstractHistoryManager<TDocumentDescriptionPtr>;

public:
    class TEntityIndex : public IEventsIndex<TObjectEvent<TDocumentDescriptionPtr>> {
    private:
        using TBase = IEventsIndex<TObjectEvent<TDocumentDescriptionPtr>>;
    protected:
        virtual TMaybe<TStringBuf> ExtractKey(TAtomicSharedPtr<TObjectEvent<TDocumentDescriptionPtr>> ev) const override {
            return (*ev)->GetName();
        }
    public:
        using TBase::TBase;
    };

public:
    TDocumentsDescriptionHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, "service_documents_history", config)
        , EntityIndex(new TEntityIndex(this, "document_name"))
    {}

    const TEntityIndex& GetIndexByName() const {
        return *EntityIndex;
    }

private:
    THolder<TEntityIndex> EntityIndex;
};

class TDocumentsDescriptionDB: public TDBEntitiesCache<TDocumentDescriptionPtr, TDocumentsDescriptionHistoryManager> {
private:
    using TBase = TDBEntitiesCache<TDocumentDescriptionPtr, TDocumentsDescriptionHistoryManager>;

public:
    using TBase::TBase;

    bool Upsert(const TDocumentDescriptionPtr& description, const TString& userId, NDrive::TEntitySession& session) const;
    bool Remove(const TVector<TString>& names, const TString& userId, NDrive::TEntitySession& session) const;
};

enum class EAssemblerStage {
    Done /* "DONE" */,
    Error /* "ERROR" */,
    NotReady /* "NOT_READY" */,
    Unknown /* "UNKNOWN" */,
    FailedDeletion /* "FAILED_DELETION" */,
    Canceled /* "CANCELED" */,
};

enum EDocumentStorage {
    MDS /* "mds" */,
    Raw /* "raw" */,
    Base64 /* "base_64" */,
    Json /* "json" */,
};

class TQueuedDocument {
public:
    class TNotifier {
        R_FIELD(TString, Notifier);
        R_FIELD(TString, Name);
        R_FIELD(TString, Title);
        R_FIELD(TString, Description);
    public:
        static NDrive::TScheme GetScheme(const IServerBase& server);
        bool DeserializeFromJson(const NJson::TJsonValue& json);
        NJson::TJsonValue SerializeToJson() const;
    };

    R_FIELD(TString, Id);
    R_FIELD(EAssemblerStage, Status, EAssemblerStage::Unknown);
    R_FIELD(NJson::TJsonValue, ResultOutput);
    R_FIELD(TString, Error);
    R_FIELD(TString, Author);
    R_FIELD(TInstant, CreateTime, TInstant::Now());
    R_FIELD(TInstant, LastUpdate, TInstant::Zero());

    R_READONLY(TString, DocumentName);
    R_READONLY(TString, DocumentFormat);
    R_READONLY(TString, DocumentQueue);
    R_READONLY(TSet<EDocumentStorage>, Storages);
    R_READONLY(TVector<TNotifier>, Notifiers);

public:
    using TId = TString;

public:
    class TDecoder : public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, Status, -1);
        R_FIELD(i32, ResultOutput, -1);
        R_FIELD(i32, InputParameters, -1);
        R_FIELD(i32, Error, -1);
        R_FIELD(i32, Author, -1);
        R_FIELD(i32, CreateTime, -1);
        R_FIELD(i32, LastUpdate, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

public:
    TQueuedDocument() = default;

    NJson::TJsonValue GetReport() const {
        NJson::TJsonValue result;
        result.InsertValue("id", Id);
        result.InsertValue("status", ToString(Status));
        if (!Error.empty()) {
            result.InsertValue("error", Error);
        }
        result.InsertValue("result_output", ResultOutput);
        result.InsertValue("inputs", InputParameters);
        result.InsertValue("author", Author);
        result.InsertValue("create_time", CreateTime.Seconds());
        result.InsertValue("last_update", LastUpdate.Seconds());
        return result;
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    NStorage::TTableRecord SerializeToTableRecord() const;

    const TString& GetInternalId() const {
        return Id;
    }

    static TString GetTableName() {
        return "document_queue";
    }

    TQueuedDocument& SetDocumentQueue(const TString& queue) {
        InputParameters["queue"] = queue;
        DocumentQueue = queue;
        return *this;
    }

    bool SetInputParameters(const NJson::TJsonValue& json) {
        InputParameters = json;
        JREAD_STRING(InputParameters, "document_name", DocumentName);
        JREAD_STRING_OPT(InputParameters, "document_format", DocumentFormat);
        JREAD_STRING_OPT(InputParameters, "queue", DocumentQueue);
        JREAD_CONTAINER(InputParameters, "storage", Storages);
        return true;
    }

    const NJson::TJsonValue& GetInputParameters() const {
        return InputParameters;
    }

private:
    NJson::TJsonValue InputParameters;
};

class TDocumentsQueueHistoryManager : public TIndexedAbstractHistoryManager<TQueuedDocument> {
private:
    using TBase = TIndexedAbstractHistoryManager<TQueuedDocument>;

public:
    class TDocumentIndex : public IEventsIndex<TObjectEvent<TQueuedDocument>> {
    private:
        using TBase = IEventsIndex<TObjectEvent<TQueuedDocument>>;
    protected:
        virtual TMaybe<TStringBuf> ExtractKey(TAtomicSharedPtr<TObjectEvent<TQueuedDocument>> ev) const override {
            return ev->GetDocumentName();
        }
    public:
        using TBase::TBase;
    };

    class TUserIndex : public IEventsIndex<TObjectEvent<TQueuedDocument>> {
    private:
        using TBase = IEventsIndex<TObjectEvent<TQueuedDocument>>;
    protected:
        virtual TMaybe<TStringBuf> ExtractKey(TAtomicSharedPtr<TObjectEvent<TQueuedDocument>> ev) const override {
            return ev->GetHistoryUserId();
        }
    public:
        using TBase::TBase;
    };

public:
    TDocumentsQueueHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, "document_queue_history", config)
        , DocumentIndex(new TDocumentIndex(this, "document_name"))
        , UserIndex(new TUserIndex(this, "history_user"))
    {}

    const TDocumentIndex& GetIndexByDocument() const {
        return *DocumentIndex;
    }

    const TUserIndex& GetIndexByUser() const {
        return *UserIndex;
    }

private:
    THolder<TDocumentIndex> DocumentIndex;
    THolder<TUserIndex> UserIndex;
};

class TQueuedDocumentDB : public TDBEntitiesCache<TQueuedDocument, TDocumentsQueueHistoryManager> {
private:
    using TBase = TDBEntitiesCache<TQueuedDocument, TDocumentsQueueHistoryManager>;
public:
    using TBase::TBase;

    bool Upsert(const TQueuedDocument& description, const TString& userId, NDrive::TEntitySession& session) const;
    bool Remove(const TVector<TString>& names, const TString& userId, NDrive::TEntitySession& session) const;
};
