#include "car_template.h"

#include <drive/backend/database/drive_api.h>

#include <drive/telematics/api/sensor/interface.h>
#include <drive/telematics/common/firmware.h>

using namespace NTemplateData;

TString TCarTemplate::Name("car");
TSet<TCarTemplate::EOutput> TCarTemplate::DateParameters = {
    TCarTemplate::EOutput::RegistrationDate,
    TCarTemplate::EOutput::ContractDate,
    TCarTemplate::EOutput::ContractIdDate,
    TCarTemplate::EOutput::PurchaseContractDate,
    TCarTemplate::EOutput::TransferDate,
    TCarTemplate::EOutput::OsagoDate,
    TCarTemplate::EOutput::OsagoDateTo,
    TCarTemplate::EOutput::DeptransDocumentsDate,
    TCarTemplate::EOutput::ParkingPermitStartDate,
    TCarTemplate::EOutput::ReturnDate,
    TCarTemplate::EOutput::FrameworkAgreementDate,
};
ITemplateData::TFactory::TRegistrator<TCarTemplate> TCarTemplate::Registrator(TCarTemplate::Name);

void TCarTemplate::AddInputsToScheme(const IServerBase& /* server */, NDrive::TScheme& scheme) const {
    if (!scheme.HasField(ToString(EInput::CarId))) {
        scheme.Add<TFSString>(ToString(EInput::CarId), "Идентификатор автомобиля").SetVisual(TFSString::EVisualType::GUID).SetRequired(true);
    }
    if (!scheme.HasField(ToString(EInput::DocDate))) {
        scheme.Add<TFSNumeric>(ToString(EInput::DocDate), "Дата, на которую нужны данные по машине").SetVisual(TFSNumeric::EVisualType::DateTime);
    }
}

void TCarTemplate::AddParameter(TCarTemplate::EOutput parameter, const NJson::TJsonValue& json) {
    THRDate date;
    if (DateParameters.contains(parameter) && IHRInstant::TryFromString(json.GetStringRobust(), date)) {
        Storage.AddParameter(parameter, date.ToString());
    } else {
        Storage.AddParameter(parameter, json.GetStringRobust());
    }
}

bool TCarTemplate::Fetch(const TString& carId, TInstant carDocDate, const NDrive::IServer& server, TMessagesCollector& errors) {
    auto tx = server.GetDriveAPI()->BuildTx<NSQL::ReadOnly>();
    auto fetchResult = server.GetDriveAPI()->GetCarsData()->FetchInfo(carId, tx);
    if (!fetchResult) {
        errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось получить информацию по авто " + carId);
        errors.AddMessage("tx_error", tx.GetStringReport());
        return false;
    }
    AddParameter(EOutput::Id, carId);
    auto dataPtr = fetchResult.GetResultPtr(carId);
    if (!dataPtr) {
        errors.AddMessage("ui_errors", "Неизвестный авто " + carId);
        return false;
    }

    const auto& attachmentManager = server.GetDriveAPI()->GetCarAttachmentAssignments();
    auto attachments = attachmentManager.GetAttachmentsByInstant(carId, carDocDate, tx, EDocumentAttachmentType::CarRegistryDocument);
    if (!attachments) {
        errors.AddMessage("ui_errors", "Данные о машине не найдены " + carId);
        errors.AddMessage("tx_error", tx.GetStringReport());
        return false;
    }

    auto baseDocuments = attachments->size() == 1 ? dynamic_cast<const TCarRegistryDocument*>(attachments->front().Get()) : nullptr;
    if (baseDocuments) {
        TCarAttachmentReportContext hrContext(server.GetDriveAPI());
        hrContext.SetRegistryDocumentTraits(NAttachmentReport::ReportAllStructure);
        NJson::TJsonValue hrReport = baseDocuments->BuildReport(hrContext);
        for (const auto& parameter : Storage.GetEnumOutputs()) {
            NJson::TJsonValue* value = hrReport.GetValueByPath(ToString(parameter));
            if (value) {
                AddParameter(parameter, *value);
            }
        }
        if (baseDocuments->GetPurchaseContractNumber() && baseDocuments->GetPurchaseContractDate()) {
            TString purchaseName = baseDocuments->GetPurchaseContractNumber() + "_" + baseDocuments->GetPurchaseContractDate().FormatGmTime("%d.%m.%Y");
            SubstGlobal(purchaseName, "/", ".");
            AddParameter(EOutput::PurchaseContractFileName, purchaseName + ".pdf");
            AddParameter(EOutput::PurchaseContractDirName, purchaseName + "/");
        }
        if (baseDocuments->GetContractId() && baseDocuments->GetContractIdDate()) {
            AddParameter(EOutput::FrameworkAgreementDate, baseDocuments->GetContractIdDate().ToString());
            AddParameter(EOutput::FrameworkAgreementNumber, baseDocuments->GetContractId());
            TString fileName = baseDocuments->GetContractId() + "_" + baseDocuments->GetContractIdDate().FormatGmTime("%d.%m.%Y") + ".pdf";
            SubstGlobal(fileName, "/", ".");
            AddParameter(EOutput::FrameworkAgreementFileName, fileName);
        } else if (baseDocuments->GetLessor() == "ООО «МСИА»") {
            AddParameter(EOutput::FrameworkAgreementDate, "2020-11-24");
            AddParameter(EOutput::FrameworkAgreementNumber, "OLA 384-2020");
            AddParameter(EOutput::FrameworkAgreementFileName, "OLA-384-2020_24.11.20.pdf");
        } else if (baseDocuments->GetContractDate() < TInstant::ParseIso8601("2018-12-29")) {
            AddParameter(EOutput::FrameworkAgreementDate, "2017-12-19");
            AddParameter(EOutput::FrameworkAgreementNumber, "LO-054/2017");
            AddParameter(EOutput::FrameworkAgreementFileName, "LO-054-2017_19.12.2017.pdf");
        } else {
            AddParameter(EOutput::FrameworkAgreementDate, "2018-12-29");
            AddParameter(EOutput::FrameworkAgreementNumber, "10208844");
            AddParameter(EOutput::FrameworkAgreementFileName, "10208844_29.12.2018.pdf");
        }
    }
    if (Storage.GetParameter(EOutput::Model).empty()) {
        Storage.AddParameter(EOutput::Model, dataPtr->GetModel());
    }
    if (Storage.GetParameter(EOutput::Lessor).empty()) {
        Storage.AddParameter(EOutput::Lessor, "ООО «Мэйджор Профи»");
    }

    auto transaction = server.GetDriveAPI()->GetDatabase().CreateTransaction(true);

    TRecordsSet recordsSet;
    if (!transaction->Exec("SELECT mcu_firmware FROM car_telematics_state WHERE car_id='" + carId + "'", &recordsSet)->IsSucceed()) {
        errors.AddMessage("sensors", "Cannot receive data from car_telematics_state: " + transaction->GetErrors().GetStringReport());
        return false;
    }
    if (recordsSet.size() != 1) {
        Storage.AddParameter(EOutput::VegaVersion, ::ToString(NDrive::NVega::TFirmwareInfo::EType::MT32K));
    } else {
        const TString& mcuFirmware = recordsSet.GetRecords().front().Get("mcu_firmware");
        if (!mcuFirmware) {
            Storage.AddParameter(EOutput::VegaVersion, ::ToString(NDrive::NVega::TFirmwareInfo::EType::MT32K));
        } else {
            auto parsedInfo = NDrive::NVega::ParseFirmwareInfo(mcuFirmware);
            Storage.AddParameter(EOutput::VegaVersion, ToString(parsedInfo.Type));
        }
    }
    return true;
}
