#include "disk_templates.h"

#include <drive/backend/doc_packages/manager.h>

#include <drive/backend/database/drive_api.h>

#include <drive/library/cpp/disk/client.h>

using namespace NTemplateData;

TString TDiskTemplate::Name("disk_date");
ITemplateData::TFactory::TRegistrator<TDiskTemplate> TDiskTemplate::Registrator(TDiskTemplate::Name);

void TDiskTemplate::AddInputsToScheme(const IServerBase& /* server */, NDrive::TScheme& scheme) const {
    if (!scheme.HasField(ToString(EInput::Date))) {
        scheme.Add<TFSNumeric>(ToString(EInput::Date), "Дата").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
    }
}

bool TDiskTemplate::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) {
    TInstant date = TInstant::Seconds(json[ToString(EInput::Date)].GetUInteger());
    return Fetch(date, server, errors);
}

bool TDiskTemplate::FindActualDocument(const TInstant date, const TString& path, EDiskOutput value, const TYandexDiskClient& client, TMessagesCollector& errors, EDiskOutput hrValue, bool addHrValue) {
    TVector<IDocumentStorage::IDocumentStorageFile::TPtr> files;
    if (!client.GetFilesRecursive(path, path, files, errors)) {
        return false;
    }
    if (files.empty()) {
        errors.AddMessage(__LOCATION__, "Incorrect path " + path + " on disk (files not found)");
        return false;
    }

    TVector<THRDiskDate> fileDates;
    for (const auto& file : files) {
        TString fileName = file->GetName();
        const TString extension = ".pdf";
        if (!fileName.Contains(extension)) {
            errors.AddMessage(__LOCATION__, "Incorrect file " + fileName + " in " + path);
            return false;
        }
        fileName = fileName.substr(0, fileName.size() - extension.size());
        THRDiskDate fileDate;
        if (!THRDiskDate::TryFromString(fileName, fileDate)) {
            errors.AddMessage(__LOCATION__, "Incorrect file " + fileName + " in " + path);
            return false;
        }
        fileDates.emplace_back(fileDate);
    }

    std::sort(fileDates.begin(), fileDates.end());
    auto it = std::upper_bound(fileDates.begin(), fileDates.end(), THRDiskDate(date));
    if (it == fileDates.end()) {
        Storage.AddParameter(value, fileDates.back().ToString());
        if (addHrValue) {
            Storage.AddParameter(hrValue, THRDate(fileDates.back().GetTime()).ToString());
        }
        return true;
    } else if (it == fileDates.begin()) {
        errors.AddMessage(__LOCATION__, "the earliest date of " + ToString(value) + " is " + it->ToString());
        return false;
    }
    --it;
    Storage.AddParameter(value, it->ToString());
    if (addHrValue) {
        Storage.AddParameter(hrValue, THRDate(it->GetTime()).ToString());
    }
    return true;
}

bool TDiskTemplate::Fetch(const TInstant date, const NDrive::IServer& server, TMessagesCollector& errors) {
    if (!server.GetDriveAPI()->HasYandexDiskClient()) {
        errors.AddMessage(__LOCATION__, "Yandex.Disk client not found");
        return false;
    }
    if (!server.GetDocumentsManager()) {
        errors.AddMessage(__LOCATION__, "document manager not found");
        return false;
    }
    const auto& client = server.GetDriveAPI()->GetYandexDiskClient();
    const auto& managerConfig = server.GetDocumentsManager()->GetConfig();

    if (!FindActualDocument(date, managerConfig.GetRentTermsPath(), EOutput::RentTermsDate, client, errors, EDiskOutput::RentTermsDateHr, true)
        || !FindActualDocument(date, managerConfig.GetRentCertifiedTermsPath(), EOutput::RentCertifiedTermsDate, client, errors)
        || !FindActualDocument(date, managerConfig.GetServiceUsingTermsPath(), EOutput::ServiceUsingTermsDate, client, errors, EDiskOutput::ServiceUsingTermsDateHr, true)
        || !FindActualDocument(date, managerConfig.GetServiceUsingCertifiedTermsPath(), EOutput::ServiceUsingCertifiedTermsDate, client, errors)) {
        return false;
    }
    return true;
}
