#include "document_external.h"

#include <drive/backend/doc_packages/manager.h>
#include <library/cpp/string_utils/relaxed_escaper/relaxed_escaper.h>

#include <drive/backend/database/drive_api.h>

TDocumentDescription::TFactory::TRegistrator<TImageDocument> TImageDocument::Registrator("image_by_storage");
TDocumentDescription::TFactory::TRegistrator<TBinaryDocument> TBinaryDocument::Registrator("binary_by_storage");
TDocumentDescription::TFactory::TRegistrator<TTextDocument> TTextDocument::Registrator("text_by_storage");
TDocumentDescription::TFactory::TRegistrator<TPdfDocument> TPdfDocument::Registrator("pdf_by_storage");

bool IExternalDocument::DoParseMeta(const NJson::TJsonValue& meta) {
    Path = meta["path"].GetString();
    Storage = meta["storage"].GetString();
    return true;
}

NJson::TJsonValue IExternalDocument::DoSaveMeta() const {
    NJson::TJsonValue meta;
    meta["path"] = Path;
    meta["storage"] = Storage;
    return meta;
}

void IExternalDocument::AddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    TBase::AddPostParametersToScheme(scheme, server);
    auto serverImpl = server.GetAs<NDrive::IServer>();
    if (serverImpl
        && serverImpl->GetDocumentsManager()
        && serverImpl->GetDocumentsManager()->HasStorage(Storage))
    {
        serverImpl->GetDocumentsManager()->GetStorage(Storage)->AddInputsToScheme(scheme);
    }
}

bool IExternalDocument::DoCreateContent(const NJson::TJsonValue& postData,
                              const TMap<TString, ITemplateData::TPtr>& parameters,
                              const NDrive::IServer& server,
                              IDocumentAssembler& builder,
                              TMessagesCollector& errors) const
{
    if (!server.GetDocumentsManager()) {
        errors.AddMessage("external_document", "document manager is absent");
        return false;
    }
    if (!server.GetDocumentsManager()->HasStorage(Storage)) {
        errors.AddMessage("external_document", Storage + " storage not configured");
        return false;
    }

    if (!server.GetDocumentsManager()->GetStorage(Storage)) {
        errors.AddMessage("external_document", "unknown storage " + Storage);
        return false;
    }

    TString localPath = Path;
    for (auto&& p : parameters) {
        p.second->Substitude(localPath, {});
    }

    TString content;
    if (!server.GetDocumentsManager()->GetStorage(Storage)->GetValue(localPath, postData, content, server, errors)) {
        errors.AddMessage("external_document", "get_value function for " + localPath + " was failed");
        return false;
    }

    if (!ApplyTransformation(content, parameters, server, builder, errors)) {
        return false;
    }

    return builder.AddBlock(content, errors, GetBlockType(), GetBlockContext());
}

NDrive::TScheme IExternalDocument::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme result;
    result.Add<TFSString>("path", "Путь к файлу").SetRequired(true);

    auto serverImpl = server.GetAs<NDrive::IServer>();
    if (serverImpl && serverImpl->GetDocumentsManager()) {
        result.Add<TFSVariants>("storage", "Хранилище").SetVariants(serverImpl->GetDocumentsManager()->GetStoragesNames()).SetRequired(true);
    } else {
        result.Add<TFSString>("storage", "Хранилище").SetRequired(true);
    }
    return result;
}

bool TImageDocument::ApplyTransformation(TString& context, const TMap<TString, ITemplateData::TPtr>& /*parameters*/, const NDrive::IServer& server, IDocumentAssembler& builder, TMessagesCollector& errors) const {
    if (!server.GetDriveAPI() || !server.GetDocumentsManager()) {
        errors.AddMessage("apply_transformation", "document manager doesn't exist");
    }
    NImageTransformation::TTransformation tr(GetTransformationConfig(), builder.GetAuthorId());
    if (!tr.Transform(context, context, errors)) {
        return false;
    }
    return true;
}

bool TImageDocument::DoParseMeta(const NJson::TJsonValue& meta) {
    if (!IExternalDocument::DoParseMeta(meta)) {
        return false;
    }
    if (meta.Has("image_transformation") && !TransformationConfig.DeserializeFromJson(meta["image_transformation"])) {
        return false;
    }
    if (meta.Has("context") && !AddBlockContext.Parse(meta["context"])) {
        return false;
    }
    return true;
}

NJson::TJsonValue TImageDocument::DoSaveMeta() const {
    NJson::TJsonValue meta = IExternalDocument::DoSaveMeta();
    meta.InsertValue("image_transformation", TransformationConfig.SerializeToJson());
    meta.InsertValue("context", AddBlockContext.SerializeToJson());
    return meta;
}

NDrive::TScheme TImageDocument::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme result = IExternalDocument::DoGetScheme(server);
    result.Add<TFSStructure>("image_transformation", "Настройки преобразования фотографий").SetStructure(TransformationConfig.GetScheme());
    result.Add<TFSStructure>("context", "Контекст вставки").SetStructure(AddBlockContext.GetScheme());
    return result;
}

bool TTextDocument::ApplyTransformation(TString& context, const TMap<TString, ITemplateData::TPtr>& parameters, const NDrive::IServer& /*server*/, IDocumentAssembler& /*builder*/, TMessagesCollector& /*errors*/) const {
    for (auto&& p : parameters) {
        p.second->Substitude(context, GetDataEscape());
    }
    if (GetDocumentFormat() == "json_string") {
        context = NEscJ::EscapeJ<false>(context);
    }
    return true;
}
