#include "document_package.h"

#include <drive/backend/doc_packages/manager.h>

#include <drive/backend/database/drive_api.h>

TDocumentDescription::TFactory::TRegistrator<TDocumentsPackage> TDocumentsPackage::Registrator("package");

bool TDocumentsPackage::DoParseMeta(const NJson::TJsonValue& meta) {
    const NJson::TJsonValue& docs = meta["documents"];
    if (!docs.IsArray()) {
        return false;
    }
    for (auto&& doc : docs.GetArray()) {
        TString name;
        // old package description
        if (doc.GetType() == NJson::EJsonValueType::JSON_STRING) {
            name = doc.GetString();
        } else {
            name = doc["document_name"].GetStringRobust();
        }
        Documents.push_back(name);
    }
    return true;
}

NJson::TJsonValue TDocumentsPackage::DoSaveMeta() const {
    NJson::TJsonValue meta;
    NJson::TJsonValue& docs = meta["documents"];
    for (auto&& doc : Documents) {
        NJson::TJsonValue documentJson;
        documentJson["document_name"] = doc;
        docs.AppendValue(documentJson);
    }
    return meta;
}

bool TDocumentsPackage::DoCreateContent(const NJson::TJsonValue& postData,
                                      const TMap<TString, ITemplateData::TPtr>& templates,
                                      const NDrive::IServer& server,
                                      IDocumentAssembler& builder,
                                      TMessagesCollector& errors) const {
    const TDocumentsManager& manager = *server.GetDocumentsManager();
    TVector<TDocumentDescriptionPtr> packageDocs;
    TVector<TString> docs;
    for (auto doc : Documents) {
        for (auto&& t : templates) {
            t.second->Substitude(doc, {});
        }
        docs.emplace_back(std::move(doc));
    }
    if (!manager.GetRegisteredDocuments(packageDocs, MakeSet(docs))) {
        errors.AddMessage("fetch documents", "cannot fetch description from cache");
        return false;
    }

    TMap<TString, TDocumentDescriptionPtr> documentSet;
    for (auto&& doc : packageDocs) {
        documentSet.emplace(doc->GetName(), doc);
    }

    for (const auto& docName : docs) {
        auto it = documentSet.find(docName);
        if (it == documentSet.end()) {
            errors.AddMessage(__LOCATION__, "Can't fetch package document '" + docName + "'");
            return false;
        }
        if (!it->second->CreateContent(postData, templates, server, builder, errors)) {
            return false;
        }
    }
    return true;
}

NDrive::TScheme TDocumentsPackage::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme result;

    auto serverImpl = server.GetAs<NDrive::IServer>();
    TVector<TDocumentDescriptionPtr> documentsPtr;
    if (serverImpl && serverImpl->GetDocumentsManager()) {
        serverImpl->GetDocumentsManager()->GetRegisteredDocuments(documentsPtr);
    }
    TSet<TString> names;
    for (const auto& doc : documentsPtr) {
        names.insert(doc.GetInternalId());
    }

    result.Add<TFSArray>("documents", "Документы").SetElement<NDrive::TScheme>().template Add<TFSVariants>("document_name", "Имя документа").SetVariants(names).SetRequired(true);
    return result;
}

 bool TDocumentsPackage::AddTemplatesFromPostData(TMap<TString, ITemplateData::TPtr>& result, const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) const {
    if (GetTemplates()) {
        return TDocumentDescription::AddTemplatesFromPostData(result, json, server, errors);
    }
    if (!server.GetDocumentsManager()) {
        errors.AddMessage("document_manager", "document manager doesn't configured");
        return false;
    }
    TVector<TDocumentDescriptionPtr> documentsPtr;
    if (!server.GetDocumentsManager()->GetRegisteredDocuments(documentsPtr, MakeSet(Documents))) {
        errors.AddMessage("fetch_documents", "cannot fetch document description from cache");
        return false;
    }

    for (const auto& doc : documentsPtr) {
        if (!doc->AddTemplatesFromPostData(result, json, server, errors)) {
            return false;
        }
    }
    return true;
}

void TDocumentsPackage::AddTemplatesToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    if (GetTemplates()) {
        TDocumentDescription::AddTemplatesToScheme(scheme, server);
        return;
    }

    auto serverImpl = server.GetAs<NDrive::IServer>();

    TVector<TDocumentDescriptionPtr> documentsPtr;
    if (!serverImpl || !serverImpl->GetDocumentsManager() || !serverImpl->GetDocumentsManager()->GetRegisteredDocuments(documentsPtr, MakeSet(Documents))) {
        ERROR_LOG << "cannot fetch document description from cache" << Endl;
        return;
    }

    for (const auto& doc : documentsPtr) {
        doc->AddTemplatesToScheme(scheme, server);
    }
}

void TDocumentsPackage::AddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    auto serverImpl = server.GetAs<NDrive::IServer>();

    TVector<TDocumentDescriptionPtr> documentsPtr;
    if (!serverImpl || !serverImpl->GetDocumentsManager() || !serverImpl->GetDocumentsManager()->GetRegisteredDocuments(documentsPtr, MakeSet(Documents))) {
        ERROR_LOG << "cannot fetch document description from cache" << Endl;
        return;
    }

    for (const auto& doc : documentsPtr) {
        doc->AddPostParametersToScheme(scheme, server);
    }
}
