#include "document_repeated.h"

#include "car_template.h"
#include "session_templates.h"
#include "static_tools.h"
#include "user_templates.h"

#include <drive/backend/doc_packages/manager.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/saas/api.h>

#include <library/cpp/string_utils/url/url.h>

#include <rtline/api/graph/router/router.h>
#include <rtline/library/geometry/coord.h>
#include <rtline/library/geometry/rect.h>
#include <rtline/library/json/merge.h>
#include <rtline/util/types/interval.h>

TDocumentDescription::TFactory::TRegistrator<TRepeatedDocument> TRepeatedDocument::Registrator("repeated");
TDocumentDescription::TFactory::TRegistrator<TSessionPaymentsDocument> TSessionPaymentsDocument::Registrator("session_payments");
TDocumentDescription::TFactory::TRegistrator<TFiltredPaymentsDocument> TFiltredPaymentsDocument::Registrator("filtred_payments");
TDocumentDescription::TFactory::TRegistrator<TFiltredRefundsDocument> TFiltredRefundsDocument::Registrator("filtred_session_refunds");
TDocumentDescription::TFactory::TRegistrator<TFiltredBillsDocument> TFiltredBillsDocument::Registrator("filtred_session_bills");
TDocumentDescription::TFactory::TRegistrator<TSessionPhotosDocument> TSessionPhotosDocument::Registrator("session_photos");
TDocumentDescription::TFactory::TRegistrator<TFiltredSessionsDocument> TFiltredSessionsDocument::Registrator4("car_sessions");
TDocumentDescription::TFactory::TRegistrator<TFiltredSessionsDocument> TFiltredSessionsDocument::Registrator3("user_sessions");
TDocumentDescription::TFactory::TRegistrator<TFiltredSessionsDocument> TFiltredSessionsDocument::Registrator2("model_sessions");

TString TFiltredSessionsDocument::TypeName = "filtred_sessions";
TDocumentDescription::TFactory::TRegistrator<TFiltredSessionsDocument> TFiltredSessionsDocument::Registrator1(TFiltredSessionsDocument::TypeName);

bool IRepeatedDocument::DoParseMeta(const NJson::TJsonValue& meta) {
    Document = meta["document"].GetString();
    if (meta.Has("empty_policy") && !TryFromString(meta["empty_policy"].GetString(), EmptyPolicy)) {
        return false;
    }
    if (EmptyPolicy == EEmptyContentPolicy::DefaultDocument) {
        DefaultDocument = meta["default_document"].GetString();
        if (!DefaultDocument) {
            return false;
        }
    }
    if (meta.Has("subdocument_error_policy") && !TryFromString(meta["subdocument_error_policy"].GetString(), SubDocumentErrorPolicy)) {
        return false;
    }
    if (SubDocumentErrorPolicy == ESubDocumentErrorPolicy::DefaultDocument) {
        DefaultSubDocument = meta["default_subdocument"].GetString();
        if (!DefaultSubDocument) {
            return false;
        }
    }
    SeparatingDocument = meta["separating_document"].GetString();
    if (SeparatingDocument && SubDocumentErrorPolicy == ESubDocumentErrorPolicy::Skip) {
        return false;
    }
    if (!NJson::ParseField(meta, "postdata_in_subdoc", UsePostInSubDoc, /* required = */ false)) {
        return false;
    }
    return !!Document;
}

NJson::TJsonValue IRepeatedDocument::DoSaveMeta() const {
    NJson::TJsonValue meta;
    meta["document"] = Document;
    meta["empty_policy"] = ToString(EmptyPolicy);
    if (EmptyPolicy == EEmptyContentPolicy::DefaultDocument) {
        meta["default_document"] = DefaultDocument;
    }
    meta["subdocument_error_policy"] = ToString(SubDocumentErrorPolicy);
    if (SubDocumentErrorPolicy == ESubDocumentErrorPolicy::DefaultDocument) {
        meta["default_subdocument"] = DefaultSubDocument;
    }
    if (SeparatingDocument) {
        meta["separating_document"] = SeparatingDocument;
    }
    if (UsePostInSubDoc) {
        meta["postdata_in_subdoc"] = UsePostInSubDoc;
    }
    return meta;
}

bool GetDocumentPtr(TDocumentDescriptionPtr& document, const TString& name, const NDrive::IServer& server, TMessagesCollector& errors) {
    TVector<TDocumentDescriptionPtr> documentsPtr;
    if (!server.GetDocumentsManager() || !server.GetDocumentsManager()->GetRegisteredDocuments(documentsPtr, { name }) || documentsPtr.size() != 1) {
        errors.AddMessage(__LOCATION__, "cannot fetch " + name + " document description from cache");
        return false;
    }

    document = documentsPtr.front();
    return !!document;
}

bool IRepeatedDocument::UseExternalParameters() const {
    return true;
}

bool IRepeatedDocument::DoCreateContent(const NJson::TJsonValue& postData,
    const TMap<TString, ITemplateData::TPtr>& parameters,
    const NDrive::IServer& server,
    IDocumentAssembler& builder,
    TMessagesCollector& errors) const
{
    TDocumentDescriptionPtr contentBuilder;
    TDocumentDescriptionPtr defaultContentBuilder;
    TDocumentDescriptionPtr emptyContentBuilder;
    TDocumentDescriptionPtr separatingContentBuilder;
    if (!GetDocumentPtr(contentBuilder, Document, server, errors)
        || (SubDocumentErrorPolicy == ESubDocumentErrorPolicy::DefaultDocument && !GetDocumentPtr(defaultContentBuilder, DefaultSubDocument, server, errors))
        || (EmptyPolicy == EEmptyContentPolicy::DefaultDocument && !GetDocumentPtr(emptyContentBuilder, DefaultDocument, server, errors))
        || (SeparatingDocument && !GetDocumentPtr(separatingContentBuilder, SeparatingDocument, server, errors))) {
        return false;
    }

    TVector<NJson::TJsonValue> jsonParameters;
    if (!GenerateJsonParameters(jsonParameters, postData, server, errors)) {
        return false;
    }

    ui32 documentCount = 0;
    for (auto&& json : jsonParameters) {
        if (!!separatingContentBuilder && documentCount) {
            if (!separatingContentBuilder->CreateContentFromJson({}, parameters, server, builder, errors)) {
                return false;
            }
        }
        NJson::TJsonValue subDocData = json;
        if (UsePostInSubDoc) {
            subDocData = postData;
            NJson::MergeJson(json, subDocData);
        }
        if (!contentBuilder->CreateContentFromJson(subDocData, UseExternalParameters() ? parameters : TMap<TString, ITemplateData::TPtr>(), server, builder, errors)) {
            switch (SubDocumentErrorPolicy) {
            case ESubDocumentErrorPolicy::Skip:
                errors.ClearMessages();
                continue;
            case ESubDocumentErrorPolicy::Error:
                errors.AddMessage("subdocument_error", json.GetStringRobust());
                return false;
            case ESubDocumentErrorPolicy::DefaultDocument:
                if (!defaultContentBuilder->CreateContentFromJson(json, server, builder, errors)) {
                    errors.AddMessage("default_subdocument_error", json.GetStringRobust());
                    return false;
                }
                break;
            }
        }
        ++documentCount;
    }
    if (!documentCount) {
        switch (EmptyPolicy) {
        case EEmptyContentPolicy::Error:
            errors.AddMessage(__LOCATION__, "empty_document");
            return false;
        case EEmptyContentPolicy::DefaultDocument:
            if (!emptyContentBuilder->CreateContentFromJson(postData, parameters, server, builder, errors)) {
                errors.AddMessage("empty_default_document_error", "cannot construct empty default document");
                return false;
            }
            break;
        }
    }
    return true;
}

NDrive::TScheme IRepeatedDocument::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme result;

    auto serverImpl = server.GetAs<NDrive::IServer>();
    TVector<TDocumentDescriptionPtr> documentsPtr;
    if (serverImpl && serverImpl->GetDocumentsManager()) {
        serverImpl->GetDocumentsManager()->GetRegisteredDocuments(documentsPtr);
    }
    TSet<TString> names;
    for (const auto& doc : documentsPtr) {
        names.insert(doc.GetInternalId());
    }
    result.Add<TFSVariants>("document", "Документ для повторения").SetMultiSelect(false).SetVariants(names).SetRequired(true);
    result.Add<TFSVariants>("subdocument_error_policy", "Действия при ошибке формирования вложенного документа").InitVariants<ESubDocumentErrorPolicy>().SetRequired(true);
    result.Add<TFSVariants>("default_subdocument", "Дефолтный вложенный документ(если требуется)").SetVariants(names).SetRequired(false);
    result.Add<TFSVariants>("empty_policy", "Действия при пустом документе").InitVariants<EEmptyContentPolicy>().SetRequired(true);
    result.Add<TFSVariants>("default_document", "Дефолтный документ(если требуется)").SetVariants(names).SetRequired(false);
    result.Add<TFSVariants>("separating_document", "Разделяющий документ(если требуется)").SetVariants(names).SetRequired(false);
    result.Add<TFSBoolean>("postdata_in_subdoc", "Передавать параметры во вложенные документы").SetDefault(false);
    return result;
}

void IRepeatedDocument::AddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    TDocumentDescription::AddPostParametersToScheme(scheme, server);
    if (EmptyPolicy == EEmptyContentPolicy::DefaultDocument) {
        TDocumentDescriptionPtr emptyDocument;
        TMessagesCollector errors;
        if (!server.GetAs<NDrive::IServer>() || !GetDocumentPtr(emptyDocument, DefaultDocument, server.GetAsSafe<NDrive::IServer>(), errors)) {
            ERROR_LOG << errors.GetStringReport() << Endl;
            return;
        }
        emptyDocument->AddFullTemplatesScheme(scheme, server);
    }
    DoAddPostParametersToScheme(scheme, server);
    if (UsePostInSubDoc) {
        TDocumentDescriptionPtr document;
        auto serverImpl = server.GetAs<NDrive::IServer>();
        TMessagesCollector errors;
        if (!serverImpl || !GetDocumentPtr(document, GetDocument(), *serverImpl, errors)) {
            return;
        }
        for (const auto& t : document->GetTemplates()) {
            if (GetTemplates().contains(t)) {
                continue;
            }
            THolder<ITemplateData> templatePtr(ITemplateData::TFactory::Construct(t));
            if (templatePtr) {
                templatePtr->AddInputsToScheme(server, scheme);
            }
        }
    }
}

NDrive::TScheme TRepeatedDocument::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme result = IRepeatedDocument::DoGetScheme(server);
    result.Remove("postdata_in_subdoc");
    return result;
}

bool TRepeatedDocument::GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
    const NDrive::IServer& /*server*/,
    TMessagesCollector& errors) const {
    TString parameterName = "document_parameters_" + GetName();
    auto simplePost = postData;
    simplePost.EraseValue(parameterName);
    if (postData.Has(parameterName)) {
        for (auto element : postData[parameterName].GetArray()) {
            result.emplace_back(NJson::MergeJson(simplePost, element));
        }
        return true;
    }
    errors.AddMessage("parse_parameter", parameterName + " not found");
    return false;
}

void TRepeatedDocument::DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    TMessagesCollector errors;
    TDocumentDescriptionPtr document;
    TDocumentDescriptionPtr defaultDocument;
    auto serverImpl = server.GetAs<NDrive::IServer>();
    if (!serverImpl || !GetDocumentPtr(document, GetDocument(), *serverImpl, errors)
        || (GetSubDocumentErrorPolicy() == ESubDocumentErrorPolicy::DefaultDocument && !GetDocumentPtr(defaultDocument, GetDefaultSubDocument(), *serverImpl, errors))) {
        ERROR_LOG << errors.GetStringReport() << Endl;
        return;
    }
    TString inputName = "document_parameters_" + GetName();
    scheme.Remove(inputName);

    NDrive::TScheme subScheme;
    for (const auto& t : document->GetTemplates()) {
        if (GetTemplates().contains(t)) {
            continue;
        }
        THolder<ITemplateData> templatePtr(ITemplateData::TFactory::Construct(t));
        if (templatePtr) {
            templatePtr->AddInputsToScheme(server, subScheme);
        }
    }

    for (const auto& t : document->GetAdditionalParameters()) {
        if (GetAdditionalParameters().contains(t)) {
            continue;
        }
        if (!subScheme.HasField(t)) {
            subScheme.Add<TFSString>(t).SetRequired(true);
        }
    }
    document->AddOverrideParametersToScheme(subScheme, server);
    if (!!defaultDocument) {
        defaultDocument->AddFullTemplatesScheme(subScheme, server);
    }
    scheme.Add<TFSArray>(inputName, "Параметры документов для формирования " + GetComment()).SetElement(subScheme);
}

NDrive::TScheme IPaymentsDocument::DoGetScheme(const IServerBase& server) const {
    auto result = TBase::DoGetScheme(server);
    if (!result.HasField("yt_necessary_age")) {
        result.Add<TFSDuration>("yt_necessary_age", "Необходимость YT с текущего момента").SetDefault(YtNecessaryAge);
    }
    return result;
}

bool IPaymentsDocument::DoParseMeta(const NJson::TJsonValue& meta) {
    return
        NJson::ParseField(meta, "yt_necessary_age", YtNecessaryAge, /* required = */ false)
        && TBase::DoParseMeta(meta);
}

NJson::TJsonValue IPaymentsDocument::DoSaveMeta() const {
    auto meta = TBase::DoSaveMeta();
    meta.InsertValue("yt_necessary_age", NJson::ToJson(NJson::Hr(YtNecessaryAge)));
    return meta;
}

TString IPaymentsDocument::PaymentTypesKey = "payment_types";
TString IPaymentsDocument::StatusKey = "payment_statuses";
TString IPaymentsDocument::IgnoreCanceledDepositKey = "ignore_canceled_deposit";


bool IPaymentsDocument::GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const {
    TSet<NDrive::NBilling::EAccount> types;
    for (const auto& jsonType : postData[IPaymentsDocument::PaymentTypesKey].GetArray()) {
        NDrive::NBilling::EAccount type;
        if (TryFromString(jsonType.GetStringRobust(), type)) {
            types.emplace(type);
        }
    }
    if (types.empty()) {
        types.insert(NDrive::NBilling::EAccount::Trust);
    }

    TSet<NDrive::NTrustClient::EPaymentStatus> statuses;
    for (const auto& jsonStatus : postData[IPaymentsDocument::StatusKey].GetArray()) {
        NDrive::NTrustClient::EPaymentStatus status;
        if (TryFromString(jsonStatus.GetStringRobust(), status)) {
            statuses.emplace(status);
        }
    }

    auto payments = FetchPayments(postData, server, errors);
    if (!payments) {
        return false;
    }

    for (auto&& payment : *payments) {
        if (!types.contains(payment.GetPaymentType())) {
            continue;
        }

        if (TBillingGlobals::FailedStatuses.contains(payment.GetStatus()) && !statuses.contains(payment.GetStatus())) {
            continue;
        }

        if (!postData.Has(IPaymentsDocument::IgnoreCanceledDepositKey) || postData[IPaymentsDocument::IgnoreCanceledDepositKey].GetBoolean()) {
            if (payment.GetBillingType() == EBillingType::Deposit && payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Canceled) {
                continue;
            }
        }

        if (!statuses.empty() && !statuses.contains(payment.GetStatus())) {
            continue;
        }

        NJson::TJsonValue fullJson = postData;
        NJson::TJsonValue& jsonPayment = fullJson["payment"];
        payment.ToJson(jsonPayment);

        result.emplace_back(fullJson);
    }
    return true;
}

void IPaymentsDocument::DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    if (GetSubDocumentErrorPolicy() == ESubDocumentErrorPolicy::DefaultDocument) {
        TMessagesCollector errors;
        TDocumentDescriptionPtr defaultDocument;
        if (!server.GetAs<NDrive::IServer>() || !GetDocumentPtr(defaultDocument, GetDefaultSubDocument(), server.GetAsSafe<NDrive::IServer>(), errors)) {
            ERROR_LOG << errors.GetStringReport() << Endl;
            return;
        }
        defaultDocument->AddFullTemplatesScheme(scheme, server);
    }
    if (!scheme.HasField(IPaymentsDocument::IgnoreCanceledDepositKey)) {
        scheme.Add<TFSBoolean>(IPaymentsDocument::IgnoreCanceledDepositKey, "Игнорировать отмененные депозиты").SetDefault(true);
    }
    if (!scheme.HasField(StatusKey)) {
        scheme.Add<TFSVariants>(IPaymentsDocument::StatusKey, "Статус транзакций").InitVariants<NDrive::NTrustClient::EPaymentStatus>().SetMultiSelect(true);
    }
    if (!scheme.HasField(IPaymentsDocument::PaymentTypesKey)) {
        scheme.Add<TFSVariants>(IPaymentsDocument::PaymentTypesKey, "Тип транзакций").InitVariants<NDrive::NBilling::EAccount>().SetMultiSelect(true);
    }
}

TOptionalPayments TSessionPaymentsDocument::FetchPayments(const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const {
    TCachedPayments payments;
    TString sessionId = postData[ToString(ESessionInput::SessionId)].GetString();
    {
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server.GetDriveAPI()->GetBillingManager().GetPaymentsManager().GetPayments(payments, sessionId, session)) {
            errors.AddMessage("payments_manager", "get payments failed: " + session.GetStringReport());
            return {};
        }
    }
    bool needYt = true;
    {
        TVector<THistoryRideObject> rides;
        if (!NDocumentManager::GetHistoryRides(sessionId, "", "", TInstant::Zero(), TInstant::Max(), rides, server, errors)) {
            errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить сессию из compiled_rides:" + sessionId);
            return {};
        }
        needYt = rides.empty() || rides.front().GetStartTS() + GetYtNecessaryAge() < Now();
    }
    if (needYt && !NDocumentManager::GetSessionPaymentsFromYT(sessionId, payments, server, errors)) {
        return {};
    }
    return payments.GetTimeline();
}

void TFiltredPaymentsDocument::DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    TBase::DoAddPostParametersToScheme(scheme, server);
    if (!scheme.HasField("rrn")) {
        scheme.Add<TFSString>("rrn", "RRN");
    }
    if (!scheme.HasField("card_mask")) {
        scheme.Add<TFSString>("card_mask", "Маска карты");
    }
    if (!scheme.HasField("sum")) {
        scheme.Add<TFSNumeric>("sum", "Сумма").SetVisual(TFSNumeric::EVisualType::Money);
    }
    if (!scheme.HasField("time_from")) {
        scheme.Add<TFSNumeric>("time_from", "Начало интервала").SetVisual(TFSNumeric::EVisualType::DateTime);
    }
    if (!scheme.HasField("time_to")) {
        scheme.Add<TFSNumeric>("time_to", "Окончание интервала").SetVisual(TFSNumeric::EVisualType::DateTime);
    }
}

TOptionalPayments TFiltredPaymentsDocument::FetchPayments(const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const {
    const TString rrn = postData["rrn"].GetString();
    if (rrn) {
        {
            auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto payments = server.GetDriveAPI()->GetBillingManager().GetPaymentsManager().GetPayments(session, NSQL::TQueryOptions().AddGenericCondition("rrn", rrn));
            if (!payments) {
                errors.AddMessage("payments_manager", "get payments failed: " + session.GetStringReport());
                return {};
            }
            if (!payments->empty()) {
                return payments;
            }
        }
        return NDocumentManager::GetPaymentByRRN(rrn, server, errors);
    }
    const TString mask = postData["card_mask"].GetString();
    if (mask.empty()) {
        errors.AddMessage("ui_errors", "Не указана маска карты");
        return {};
    }
    const ui64 sum = postData["sum"].GetUInteger();
    if (!sum) {
        errors.AddMessage("ui_errors", "Не указана сумма списания");
        return {};
    }
    TInstant since = TInstant::Seconds(postData["time_from"].GetUInteger());
    if (!since) {
        errors.AddMessage("ui_errors", "Не указано начало интервала");
        return {};
    }
    TInstant until = TInstant::Seconds(postData["time_to"].GetUInteger());
    TOptionalPayments payments;
    if (!until || until + GetYtNecessaryAge() > Now()) {
        auto options = NSQL::TQueryOptions()
            .AddGenericCondition("card_mask", mask)
            .SetGenericCondition("sum", TSet<ui64>{ sum })
            .SetGenericCondition("last_update_ts", TRange<ui64>(since.Seconds(), until.Seconds()));
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        payments = server.GetDriveAPI()->GetBillingManager().GetPaymentsManager().GetPayments(session, options);
        if (!payments) {
            errors.AddMessage("payments_manager", "get payments failed: " + session.GetStringReport());
            return {};
        }
    }
    if (since + GetYtNecessaryAge() < Now()) {
        TOptionalPayments ytPayments = NDocumentManager::GetPaymentsByCard(mask, sum, {since, until}, server, errors);
        if (!ytPayments) {
            return {};
        }
        if (!payments) {
            SortBy(ytPayments->begin(), ytPayments->end(), [](const auto& payment) { return payment.GetLastUpdate(); });
            return ytPayments;
        }
        TSet<TString> knownPayments;
        Transform(payments->begin(), payments->end(), std::inserter(knownPayments, knownPayments.begin()), [](const auto& payment) { return payment.GetPaymentId(); });
        for (auto&& payment : *ytPayments) {
            if (!knownPayments.contains(payment.GetPaymentId())) {
                payments->emplace_back(std::move(payment));
            }
        }
        SortBy(payments->begin(), payments->end(), [](const auto& payment) { return payment.GetLastUpdate(); });
    }
    return payments;
}

bool TFiltredRefundsDocument::GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
    const NDrive::IServer& server,
    TMessagesCollector& errors) const {

    TInstant since;
    TInstant until = ModelingNow();
    if (!NJson::ParseField(postData, "bill_since", since, true, errors) || !NJson::ParseField(postData, "bill_until", until, false, errors)) {
        return false;
    }

    const TBillingManager& manager = server.GetDriveAPI()->GetBillingManager();

    TVector<TObjectEvent<NDrive::NBilling::TCompiledImpl>> compiledBills;
    {
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!NDrive::NBilling::GetEventsFromDB<TCompiledRefund>(since, until, compiledBills, session)) {
            errors.AddMessage("ui_errors", "Внутренняя ошибка при получении возвратов");
            return false;
        }
    }

    NDrive::NBilling::TAccountParentFilter accountFilter;
    accountFilter.WithParent = true;
    if (!NJson::ParseField(postData, "with_bonus_accounts", accountFilter.WithBonusAccounts, false, errors)
        || !NJson::ParseField(postData, "parent_id", accountFilter.ParentIds, false, errors)
        || !NJson::ParseField(postData, "accounts", accountFilter.Accounts, false, errors)) {
        return false;
    }

    auto availableAccounts = manager.GetAccountsManager().GetAccountsChildren(accountFilter);
    if (!availableAccounts) {
        errors.AddMessage("ui_errors", "Внутренняя ошибка: невозможно получить аккаунты");
        return false;
    }

    TSet<EBillingType> billingTypes;
    JREAD_CONTAINER_OPT(postData, "billing_types", billingTypes);

    for (const auto& bill : compiledBills) {
        if (!billingTypes.contains(bill.GetBillingType())) {
            continue;
        }

        if (bill.GetBill() == 0) {
            continue;
        }

        for (auto&& item : bill.GetDetails().GetItems()) {
            auto itAccountParent = availableAccounts->find(item.GetUniqueName());
            if (itAccountParent == availableAccounts->end()) {
                continue;
            }
            if (item.GetSum() == 0) {
                continue;
            }
            ui64 promocodeSum = 0;
            TString paymentType = ::ToString(item.GetType());
            ui64 totalSum = item.GetSum();
            if (item.GetType() == NDrive::NBilling::EAccount::Bonus) {
                promocodeSum = item.GetSum();
                paymentType = "card";
                totalSum = 0;
            }

            NJson::TJsonValue fullJson = postData;
            fullJson["session_id"] = bill.GetSessionId();
            NJson::TJsonValue& refundJson = fullJson["refund"];

            refundJson.InsertValue("session_id", bill.GetSessionId());
            refundJson.InsertValue("bill", totalSum);
            refundJson.InsertValue("billing_type", ::ToString(bill.GetBillingType()));
            refundJson.InsertValue("promocode_sum", promocodeSum);
            refundJson.InsertValue("timestamp", bill.GetHistoryInstant().Seconds());
            refundJson.InsertValue("external_id", item.GetTransactionId());
            refundJson.InsertValue("payment_type", paymentType);
            refundJson.InsertValue("account_name", item.GetUniqueName());
            result.emplace_back(fullJson);
        }
    }
    return true;
}

void TFiltredRefundsDocument::DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    if (GetSubDocumentErrorPolicy() == ESubDocumentErrorPolicy::DefaultDocument) {
        TMessagesCollector errors;
        TDocumentDescriptionPtr defaultDocument;
        if (!server.GetAs<NDrive::IServer>() || !GetDocumentPtr(defaultDocument, GetDefaultSubDocument(), server.GetAsSafe<NDrive::IServer>(), errors)) {
            ERROR_LOG << errors.GetStringReport() << Endl;
            return;
        }
        defaultDocument->AddFullTemplatesScheme(scheme, server);
    }
    scheme.Add<TFSArray>("parent_id", "Родительские кошельки").SetElement<TFSNumeric>();
    scheme.Add<TFSBoolean>("with_bonus_accounts", "Подклеить все операции с бонусами").SetDefault(false);
    scheme.Add<TFSVariants>("accounts", "Aккаунты для отгрузки").SetMultiSelect(true).SetReference("accounts");
    scheme.Add<TFSVariants>("billing_types", "Виды платежей").InitVariants<EBillingType>().SetMultiSelect(true);
    scheme.Add<TFSNumeric>("bill_since", "Начало периода").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
    scheme.Add<TFSNumeric>("bill_until", "Окончание периода").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSBoolean>("partial_bill", "Использовать частичные счета").SetDefault(false);
}

bool TFiltredBillsDocument::GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
    const NDrive::IServer& server,
    TMessagesCollector& errors) const {

    TInstant since;
    TInstant now = ModelingNow();
    TInstant until = now;
    bool partialBill = false;

    if (!NJson::ParseField(postData, "bill_since", since, true, errors)
    || !NJson::ParseField(postData, "bill_until", until, false, errors)
    || !NJson::ParseField(postData, "partial_bill", partialBill, false, errors)) {
        return false;
    }

    TSet<EBillingType> billingTypes;
    JREAD_CONTAINER_OPT(postData, "billing_types", billingTypes);

    if (!server.GetDriveAPI()->HasBillingManager()) {
        errors.AddMessage("ui_errors", "Внутренняя ошибка: нет BillingManager-a");
        return false;
    }

    const TBillingManager& manager = server.GetDriveAPI()->GetBillingManager();

    NDrive::NBilling::TAccountParentFilter accountFilter;
    accountFilter.WithParent = true;
    if (!NJson::ParseField(postData, "with_bonus_accounts", accountFilter.WithBonusAccounts, false, errors)
        || !NJson::ParseField(postData, "parent_id", accountFilter.ParentIds, false, errors)
        || !NJson::ParseField(postData, "accounts", accountFilter.Accounts, false, errors)) {
        return false;
    }

    auto availableAccounts = manager.GetAccountsManager().GetAccountsChildren(accountFilter);
    if (!availableAccounts) {
        errors.AddMessage("ui_errors", "Внутренняя ошибка: невозможно получить аккаунты");
        return false;
    }
    if (availableAccounts->empty()) {
        return true;
    }

    auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    NSQL::TQueryOptions options;
    options.SetOrderBy({
        "history_event_id",
        "history_timestamp"
    });

    for (auto&& type : billingTypes) {
        options.AddGenericCondition("billing_type", ToString(type));
    }

    if (availableAccounts->size() < 10) {
        options.SetGenericCondition("billing_wallets", MakeSet(NContainer::Keys(*availableAccounts)));
    }

    auto optionalBills = manager.GetCompiledBills().GetEvents({}, { since, until }, session, options);
    if (!optionalBills) {
        errors.AddMessage("ui_errors", "Внутренняя ошибка при получении счетов " + session.GetStringReport());
        return false;
    }

    TMaybe<TMap<TString, TCompiledBill>> optionalFullBills;
    if (partialBill) {
        optionalFullBills = manager.GetCompiledBills().GetFullBills(*optionalBills, session);
    } else {
        TVector<TString> finalSessions;
        for (auto&& event : *optionalBills) {
            finalSessions.push_back(event.GetSessionId());
        }

        optionalFullBills = manager.GetCompiledBills().GetFullBillsFromDB(finalSessions, session);
    }

    if (!optionalFullBills) {
        errors.AddMessage("ui_errors", "Внутренняя ошибка при получении полных счетов " + session.GetStringReport());
        return false;
    }

    auto compiledBills = MakeVector(NContainer::Values(*optionalFullBills));
    Sort(compiledBills.begin(), compiledBills.end(), [](const TCompiledBill& l, const TCompiledBill& r) {return l.GetFinalTime() < r.GetFinalTime();});

    for (const auto& bill : compiledBills) {
        if (until < now && (!bill.GetFinal() || bill.GetFinalTime() > until)) {
            continue;
        }
        if (!partialBill && !bill.GetFinal()) {
            continue;
        }
        for (auto&& item : bill.GetDetails().GetItems()) {
            auto itAccountParent = availableAccounts->find(item.GetUniqueName());
            if (itAccountParent == availableAccounts->end()) {
                continue;
            }

            ui64 promocodeSum = 0;
            TString paymentType = ::ToString(item.GetType());
            ui64 totalSum = item.GetSum();
            if (item.GetType() == NDrive::NBilling::EAccount::Bonus) {
                promocodeSum = item.GetSum();
                paymentType = "card";
                totalSum = 0;
            }

            NJson::TJsonValue fullJson = postData;
            fullJson["session_id"] = bill.GetRealSessionId();
            NJson::TJsonValue& refundJson = fullJson["refund"];

            refundJson.InsertValue("session_id", bill.GetSessionId());
            refundJson.InsertValue("bill", totalSum);
            refundJson.InsertValue("billing_type", ::ToString(bill.GetBillingType()));
            refundJson.InsertValue("promocode_sum", promocodeSum);
            refundJson.InsertValue("timestamp", bill.GetFinalTime().Seconds());
            refundJson.InsertValue("external_id", item.GetTransactionId());
            refundJson.InsertValue("payment_type", paymentType);
            refundJson.InsertValue("account_name", item.GetUniqueName());
            result.emplace_back(fullJson);
        }
    }
    return true;
}

void TFiltredBillsDocument::DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    if (GetSubDocumentErrorPolicy() == ESubDocumentErrorPolicy::DefaultDocument) {
        TMessagesCollector errors;
        TDocumentDescriptionPtr defaultDocument;
        if (!server.GetAs<NDrive::IServer>() || !GetDocumentPtr(defaultDocument, GetDefaultSubDocument(), server.GetAsSafe<NDrive::IServer>(), errors)) {
            ERROR_LOG << errors.GetStringReport() << Endl;
            return;
        }
        defaultDocument->AddFullTemplatesScheme(scheme, server);
    }
    scheme.Add<TFSArray>("parent_id", "Родительские кошельки").SetElement<TFSNumeric>();
    scheme.Add<TFSBoolean>("with_bonus_accounts", "Подклеить все операции с бонусами").SetDefault(false);
    scheme.Add<TFSVariants>("accounts", "Aккаунты для отгрузки").SetMultiSelect(true).SetReference("accounts");
    scheme.Add<TFSVariants>("billing_types", "Виды платежей").InitVariants<EBillingType>().SetMultiSelect(true);
    scheme.Add<TFSNumeric>("bill_since", "Начало периода").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSNumeric>("bill_until", "Окончание периода").SetVisual(TFSNumeric::EVisualType::DateTime);
}

bool TSessionPhotosDocument::GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
    const NDrive::IServer& server,
    TMessagesCollector& errors) const {
    TString sessionId = postData[ToString(ESessionInput::SessionId)].GetString();

    try {
        auto reader = server.GetDocumentsManager()->GetPaymentsTableReaderBySession(sessionId);
        for (; reader->IsValid(); reader->Next()) {
            NYT::TNode inputRow = reader->GetRow();
            if (inputRow["session_id"].AsString() == sessionId) {
                if (inputRow.HasKey("photos") && inputRow["photos"].IsList()) {
                    for (const auto& photo : inputRow["photos"].AsList()) {
                        NJson::TJsonValue json;
                        TString host, path;
                        SplitUrlToHostAndPath(photo.AsString(), host, path);
                        path = path.substr(path.find_first_not_of("/"));
                        json.InsertValue("photo_link", path);

                        host = host.substr(host.rfind("/") + 1);
                        json.InsertValue("photo_bucket", host.substr(0, host.find('.')));
                        result.emplace_back(std::move(json));
                    }
                    return true;
                }
                break;
            }
        }
    } catch (const std::exception& e) {
        errors.AddMessage("YTError", FormatExc(e));
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        return false;
    }
    errors.AddMessage("ui_errors", "Фотографии для сессии с идентификатором " + sessionId + " не найдены");
    return false;
}

void TSessionPhotosDocument::DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    if (!scheme.HasField("session_id")) {
        scheme.Add<TFSString>("session_id", "Идентификатор сессии").SetRequired(true);
    }
    if (GetSubDocumentErrorPolicy() == ESubDocumentErrorPolicy::DefaultDocument) {
        TMessagesCollector errors;
        TDocumentDescriptionPtr defaultDocument;
        if (!server.GetAs<NDrive::IServer>() || !GetDocumentPtr(defaultDocument, GetDefaultSubDocument(), server.GetAsSafe<NDrive::IServer>(), errors)) {
            ERROR_LOG << errors.GetStringReport() << Endl;
            return;
        }
        defaultDocument->AddFullTemplatesScheme(scheme, server);
    }
}

NJson::TJsonValue TFiltredSessionsDocument::TContext::SerializeToJson() const {
    NJson::TJsonValue json;
    NJson::InsertField(json, ToString(ESessionInput::SessionId), SessionId);
    NJson::InsertField(json, ToString(TCarTemplate::EInput::CarId), CarId);
    NJson::InsertField(json, ToString(EUserInput::UserId), UserId);

    NJson::InsertField(json, "models", Models);
    TJsonProcessor::WriteDurationString(json, "min_session_duration", MinDuration);
    NJson::InsertField(json, "offers_filter", OffersFilter);
    TJsonProcessor::WriteInstant(json, "session_start_timestamp", Since);
    TJsonProcessor::WriteInstant(json, "session_finish_timestamp", Until);
    NJson::InsertField(json, "polyline", Polyline);
    return json;
}

bool TFiltredSessionsDocument::TContext::DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors) {
    JREAD_STRING_OPT(json, ToString(ESessionInput::SessionId), SessionId);
    JREAD_STRING_OPT(json, ToString(TCarTemplate::EInput::CarId), CarId);
    JREAD_STRING_OPT(json, ToString(EUserInput::UserId), UserId);

    if (!NJson::ParseField(json, "models", Models, false)) {
        errors.AddMessage("ui_errors", "Ошибка парсинга: некорректный формат поля models");
        return false;
    }

    if (!NJson::ParseField(json, "min_session_duration", MinDuration, false)) {
        errors.AddMessage("ui_errors", "Ошибка парсинга: некорректно указана минимальная продолжительность сессии");
        return false;
    }

    if (!NJson::ParseField(json, "offers_filter", OffersFilter, false)) {
        errors.AddMessage("ui_errors", "Ошибка парсинга фильтра офферов");
        return false;
    }

    if (!SessionId) {
        if (!NJson::ParseField(json, "session_start_timestamp", Since, true)) {
            errors.AddMessage("ui_errors", "Не указано начало интервала сессий");
            return false;
        }
        Until = Since + TDuration::Minutes(1);
        if (!NJson::ParseField(json, "session_finish_timestamp", Until, false)) {
            errors.AddMessage("ui_errors", "Ошибка парсинга: некорректный формат поля session_finish_timestamp");
            return false;
        }
    }

    if (!NJson::ParseField(json, "polyline", Polyline, false) || Polyline.size() == 1) {
        errors.AddMessage("ui_errors", "Ошибка парсинга: некорректный формат поля polyline");
        return false;
    }
    return true;
}

void TFiltredSessionsDocument::TContext::AddScheme(NDrive::TScheme& scheme, const IServerBase& server) {
    if (!scheme.HasField(ToString(ESessionInput::SessionId))) {
        scheme.Add<TFSString>(ToString(ESessionInput::SessionId), "Идентификатор сессии").SetVisual(TFSString::EVisualType::GUID);
    }
    if (!scheme.HasField(ToString(EUserInput::UserId))) {
        scheme.Add<TFSString>(ToString(EUserInput::UserId), "Пользователь").SetVisual(TFSString::EVisualType::UUID);
    }
    if (!scheme.HasField(ToString(TCarTemplate::EInput::CarId))) {
        scheme.Add<TFSString>(ToString(TCarTemplate::EInput::CarId), "Автомобиль").SetVisual(TFSString::EVisualType::UUID);
    }

    auto serverImpl = server.GetAs<NDrive::IServer>();

    if (!scheme.HasField("models")) {
        TSet<TString> modelIds;
        if (serverImpl && serverImpl->GetDriveAPI() && serverImpl->GetDriveAPI()->GetModelsData()) {
            modelIds = MakeSet(NContainer::Keys(serverImpl->GetDriveAPI()->GetModelsData()->GetCached().GetResult()));
        }
        scheme.Add<TFSVariants>("models", "модели авто").SetMultiSelect(true).SetVariants(modelIds);
    }

    if (!scheme.HasField("offers_filter")) {
        scheme.Add<TFSVariants>("offers_filter", "Фильтр офферов").SetVariants(IOfferBuilderAction::GetNames(serverImpl)).SetMultiSelect(true);
    }
    if (!scheme.HasField("min_session_duration")) {
        scheme.Add<TFSDuration>("min_session_duration", "Минимальная продолжительность сессии");
    }

    if (!scheme.HasField("session_start_timestamp")) {
        scheme.Add<TFSNumeric>("session_start_timestamp", "Начало интервала сессий").SetVisual(TFSNumeric::EVisualType::DateTime);
    }
    if (!scheme.HasField("session_finish_timestamp")) {
        scheme.Add<TFSNumeric>("session_finish_timestamp", "Окончание интервала сессий").SetVisual(TFSNumeric::EVisualType::DateTime);
    }

    if (!scheme.HasField("polyline")) {
        scheme.Add<TFSJson>("polyline", "Полигон");
    }
}

void TFiltredSessionsDocument::DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    TContext::AddScheme(scheme, server);
    if (GetSubDocumentErrorPolicy() == ESubDocumentErrorPolicy::DefaultDocument) {
        TMessagesCollector errors;
        TDocumentDescriptionPtr defaultDocument;
        if (!server.GetAs<NDrive::IServer>() || !GetDocumentPtr(defaultDocument, GetDefaultSubDocument(), server.GetAsSafe<NDrive::IServer>(), errors)) {
            ERROR_LOG << errors.GetStringReport() << Endl;
            return;
        }
        defaultDocument->AddFullTemplatesScheme(scheme, server);
    }
}

bool TFiltredSessionsDocument::GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
    const NDrive::IServer& server,
    TMessagesCollector& errors) const {
    const TString guid = CreateGuidAsString();
    NDrive::TEventLog::Log("GenerateDocParameters", NJson::TMapBuilder
        ("status", "start")
        ("post_data", postData)
        ("guid", guid)
    );

    TContext context;
    if (!context.DeserializeFromJson(postData, errors)) {
        return false;
    }

    TVector<TMinimalCompiledRiding> filtredRides;
    if (context.GetSessionId()) {
        TObjectEvent<TMinimalCompiledRiding> ride;
        if (!NDocumentManager::GetMinimalCompiledRiding(context.GetSessionId(), ride, server, errors)) {
            return false;
        }
        if (!context.GetMinDuration() || ride.GetDuration() > context.GetMinDuration()) {
            if (context.GetOffersFilter()) {
                TObjectEvent<TFullCompiledRiding> fullRide;
                if (!NDocumentManager::GetFullCompiledRiding(context.GetSessionId(), fullRide, server, errors)) {
                    return false;
                }
                if (fullRide.GetOffer() && (context.GetOffersFilter().contains(fullRide.GetOffer()->GetBehaviourConstructorId()) || context.GetOffersFilter().contains(fullRide.GetOffer()->GetPriceConstructorId()))) {
                    filtredRides.emplace_back(ride);
                }
            } else {
                filtredRides.emplace_back(ride);
            }
        }
    } else {
        TVector<TMinimalCompiledRiding> rides;
        if (context.GetUserId()) {
            if (!NDocumentManager::GetCompiledRidingsByUser(context.GetUserId(), context.GetSince(), context.GetUntil(), rides, server, errors)) {
                return false;
            }
        } else if (context.GetCarId()) {
            if (!NDocumentManager::GetCompiledRidingsByCar(context.GetCarId(), context.GetSince(), context.GetUntil(), rides, server, errors)) {
                return false;
            }
        } else {
            if (!NDocumentManager::GetCompiledRidingsByInterval(context.GetSince(), context.GetUntil(), rides, server, errors)) {
                return false;
            }
        }

        TCarsDB::TFetchResult gCars;
        if (!context.GetCarId() && context.GetModels()) {
            TSet<TString> allCars;
            for (const auto& ride : rides) {
                allCars.emplace(ride.GetObjectId());
            }
            gCars = server.GetDriveAPI()->GetCarsData()->FetchInfo(allCars, TInstant::Zero());
            if (!gCars) {
                errors.AddMessage("ui_message", "Не удалось восстановить автомобили");
                return false;
            }
        }
        for (const auto& ride : rides) {
            if (context.GetMinDuration() && ride.GetDuration() < context.GetMinDuration()) {
                continue;
            }
            if (context.GetCarId() && ride.GetObjectId() != context.GetCarId()) {
                continue;
            }
            if (!context.GetCarId() && context.GetModels()) {
                auto* carPtr = gCars.GetResultPtr(ride.GetObjectId());
                if (!carPtr || !context.GetModels().contains(carPtr->GetModel())) {
                    continue;
                }
            }
            if (context.GetOffersFilter()) {
                TObjectEvent<TFullCompiledRiding> fullRide;
                if (!NDocumentManager::GetFullCompiledRiding(ride.GetSessionId(), fullRide, server, errors)) {
                    return false;
                }
                if (!fullRide.GetOffer() || (!context.GetOffersFilter().contains(fullRide.GetOffer()->GetBehaviourConstructorId()) && !context.GetOffersFilter().contains(fullRide.GetOffer()->GetPriceConstructorId()))) {
                    continue;
                }
            }
            filtredRides.emplace_back(ride);
        }
        NDrive::TEventLog::Log("GenerateDocParameters", NJson::TMapBuilder
            ("status", "car_filtering")
            ("guid", guid)
            ("filtred_rides_size", filtredRides.size())
            ("total_rides_size", rides.size())
        );
    }

    if (filtredRides.size() && context.GetPolyline()) {
        ::TRect<TGeoCoord> rect(context.GetPolyline().front());
        rect.ExpandTo(context.GetPolyline());

        TInterval<ui32> intervalTime;
        intervalTime.SetMin(context.GetSince().Seconds());
        intervalTime.SetMax(context.GetUntil().Seconds());

        try {
            TSet<TString> sessionIds;
            TInstant maxFinishTimestamp = context.GetSince();
            TInstant minFinishTimestamp = TInstant::Max();
            const TDuration limit = server.GetSettings().GetValueDef<TDuration>("filtred_sessions_document.session_finish_limit", TDuration::Minutes(30));
            TInstant maxFinish = context.GetUntil() + limit;
            TVector<TInstant> oldFinishes;
            for (const auto& session : filtredRides) {
                sessionIds.emplace(session.GetSessionId());
                maxFinishTimestamp = Max(maxFinishTimestamp, Min(maxFinish, session.GetFinishInstant().Get()));
                minFinishTimestamp = Min(minFinishTimestamp, session.GetFinishInstant().Get());
                if (session.GetFinishInstant().Get() > maxFinish) {
                    oldFinishes.push_back(session.GetFinishInstant().Get());
                }
            }
            ui64 tracksCount = 0;
            ui64 totalTracksCount = 0;
            const auto lastTimestamp = server.GetDocumentsManager()->GetLastTracksTimestamp();
            if (context.GetUntil() > lastTimestamp) {
                errors.AddMessage("YTError", "Повторите попытку позднее, недостаточно данных");
                return false;
            }
            TVector<TString> finishesConditions;
            finishesConditions.emplace_back(ToString(minFinishTimestamp.Seconds()) + ":" + ToString(maxFinishTimestamp.Seconds()));
            {
                const ui32 batch = server.GetSettings().GetValueDef<ui32>("filtred_sessions_document.batch_size", 1000);
                while (!oldFinishes.empty()) {
                    TString cond;
                    for (ui32 i = 0; i < Min((ui32)oldFinishes.size(), batch); ++i) {
                        if (cond) {
                            cond += ",";
                        }
                        cond += ToString(oldFinishes.back().Seconds());
                        oldFinishes.pop_back();
                    }
                    finishesConditions.emplace_back(std::move(cond));
                }

            }
            for (auto&& cond : finishesConditions) {
                auto reader = server.GetDocumentsManager()->GetTracksTableReaderByTimestamp(cond);
                for (; reader->IsValid(); reader->Next()) {
                    ++totalTracksCount;
                    NYT::TNode inputRow = reader->GetRow();
                    if (!sessionIds.contains(inputRow["session_id"].AsString())) {
                        continue;
                    }
                    ++tracksCount;
                    NGraph::TRouter::TTimedGeoCoordinates ytCoordinates;
                    const auto& route = inputRow["raw_route"].AsList();
                    for (const auto& pointData : route) {
                        if (pointData.Size() < 3) {
                            errors.AddMessage("ui_errors", "Некорректные данные трека для сессии " + inputRow["session_id"].AsString());
                            return false;
                        }

                        TSpeed speed = 1;
                        if (pointData.Size() > 3) {
                            speed = pointData[3].AsDouble();
                            if (speed < 0.001) {
                                continue;
                            }
                        }
                        ytCoordinates.emplace_back(TGeoCoord(pointData[0].AsDouble(), pointData[1].AsDouble()), TInstant::Seconds(pointData[2].AsInt64()), speed);
                    }
                    std::sort(ytCoordinates.begin(), ytCoordinates.end(), [](const NGraph::TRouter::TTimedGeoCoordinate& a, const NGraph::TRouter::TTimedGeoCoordinate& b) -> bool {
                        return a.Timestamp < b.Timestamp;
                    });
                    for (ui32 p = 0; p + 1 < ytCoordinates.size(); ++p) {
                        ::TRect<TGeoCoord> rLocal(ytCoordinates[p], ytCoordinates[p + 1]);
                        TInterval<ui32> intervalSegment(ytCoordinates[p].Timestamp.Seconds(), ytCoordinates[p + 1].Timestamp.Seconds());
                        if (rLocal.Cross(rect) && intervalTime.Intersection(intervalSegment)) {
                            if (intervalSegment.GetLength() && (ytCoordinates[p].GetLengthTo(ytCoordinates[p + 1]) / intervalSegment.GetLength()) > 250 / 3.6) {
                                continue;
                            }
                            NJson::TJsonValue json;
                            json.InsertValue(ToString(ESessionInput::SessionId), inputRow["session_id"].AsString());
                            result.emplace_back(json);
                            break;
                        }
                    }
                }
            }
            NDrive::TEventLog::Log("GenerateDocParameters", NJson::TMapBuilder
                ("status", "polyline_filtering")
                ("guid", guid)
                ("sessions_size", sessionIds.size())
                ("tracks_size", tracksCount)
                ("total_tracks_size", totalTracksCount)
                ("result_size", result.size())
            );
        } catch (const std::exception& e) {
            errors.AddMessage("YTError", FormatExc(e));
            ERROR_LOG << "YTError " << FormatExc(e) << Endl;
            return false;
        }
    } else {
        for (const auto& session : filtredRides) {
            NJson::TJsonValue json;
            json.InsertValue(ToString(ESessionInput::SessionId), session.GetSessionId());
            result.emplace_back(json);
        }
    }

    NDrive::TEventLog::Log("GenerateDocParameters", NJson::TMapBuilder
        ("status", "finish")
        ("guid", guid)
    );
    return true;
}


TDocumentDescription::TFactory::TRegistrator<TRepeatedBeforeDocument> TRepeatedBeforeDocument::Registrator("sessions_before");

bool TRepeatedBeforeDocument::GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const {
    TString sessionId;
    if (!NJson::ParseField(postData, ToString(ESessionInput::SessionId), sessionId, /* required = */ true)) {
        errors.AddMessage("ui_errors", "Не указана сессия");
        return false;
    }
    TInstant since = TInstant::Zero();
    if (!NJson::ParseField(postData, "session_start_timestamp", since, /* required = */ false)) {
        errors.AddMessage("ui_errors", "Ошибка парсинга: некорректный формат поля session_start_timestamp");
        return false;
    }
    ui32 sessionsBefore = 10;
    if (!NJson::ParseField(postData, "sessions_before", sessionsBefore, /* required = */ false)) {
        errors.AddMessage("ui_errors", "Ошибка парсинга: некорректный формат поля sessions_before");
        return false;
    }
    TObjectEvent<TMinimalCompiledRiding> ride;
    if (!NDocumentManager::GetMinimalCompiledRiding(sessionId, ride, server, errors)) {
        return false;
    }
    TVector<TMinimalCompiledRiding> localSessions;
    if (!NDocumentManager::GetCompiledRidingsByCar(ride.GetObjectId(), since ? since : ride.GetStartInstant() - TDuration::Days(5), ride.GetStartInstant(), localSessions, server, errors)) {
        return false;
    }
    std::sort(localSessions.begin(), localSessions.end(), [](const TMinimalCompiledRiding& a, const TMinimalCompiledRiding& b) -> bool {
        return a.GetStartInstant() < b.GetStartInstant();
    });
    if (!localSessions.empty() && localSessions.back().GetSessionId() == sessionId) {
        localSessions.pop_back();
    }
    for (auto it = localSessions.end() - Min(localSessions.size(), (size_t)sessionsBefore); it != localSessions.end(); it++) {
        TString sesId = it->GetSessionId();
        result.emplace_back(NJson::TMapBuilder(ToString(ESessionInput::SessionId), sesId));
    }
    return true;
}

void TRepeatedBeforeDocument::DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& /* server */) const {
    if (!scheme.HasField(ToString(ESessionInput::SessionId))) {
        scheme.Add<TFSString>(ToString(ESessionInput::SessionId), "Идентификатор сессии").SetVisual(TFSString::EVisualType::GUID).SetRequired(true);
    }
    if (!scheme.HasField("sessions_before")) {
        scheme.Add<TFSNumeric>("sessions_before", "Максимум сессий до").SetDefault(10);
    }
    if (!scheme.HasField("session_start_timestamp")) {
        scheme.Add<TFSNumeric>("session_start_timestamp", "Начало интервала сессий").SetVisual(TFSNumeric::EVisualType::DateTime);
    }
}

bool TRepeatedBeforeDocument::UseExternalParameters() const {
    return false;
}


TDocumentDescription::TFactory::TRegistrator<TRepeatedAfterDocument> TRepeatedAfterDocument::Registrator("sessions_after");

bool TRepeatedAfterDocument::GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const {
    TString sessionId;
    if (!NJson::ParseField(postData, ToString(ESessionInput::SessionId), sessionId, /* required = */ true)) {
        errors.AddMessage("ui_errors", "Не указана сессия");
        return false;
    }
    TInstant until = TInstant::Zero();
    if (!NJson::ParseField(postData, "session_finish_timestamp", until, /* required = */ false)) {
        errors.AddMessage("ui_errors", "Ошибка парсинга: некорректный формат поля session_finish_timestamp");
        return false;
    }
    ui32 sessionsAfter = 5;
    if (!NJson::ParseField(postData, "sessions_after", sessionsAfter, /* required = */ false)) {
        errors.AddMessage("ui_errors", "Ошибка парсинга: некорректный формат поля sessions_after");
        return false;
    }
    TObjectEvent<TMinimalCompiledRiding> ride;
    if (!NDocumentManager::GetMinimalCompiledRiding(sessionId, ride, server, errors)) {
        return false;
    }
    TVector<TMinimalCompiledRiding> localSessions;
    if (!NDocumentManager::GetCompiledRidingsByCar(ride.GetObjectId(), ride.GetFinishInstant(), until ? until : ride.GetFinishInstant() + TDuration::Days(5), localSessions, server, errors)) {
        return false;
    }
    std::sort(localSessions.begin(), localSessions.end(), [](const TMinimalCompiledRiding& a, const TMinimalCompiledRiding& b) -> bool {
        return a.GetStartInstant() < b.GetStartInstant();
    });
    for (const auto& ses : localSessions) {
        if (ses.GetSessionId() == sessionId) {
            continue;
        }
        if (result.size() >= sessionsAfter) {
            break;
        }
        result.emplace_back(NJson::TMapBuilder(ToString(ESessionInput::SessionId), ses.GetSessionId()));
    }
    return true;
}

void TRepeatedAfterDocument::DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& /* server */) const {
    if (!scheme.HasField(ToString(ESessionInput::SessionId))) {
        scheme.Add<TFSString>(ToString(ESessionInput::SessionId), "Идентификатор сессии").SetVisual(TFSString::EVisualType::GUID);
    }
    if (!scheme.HasField("sessions_after")) {
        scheme.Add<TFSNumeric>("sessions_after", "Максимум сессий после").SetDefault(5);
    }
    if (!scheme.HasField("session_finish_timestamp")) {
        scheme.Add<TFSNumeric>("session_finish_timestamp", "Окончание интервала сессий").SetVisual(TFSNumeric::EVisualType::DateTime);
    }
}

bool TRepeatedAfterDocument::UseExternalParameters() const {
    return false;
}
