#pragma once

#include <drive/backend/doc_packages/document.h>

#include <drive/backend/billing/interfaces/tasks.h>

#include <rtline/library/geometry/coord.h>

class TCompiledRiding;
class TCompiledBill;
class TCompiledRefund;

enum class EEmptyContentPolicy {
    Error /* "error" */,
    DefaultDocument /* "default_document" */,
};

enum class ESubDocumentErrorPolicy {
    Error /* "error" */,
    Skip /* "skip" */,
    DefaultDocument /* "default_document" */,
};

class IRepeatedDocument : public TDocumentDescription {
    R_READONLY(TString, Document);
    R_READONLY(EEmptyContentPolicy, EmptyPolicy, EEmptyContentPolicy::Error);
    R_READONLY(TString, DefaultDocument);
    R_READONLY(ESubDocumentErrorPolicy, SubDocumentErrorPolicy, ESubDocumentErrorPolicy::Error);
    R_READONLY(TString, DefaultSubDocument);
    R_READONLY(TString, SeparatingDocument);
    R_READONLY(bool, UsePostInSubDoc, false);
public:
    virtual bool DoParseMeta(const NJson::TJsonValue& meta) override;
    virtual NJson::TJsonValue DoSaveMeta() const override;

    virtual bool DoCreateContent(const NJson::TJsonValue& postData,
        const TMap<TString, ITemplateData::TPtr>& parameters,
        const NDrive::IServer& server,
        IDocumentAssembler& builder,
        TMessagesCollector& errors) const override;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual void AddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override final;

private:
    virtual bool GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
        const NDrive::IServer& server,
        TMessagesCollector& errors) const = 0;

    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const = 0;

    virtual bool UseExternalParameters() const;
};

class TRepeatedDocument : public IRepeatedDocument {
private:
    virtual bool GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
        const NDrive::IServer& server,
        TMessagesCollector& errors) const override;

    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

private:
    static TFactory::TRegistrator<TRepeatedDocument> Registrator;
};

class IPaymentsDocument : public IRepeatedDocument {
private:
    using TBase = IRepeatedDocument;
    R_READONLY(TDuration, YtNecessaryAge, TDuration::Days(365));

public:
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoParseMeta(const NJson::TJsonValue& meta) override;
    virtual NJson::TJsonValue DoSaveMeta() const override;

private:
    virtual bool GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
        const NDrive::IServer& server,
        TMessagesCollector& errors) const override;

    virtual TOptionalPayments FetchPayments(const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const = 0;

protected:
    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override;

protected:
    static TString PaymentTypesKey;
    static TString StatusKey;
    static TString IgnoreCanceledDepositKey;
};

class TSessionPaymentsDocument : public IPaymentsDocument {
private:
    virtual TOptionalPayments FetchPayments(const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const override;

private:
    static TFactory::TRegistrator<TSessionPaymentsDocument> Registrator;
};

class TFiltredPaymentsDocument : public IPaymentsDocument {
    using TBase = IPaymentsDocument;

private:
    virtual TOptionalPayments FetchPayments(const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const override;

    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override;

private:
    static TFactory::TRegistrator<TFiltredPaymentsDocument> Registrator;
};

class TFiltredRefundsDocument : public IRepeatedDocument {
private:
    virtual bool GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
        const NDrive::IServer& server,
        TMessagesCollector& errors) const override;

    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override;

private:
    static TFactory::TRegistrator<TFiltredRefundsDocument> Registrator;
};

class TFiltredBillsDocument : public IRepeatedDocument {
private:
    virtual bool GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
        const NDrive::IServer& server,
        TMessagesCollector& errors) const override;

    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override;

private:
    static TFactory::TRegistrator<TFiltredBillsDocument> Registrator;
};

class TSessionPhotosDocument : public IRepeatedDocument {
private:
    virtual bool GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
        const NDrive::IServer& server,
        TMessagesCollector& errors) const override;

    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override;

private:
    static TFactory::TRegistrator<TSessionPhotosDocument> Registrator;
};

class TFiltredSessionsDocument : public IRepeatedDocument {
public:
    class TContext {
    public:
        R_FIELD(TString, SessionId);
        R_FIELD(TString, CarId);
        R_FIELD(TString, UserId);
        R_FIELD(TSet<TString>, Models);
        R_FIELD(TInstant, Since);
        R_FIELD(TInstant, Until);
        R_FIELD(TDuration, MinDuration);
        R_FIELD(TSet<TString>, OffersFilter);
        R_FIELD(TVector<TGeoCoord>, Polyline);

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector& errors);
        static void AddScheme(NDrive::TScheme& scheme, const IServerBase& server);
        NJson::TJsonValue SerializeToJson() const;
    };

    static TString TypeName;

private:
    virtual bool GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData,
        const NDrive::IServer& server,
        TMessagesCollector& errors) const override;

    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override;

    static TFactory::TRegistrator<TFiltredSessionsDocument> Registrator1;
    static TFactory::TRegistrator<TFiltredSessionsDocument> Registrator2;
    static TFactory::TRegistrator<TFiltredSessionsDocument> Registrator3;
    static TFactory::TRegistrator<TFiltredSessionsDocument> Registrator4;
};

class TRepeatedBeforeDocument : public IRepeatedDocument {
    virtual bool GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const override;
    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override;
    virtual bool UseExternalParameters() const override;
    static TFactory::TRegistrator<TRepeatedBeforeDocument> Registrator;
};

class TRepeatedAfterDocument : public IRepeatedDocument {
    virtual bool GenerateJsonParameters(TVector<NJson::TJsonValue>& result, const NJson::TJsonValue& postData, const NDrive::IServer& server, TMessagesCollector& errors) const override;
    virtual void DoAddPostParametersToScheme(NDrive::TScheme& scheme, const IServerBase& server) const override;
    virtual bool UseExternalParameters() const override;
    static TFactory::TRegistrator<TRepeatedAfterDocument> Registrator;
};
