#include "fine_template.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/fines/manager.h>


TString TFineTemplate::Name("fine_template");

ITemplateData::TFactory::TRegistrator<TFineTemplate> TFineTemplate::Registrator(TFineTemplate::Name);


const TString& TFineTemplate::GetName() const {
    return Name;
}

TVector<TString> TFineTemplate::GetInputs() const {
    TVector<TString> result;
    for (auto&& var : GetEnumAllValues<EInput>()) {
        result.push_back(ToString(var));
    }
    return result;
}

void TFineTemplate::AddInputsToScheme(const IServerBase& /* server */, NDrive::TScheme& scheme) const {
    if (!scheme.HasField(ToString(EInput::FineId))) {
        scheme.Add<TFSString>(ToString(EInput::FineId), "Идентификатор штрафа").SetVisual(TFSString::EVisualType::GUID).SetRequired(true);
    }
}

bool TFineTemplate::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) {
    const auto& api = *Yensured(server.GetDriveAPI());
    if (!api.HasFinesManager()) {
        errors.AddMessage(__LOCATION__, "no fines manager");
        errors.AddMessage("ui_errors", "Внутренняя неконсистентность: не удалось найти manager");
        return false;
    }
    const auto& finesManager = api.GetFinesManager();
    const TString fineId = json[ToString(EInput::FineId)].GetString();
    if (!fineId) {
        errors.AddMessage(__LOCATION__, "empty fine id");
        errors.AddMessage("ui_errors", "Недостаточно данных: пустой идентификатор штрафа");
        return false;
    }

    NDrive::NFine::TAutocodeFineEntry fine;
    {
        auto tx = finesManager.BuildTx<NSQL::ReadOnly>();
        auto dbFine = finesManager.GetFineById(fineId, tx);
        if (!dbFine) {
            errors.AddMessage(__LOCATION__, "fail to fetch fine: " + tx.GetStringReport());
            errors.AddMessage("ui_errors", "Не удалось найти штраф");
            return false;
        }
        fine = std::move(*dbFine);
    }
    Storage.AddParameter(EOutput::RulingDate, NTemplateData::THRDate(fine.GetRulingDate()).ToString());
    Storage.AddParameter(EOutput::RulingNumber, fine.GetRulingNumber());
    Storage.AddParameter(EOutput::ViolationDate, NTemplateData::THRDate(fine.GetViolationTime()).ToString());
    Storage.AddParameter(EOutput::ViolationPlace, fine.GetViolationPlace());
    Storage.AddParameter(EOutput::ArticleCode, fine.GetArticleCode());
    Storage.AddParameter(EOutput::ArticleKoap, fine.GetArticleKoap());

    if (fine.GetSessionId() && !SessionTemplate.Fetch(NJson::TMapBuilder(ToString(ESessionInput::SessionId), fine.GetSessionId()), server, errors)) {
        return false;
    }
    return true;
}

void TFineTemplate::FetchOverride(const NJson::TJsonValue& json) {
    ITemplateData::FetchOverride(json);
    SessionTemplate.FetchOverride(json);
}

void TFineTemplate::Substitude(TString& inputData, const TDataEscape escape) const {
    ITemplateData::Substitude(inputData, escape);
    SessionTemplate.Substitude(inputData, escape);
}
