#include "session_templates.h"

#include "static_tools.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/offers/standart.h>
#include <drive/backend/saas/api.h>

#include <drive/library/cpp/geocoder/api/client.h>
#include <drive/library/cpp/startrek/client.h>
#include <drive/library/cpp/tex_builder/tex_builder.h>
#include <drive/library/cpp/tracks/client.h>

#include <library/cpp/json/yson/json2yson.h>
#include <library/cpp/yson/node/node_io.h>

#include <mapreduce/yt/interface/io.h>
#include <mapreduce/yt/interface/node.h>

#include <rtline/api/graph/router/router.h>

using namespace NTemplateData;

TString TSessionTrackTemplate::UseCustomStartTimestampKey = "use_custom_start_timestamp";
TString TSessionTrackTemplate::CustomStartTimestampKey = "custom_start_timestamp";
TString TSessionTrackTemplate::UseCustomFinishTimestampKey = "use_custom_finish_timestamp";
TString TSessionTrackTemplate::CustomFinishTimestampKey = "custom_finish_timestamp";
TString TSessionTrackTemplate::TrackTypesKey = "track_types";

TString TSessionTemplate::Name("session");
TString TSessionHeavyTemplate::Name("session_heavy");
TString TSessionDiskTemplate::Name("session_disk_date");
TString TSessionTrackTemplate::Name("session_track");
TString TSessionIllegalParkingTemplate::Name("illegal_parking");
TString TSessionEvacuationTemplate::Name("evacuation");

ITemplateData::TFactory::TRegistrator<TSessionTemplate> TSessionTemplate::Registrator(TSessionTemplate::Name);
ITemplateData::TFactory::TRegistrator<TSessionHeavyTemplate> TSessionHeavyTemplate::Registrator(TSessionHeavyTemplate::Name);
ITemplateData::TFactory::TRegistrator<TSessionDiskTemplate> TSessionDiskTemplate::Registrator(TSessionDiskTemplate::Name);
ITemplateData::TFactory::TRegistrator<TSessionTrackTemplate> TSessionTrackTemplate::Registrator(TSessionTrackTemplate::Name);
ITemplateData::TFactory::TRegistrator<TSessionIllegalParkingTemplate> TSessionIllegalParkingTemplate::Registrator(TSessionIllegalParkingTemplate::Name);
ITemplateData::TFactory::TRegistrator<TSessionEvacuationTemplate> TSessionEvacuationTemplate::Registrator(TSessionEvacuationTemplate::Name);

struct TSessionEvent {
public:
    R_READONLY(EObjectHistoryAction, Action, EObjectHistoryAction::Unknown);
    R_READONLY(TString, Name);
    R_READONLY(TInstant, StartTimestamp, TInstant::Zero());
    R_FIELD(TInstant, FinishTimestamp, TInstant::Zero());

public:
    TSessionEvent() = default;

    TSessionEvent(const EObjectHistoryAction action, const TString& name, const TInstant startTimestamp, const TInstant finishTimestamp)
        : Action(action)
        , Name(name)
        , StartTimestamp(startTimestamp)
        , FinishTimestamp(finishTimestamp)
    {}

    TSessionEvent(const NYT::TNode& node)
        : Name(node["name"].AsString())
        , StartTimestamp(TInstant::Seconds(node["instant"].AsUint64()))
        , FinishTimestamp(TInstant::Zero())
    {
        TryFromString(node["action"].AsString(), Action);
    }

    TSessionEvent(const TCompiledLocalEvent& event)
        : Action(event.GetHistoryAction())
        , Name(event.GetTagName())
        , StartTimestamp(event.GetInstant())
        , FinishTimestamp(TInstant::Zero())
    {}

    void ParseOldNode(const NYT::TNode& node) {
        if (node.HasKey("started_at")) {
            StartTimestamp = TInstant::Seconds((ui32)node["started_at"].AsDouble());
        } else if (node.HasKey("start_timestamp")) {
            StartTimestamp = TInstant::Seconds((ui32)node["start_timestamp"].AsInt64());
        } else {
            ythrow yexception() << "Cannot find start timestamp";
        }

        if (node.HasKey("finished_at")) {
            FinishTimestamp = TInstant::Seconds((ui32)node["finished_at"].AsDouble());
        } else if (node.HasKey("finish_timestamp")) {
            FinishTimestamp = TInstant::Seconds((ui32)node["finish_timestamp"].AsInt64());
        } else {
            ythrow yexception() << "Cannot find finish timestamp";
        }

        const TString type = node["type"].AsString();
        if (type == "carsharing_reservation" || type == "old_state_reservation") {
            Name = "old_state_reservation";
            Action = EObjectHistoryAction::SetTagPerformer;
            return;
        }
        Action = EObjectHistoryAction::TagEvolve;
        if (type == "carsharing_ride") {
            Name = "old_state_riding";
        }
        if (type == "carsharing_parking") {
            Name = "old_state_parking";
        }
        if (type == "carsharing_acceptance") {
            Name = "old_state_acceptance";
        }
        if (type == "carsharing_reservation_paid") {
            Name = "old_state_reservation";
        }
        Name = type;
    }

    template <class TContainer>
    static TVector<TSessionEvent> ConstructSessionEvents(const TContainer& events) {
        TVector<TSessionEvent> result;
        for (const auto& event : events) {
            TSessionEvent localEvent(event);
            if (localEvent.GetAction() == EObjectHistoryAction::UpdateData || localEvent.GetAction() == EObjectHistoryAction::Unknown) {
                continue;
            }
            result.emplace_back(std::move(localEvent));
        }
        for (size_t i = 0; i + 1 < result.size(); ++i) {
            result[i].SetFinishTimestamp(result[i + 1].GetStartTimestamp());
        }
        if (result.size()) {
            result.back().SetFinishTimestamp(result.back().GetStartTimestamp());
        }
        return result;
    }

    static TVector<TSessionEvent> ConstructOldSessionEvents(const TVector<NYT::TNode>& events) {
        TVector<TSessionEvent> result;
        for (const auto& event : events) {
            TSessionEvent localEvent;
            localEvent.ParseOldNode(event);
            result.emplace_back(std::move(localEvent));
        }
        return result;
    }
};

TString GetAddress(const NDrive::TSimpleLocation& location, const TDriveAPI& driveApi) {
    if (driveApi.HasGeocoderClient()) {
        TGeoCoord coord(location.Longitude, location.Latitude);
        try {
            auto decodedAsync = driveApi.GetGeocoderClient().Decode(coord);
            NDrive::TGeocoder::TResponse decoded = decodedAsync.GetValueSync();
            return decoded.Title;
        } catch (const std::exception& e) {
            ERROR_LOG << "Geocoder " << FormatExc(e) << Endl;
        }
    }
    return Sprintf("%.5f %.5f", location.Longitude, location.Latitude);
}

TSet<TString> GetFetchedGeoTags(const NDrive::TSimpleLocation& location, const TDriveAPI& driveApi) {
    TSet<TString> result;
    const auto actor = [&result](const TFullAreaInfo& area) {
        for (auto&& tag : area.GetArea().GetTags()) {
            if (tag == "global_area") {
                result.emplace(area.GetArea().GetIdentifier());
                break;
            }
        }
        return true;
    };
    driveApi.GetAreasDB()->ProcessTagsInPoint(location.GetCoord(), actor, TInstant::Zero());
    return result;
}

class TSimpleSessionInfo : public TObjectEvent<TFullCompiledRiding> {
    using TBase = TObjectEvent<TFullCompiledRiding>;
    R_READONLY(TInstant, StartTime);
    R_READONLY(TInstant, FinishTime);
    R_READONLY(TDuration, Duration, TDuration::Zero());
    R_READONLY(TString, CarId);
    R_READONLY(TString, UserId);
    R_READONLY(TString, Id);
    R_READONLY(ui64, SumPrice, 0);
    R_READONLY(ui64, Bonus, 0);
    R_READONLY(TVector<TSessionEvent>, LocalEvents);
    R_READONLY(TInstant, AcceptanceTime);
    R_OPTIONAL(double, Mileage);
    R_READONLY(bool, Closed, true);

    R_FIELD(TRange<TInstant>, TrackCutPeriod);

public:
    TSimpleSessionInfo(const TString& sessionId) {
        SetSessionId(sessionId);
    }

    TObjectEvent<TFullCompiledRiding>& MutableBase() {
        return *this;
    }

    bool InitSimple(const NDrive::IServer& server, TMessagesCollector& errors) {
        if (!NDocumentManager::GetFullCompiledRiding(TBase::GetSessionId(), *this, server, errors)) {
            return false;
        }

        {
            auto sessionBuilder = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Now());
            auto session = sessionBuilder->GetSession(TBase::GetSessionId());
            if (session && !session->GetClosed()) {
                Closed = false;
            }
        }

        StartTime = TBase::GetStartInstant();
        FinishTime = TBase::GetFinishInstant();
        Duration = TBase::GetDuration();

        CarId = TBase::TCompiledRiding::GetObjectId();
        UserId = TBase::GetHistoryUserId();
        Id = TBase::GetSessionId();

        SumPrice = TBase::GetSumPrice();
        if (TBase::HasBill()) {
            for (const auto& record : TBase::GetBillUnsafe().GetRecords()) {
                if (record.GetType() == "billing_bonus") {
                    Bonus += record.GetCost();
                }
            }
        }

        auto localEvents = TBase::GetLocalEvents();
        if (!localEvents.empty()) {
            std::sort(localEvents.begin(), localEvents.end(), [](const TCompiledLocalEvent& a, const TCompiledLocalEvent& b) -> bool {
                return a.GetHistoryEventId() < b.GetHistoryEventId();
            });
            LocalEvents = TSessionEvent::ConstructSessionEvents(localEvents);
        } else {
            if (Closed && !GetYtLocalEvents(LocalEvents, server, errors)) {
                return false;
            }
        }

        static const TSet<TString> acceptanceStates = {"old_state_acceptance", "carsharing_acceptance"};
        static const TSet<TString> ridingStates = {"old_state_riding", "carsharing_ride"};

        for (size_t i = 0; i + 1 < LocalEvents.size(); ++i) {
            if (acceptanceStates.contains(LocalEvents[i].GetName()) && ridingStates.contains(LocalEvents[i + 1].GetName())) {
                AcceptanceTime = LocalEvents[i].GetFinishTimestamp();
                break;
            }
        }

        if (TBase::HasSnapshotsDiff() && TBase::GetSnapshotsDiffUnsafe().HasMileage()) {
            Mileage = TBase::GetSnapshotsDiffUnsafe().GetMileageUnsafe();
        }

        return true;
    }

    bool Init(const NDrive::IServer& server, TMessagesCollector& errors) {
        if (!InitSimple(server, errors)) {
            return false;
        }
        return true;
    }

private:
    bool GetYtLocalEvents(TVector<TSessionEvent>& events, const NDrive::IServer& server, TMessagesCollector& errors) {
        if (!server.GetDocumentsManager()) {
            errors.AddMessage("ui_errors", "Внутренняя неконсистентность: не удалось найти manager");
            return false;
        }
        try {
            auto reader = server.GetDocumentsManager()->GetPaymentsTableReaderBySession(GetSessionId());
            for (; reader->IsValid(); reader->Next()) {
                NYT::TNode inputRow = reader->GetRow();
                if (inputRow.HasKey("events_details") && inputRow["events_details"].IsList()) {
                    events = TSessionEvent::ConstructSessionEvents(inputRow["events_details"].AsList());
                    return true;
                }
                break;
            }
        } catch (const std::exception& e) {
            errors.AddMessage("YTError", FormatExc(e));
            ERROR_LOG << "YTError " << FormatExc(e) << Endl;
            return false;
        }

        try {
            auto reader = server.GetDocumentsManager()->GetOrderTableReader(GetFinishInstant());
            for (; reader->IsValid(); reader->Next()) {
                NYT::TNode inputRow = reader->GetRow();
                if (!inputRow.HasKey("id") || inputRow["id"].AsString() != GetSessionId()) {
                    continue;
                }
                if (inputRow.HasKey("items") && inputRow["items"].IsList()) {
                    events = TSessionEvent::ConstructOldSessionEvents(inputRow["items"].AsList());
                    if (events.size()) {
                        events.emplace_back(EObjectHistoryAction::TagEvolve, "old_state_reservation", events.back().GetFinishTimestamp(), events.back().GetFinishTimestamp());
                        events.emplace_back(EObjectHistoryAction::DropTagPerformer, "old_state_reservation", events.back().GetFinishTimestamp(), events.back().GetFinishTimestamp());
                    }
                    return true;
                }
                break;
            }
        } catch (const std::exception& e) {
            errors.AddMessage("YTError", FormatExc(e));
            ERROR_LOG << "YTError " << FormatExc(e) << Endl;
            return false;
        }
        try {
            auto reader = server.GetDocumentsManager()->GetOrdersByTimestamp(GetFinishInstant());
            for (; reader->IsValid(); reader->Next()) {
                NYT::TNode inputRow = reader->GetRow();
                if (!inputRow.HasKey("session_id") || inputRow["session_id"].AsString() != GetSessionId()) {
                    continue;
                }
                if (inputRow.HasKey("steps") && inputRow["steps"].IsList()) {
                    events = TSessionEvent::ConstructOldSessionEvents(inputRow["steps"].AsList());
                    if (events.size()) {
                        events.emplace_back(EObjectHistoryAction::TagEvolve, "old_state_reservation", events.back().GetFinishTimestamp(), events.back().GetFinishTimestamp());
                        events.emplace_back(EObjectHistoryAction::DropTagPerformer, "old_state_reservation", events.back().GetFinishTimestamp(), events.back().GetFinishTimestamp());
                    }
                    return true;
                }
                break;
            }
        } catch (const std::exception& e) {
            errors.AddMessage("YTError", FormatExc(e));
            ERROR_LOG << "YTError " << FormatExc(e) << Endl;
            return false;
        }
        if (GetStartInstant().Seconds() > 1535760000) {
            errors.AddMessage("ui_errors", "События для сессии с идентификатором " + GetSessionId() + " не найдены");
            return false;
        }
        return true;
    }
};

TExpected<ui32, TString> GetDebt(const TString& sessionId, const NDrive::IServer& server) {
    if (!server.GetDriveAPI()->HasBillingManager()) {
        return MakeUnexpected<TString>("undefined billing manager");
    }
    auto session = server.GetDriveAPI()->GetBillingManager().BuildSession(true);
    auto restored = server.GetDriveAPI()->GetBillingManager().GetActiveTasksManager().GetTask(sessionId, session);
    if (!restored) {
        return MakeUnexpected<TString>("cannot GetBillingTask: " + session.GetStringReport());
    }
    if (!(*restored)) {
        return 0;
    }

    TCachedPayments snapshot;
    if (!server.GetDriveAPI()->GetBillingManager().GetPaymentsManager().GetPayments(snapshot, sessionId, session)) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }
    auto payment = TPaymentsData::BuildPaymentsData(std::move(*restored), std::move(snapshot));
    if (!payment) {
        return MakeUnexpected(payment.GetError());
    }
    return payment->GetDebt();
}

bool TSessionTemplate::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) {
    TString sessionId = json[ToString(EInput::SessionId)].GetString();

    TSimpleSessionInfo sessionInfo(sessionId);
    if (!sessionInfo.Init(server, errors)) {
        return false;
    }

    Storage.AddParameter(EOutput::Id, sessionInfo.GetId());
    Storage.AddParameter(EOutput::StartTime, THRTime(sessionInfo.GetStartTime()).ToString());
    Storage.AddParameter(EOutput::StartTimestamp, ::ToString(sessionInfo.GetStartTime().Seconds()));
    if (sessionInfo.GetFinishTime()) {
        Storage.AddParameter(EOutput::FinishTime, THRTime(sessionInfo.GetFinishTime()).ToString());
        Storage.AddParameter(EOutput::SimpleHrFinishTime, sessionInfo.GetClosed()
            ? THRTime(sessionInfo.GetFinishTime()).ToString()
            : server.GetSettings().GetValueDef<TString>("document_manager.session_template.simple_hr_finish_time", "настоящее время"));
        Storage.AddParameter(EOutput::FinishHrTime, sessionInfo.GetClosed() ? THRTime(sessionInfo.GetFinishTime()).ToString() : "По наст. время");
        Storage.AddParameter(EOutput::FinishTimestamp, ::ToString(sessionInfo.GetFinishTime().Seconds()));
    }
    Storage.AddParameter(EOutput::Price, server.GetLocalization()->FormatPrice(ELocalization::Rus, sessionInfo.GetSumPrice(), { "руб" } , " "));
    Storage.AddParameter(EOutput::PriceRaw, server.GetLocalization()->FormatPrice(ELocalization::Rus, sessionInfo.GetSumPrice()));
    Storage.AddParameter(EOutput::TotalDuration, GetDurationString(sessionInfo.GetDuration()));
    Storage.AddParameter(EOutput::Bonus, server.GetLocalization()->FormatBonusRubles(ELocalization::Rus, sessionInfo.GetBonus() / 100, true));
    Storage.AddParameter(EOutput::BonusRaw, server.GetLocalization()->FormatPrice(ELocalization::Rus, sessionInfo.GetBonus()));

    auto sessionDebt = GetDebt(sessionId, server);
    if (!sessionDebt) {
        errors.AddMessage("ui_message", "Не удалось определить долг");
        return false;
    }
    Storage.AddParameter(EOutput::SessionDeptRaw, server.GetLocalization()->FormatPrice(ELocalization::Rus, *sessionDebt));
    Storage.AddParameter(EOutput::SessionDept, server.GetLocalization()->FormatPrice(ELocalization::Rus, *sessionDebt, { "руб" }, " "));

    if (sessionInfo.HasMileage()) {
        Storage.AddParameter(EOutput::MileageRaw, Sprintf("%0.3f", sessionInfo.GetMileageUnsafe()));
        Storage.AddParameter(EOutput::Mileage, server.GetLocalization()->DistanceFormatKm(ELocalization::Rus, sessionInfo.GetMileageUnsafe()));
    }

    if (sessionInfo.GetAcceptanceTime()) {
        Storage.AddParameter(EOutput::AcceptanceTime, THRTime(sessionInfo.GetAcceptanceTime()).ToString());
        Storage.AddParameter(EOutput::AcceptanceTimestamp, ::ToString(sessionInfo.GetAcceptanceTime().Seconds()));
        Storage.AddParameter(EOutput::AcceptanceTimeDate, THRDate(sessionInfo.GetAcceptanceTime()).ToString());
        Storage.AddParameter(EOutput::AcceptanceTimeMinutes, sessionInfo.GetAcceptanceTime().FormatLocalTime("%R"));
    }
    Storage.AddParameter(EOutput::OfferName, sessionInfo.GetOfferName());

    if (sessionInfo.HasSnapshotsDiff()) {
        if (sessionInfo.GetSnapshotsDiffUnsafe().OptionalStart().Defined()) {
            auto point = sessionInfo.GetSnapshotsDiffUnsafe().GetStartUnsafe();
            Storage.AddParameter(EOutput::StartPoint, Sprintf("%.5f %.5f", point.Longitude, point.Latitude));
            Storage.AddParameter(EOutput::StartAddress, GetAddress(point, *server.GetDriveAPI()));
            Storage.AddParameter(EOutput::StartGeoTags, JoinSeq(",", GetFetchedGeoTags(point, *server.GetDriveAPI())));
        }
        if (sessionInfo.GetSnapshotsDiffUnsafe().OptionalLast().Defined()) {
            auto point = sessionInfo.GetSnapshotsDiffUnsafe().GetLastUnsafe();
            Storage.AddParameter(EOutput::LastPoint, Sprintf("%.5f %.5f", point.Longitude, point.Latitude));
            Storage.AddParameter(EOutput::LastAddress, GetAddress(point, *server.GetDriveAPI()));
            Storage.AddParameter(EOutput::LastGeoTags, JoinSeq(",", GetFetchedGeoTags(point, *server.GetDriveAPI())));
        }
    }
    Storage.AddParameter(EOutput::Closed, sessionInfo.GetClosed() ? "+": "-");

    {
        TStringBuilder eventTexTableBuilder;
        eventTexTableBuilder << "\\setlength\\LTleft{0pt}" << Endl;
        eventTexTableBuilder << "\\setlength\\LTright{0pt}" << Endl;
        eventTexTableBuilder << "\\begin{longtable}{@{\\extracolsep{\\fill}}|l|r|r|}" << Endl;
        eventTexTableBuilder << "\\hline" << Endl;

        eventTexTableBuilder << " \\textbf{Операция} & \\textbf{Начало} & \\textbf{Конец} \\\\ \\hline" << Endl;
        for (const auto& event : sessionInfo.GetLocalEvents()) {
            eventTexTableBuilder << NTexBuilder::TTextStyle::Quote(::ToString(event.GetAction())) << " (" << NTexBuilder::TTextStyle::Quote(event.GetName()) << ")" << " & " << THRTime(event.GetStartTimestamp()).ToString() << " & " << THRTime(event.GetFinishTimestamp()).ToString() << " \\\\" << Endl;
            eventTexTableBuilder << "\\hline" << Endl;
        }
        eventTexTableBuilder << "\\end{longtable}" << Endl;

        Storage.AddParameter(EOutput::EventsTexTable, eventTexTableBuilder);
    }

    const TString ridingResultMsg = sessionInfo.GetSumPrice() == 0 || (sessionInfo.HasMileage() && sessionInfo.GetMileageUnsafe() < 0.01)
        ? server.GetSettings().GetValueDef<TString>("document_manager.session_template.riding_canceled", "Поездка не состоялась")
        : server.GetSettings().GetValueDef<TString>("document_manager.session_template.riding_completed", "");
    Storage.AddParameter(EOutput::RidingResult, ridingResultMsg);

    if (sessionInfo.GetOffer()) {
        auto locale = DefaultLocale;
        Storage.AddParameter(EOutput::OfferFullDetails, sessionInfo.GetOffer()->BuildCalculationDescription(locale, sessionInfo, server));
        {
            auto standartOffer = dynamic_cast<const TStandartOffer*>(sessionInfo.GetOffer().Get());
            Storage.AddParameter(EOutput::Agreement, (standartOffer && standartOffer->GetAgreement()) ? standartOffer->GetAgreement() : DefaultAgreementId);
        }
        if (sessionInfo.GetOffer()->GetSelectedCharge()) {
            auto accountDescription = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetDescriptionByName(sessionInfo.GetOffer()->GetSelectedCharge());
            if (accountDescription.Defined() && accountDescription->GetParentId()) {
                auto parent = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetAccountById(accountDescription->GetParentId());
                if (parent && parent->GetRecord()) {
                    Storage.AddParameter(EOutput::CompanyName, parent->GetRecord()->GetCompany());
                }
            }
        }
    } else {
        Storage.AddParameter(EOutput::Agreement, DefaultAgreementId);
    }

    if (sessionInfo.GetCarId() && !CarTemplate.Fetch(sessionInfo.GetCarId(), sessionInfo.GetStartTime(), server, errors)) {
        return false;
    }

    auto userPhotoParam = json;
    userPhotoParam.InsertValue(ToString(EUserInput::UserId), sessionInfo.GetUserId());
    return UserTemplate.Fetch(sessionInfo.GetUserId(), server, errors)
        && UserPhotoTemplate.Fetch(userPhotoParam, server, errors);
}

NGraph::TRouter::TTimedGeoCoordinates FilterNearestPoints(const NGraph::TRouter::TTimedGeoCoordinate& center, double r, const NGraph::TRouter::TTimedGeoCoordinates& coordinates) {
    const TDuration rDuration = TDuration::Minutes(10);
    const ui32 rCount = 100;
    auto itBegin = std::lower_bound(coordinates.begin(), coordinates.end(), center.Timestamp, [](const NGraph::TRouter::TTimedGeoCoordinate& l, const TInstant timestamp) {
        return l.Timestamp < timestamp;
    });
    auto itEnd = itBegin;

    for (ui32 count = 0; itEnd != coordinates.end(); ++itEnd, ++count) {
        if (count >= rCount && (itEnd->Timestamp - center.Timestamp) > rDuration && itEnd->GetLengthTo(center) > r) {
            break;
        }
    }

    if (itBegin == coordinates.end()) {
        auto begin = coordinates.rbegin();
        for (ui32 count = 0; begin != coordinates.rend(); ++begin, ++count) {
            if (count >= rCount && (center.Timestamp - begin->Timestamp) > rDuration && begin->GetLengthTo(center) > r) {
                break;
            }
        }
    } else {
        for (ui32 count = 0; itBegin != coordinates.begin(); --itBegin, ++count) {
            if (count >= rCount && (center.Timestamp - itBegin->Timestamp) > rDuration && itBegin->GetLengthTo(center) > r) {
                break;
            }
        }
    }

    return {itBegin, itEnd};
}

TString GetTrackString(const NGraph::TRouter::TTimedGeoCoordinates& coordinates) {
    NGraph::TRouter::TTimedGeoCoordinates filtredCordinates;
    size_t step = coordinates.size() / 99 + 1;
    for (size_t i = 0; i < coordinates.size(); i += step) {
        filtredCordinates.push_back(coordinates[i]);
    }
    if (!coordinates.empty()) {
        filtredCordinates.push_back(coordinates.back());
    }

    TStringBuilder coordinatesStr;
    for (size_t i = 0; i < filtredCordinates.size(); ++i) {
        coordinatesStr << filtredCordinates[i].X << "," << filtredCordinates[i].Y;
        if (i + 1 < filtredCordinates.size()) {
            coordinatesStr << ",";
        }
    }

    return coordinatesStr;
}

TString GetCoordinateString(const NGraph::TRouter::TTimedGeoCoordinate& coordinate) {
    return ToString(coordinate.X) + "," + ToString(coordinate.Y);
}

NGraph::TRouter::TTimedGeoCoordinates CutStart(TInstant ts, const NGraph::TRouter::TTimedGeoCoordinates& coordinates) {
    auto finishIter = std::lower_bound(coordinates.begin(), coordinates.end(), ts, [](const NGraph::TRouter::TTimedGeoCoordinate& a, TInstant timestamp) -> bool {
        return a.Timestamp < timestamp;
    });
    if (finishIter == coordinates.end()) {
        return {};
    }
    NGraph::TRouter::TTimedGeoCoordinates result(finishIter, coordinates.end());
    return result;
}

NGraph::TRouter::TTimedGeoCoordinates CutFinish(TInstant finishTimestamp, const NGraph::TRouter::TTimedGeoCoordinates& coordinates) {
    NGraph::TRouter::TTimedGeoCoordinates result = coordinates;
    auto finishIter = std::lower_bound(result.begin(), result.end(), finishTimestamp, [](const NGraph::TRouter::TTimedGeoCoordinate& a, TInstant timestamp) -> bool {
        return a.Timestamp < timestamp;
    });

    if (finishIter != result.end()) {
        result.erase(finishIter, result.end());
    }
    return result;
}

bool GetFullSessionTrack(const TSimpleSessionInfo& sessionInfo, const TSet<NDrive::ECarStatus>& trackTypes, NGraph::TRouter::TTimedGeoCoordinates& coordinates, const NDrive::IServer& server, TMessagesCollector& errors) {
    const auto& sessionId = sessionInfo.GetSessionId();
    NGraph::TRouter::TTimedGeoCoordinates allCoordinates;
    if (!sessionInfo.IsClosed() || Max(sessionInfo.GetStartTime(), sessionInfo.GetTrackCutPeriod().From.GetOrElse(TInstant::Zero())) + server.GetDocumentsManager()->GetConfig().GetTracksApiLifetime() > ModelingNow()) {
        const TString& tracksService = server.GetDocumentsManager()->GetConfig().GetTracksService();
        auto tracksApi = server.GetRTLineAPI(tracksService);
        if (!tracksApi) {
            errors.AddMessage("ui_errors", "Внутренняя ошибка: cannot find Tracks API " + tracksService);
            return false;
        }
        auto timeout = server.GetDocumentsManager()->GetConfig().GetTracksApiTimeout();

        NDrive::TTracksClient tracksClient(tracksApi->GetSearchClient());
        NDrive::TTrackQuery trackQuery;
        trackQuery.SessionId = sessionId;
        const auto& TrackCutPeriod = sessionInfo.GetTrackCutPeriod();
        trackQuery.Since = TrackCutPeriod.From.GetOrElse(trackQuery.Since);
        trackQuery.Until = TrackCutPeriod.To.GetOrElse(trackQuery.Until);
        auto asyncTracks = tracksClient.GetTracks(trackQuery, timeout);
        if (!asyncTracks.Wait(timeout) || !asyncTracks.HasValue()) {
            errors.AddMessage("errors", "TracksClient error: " + NThreading::GetExceptionMessage(asyncTracks));
            errors.AddMessage("ui_errors", "Трек для сессии с идентификатором " + sessionId + " не найден");
            return false;
        }

        const auto& tracks = asyncTracks.GetValue();
        for (const auto& track : tracks) {
            if (trackTypes.contains(track.Status)) {
                allCoordinates.insert(allCoordinates.end(), track.Coordinates.begin(), track.Coordinates.end());
            }
        }
    } else {
        bool hasRoute = false;
        try {
            auto reader = server.GetDocumentsManager()->GetTracksTableReaderByTimestamp(sessionInfo.GetFinishTime());
            NGraph::TRouter::TTimedGeoCoordinates ytCoordinates;
            for (; reader->IsValid(); reader->Next()) {
                NYT::TNode inputRow = reader->GetRow();
                TMaybe<TGeoCoord> previousCoord;
                const ui32 jumpLimit = server.GetSettings().GetValueDef<ui32>("document_manager.session_track.jump_limit", 1000);
                if (inputRow["session_id"].AsString() == sessionId) {
                    for (const auto& pointNode : inputRow["raw_route"].AsList()) {
                        const auto& pointData = pointNode.AsList();
                        if (pointData.size() < 3) {
                            errors.AddMessage("ui_errors", "Некорректные даные трека для сессии " + sessionId);
                            return false;
                        }
                        TSpeed speed = 1;
                        if (pointData.size() > 3) {
                            speed = pointData[3].AsDouble();
                            if (speed < 0.001
                                && (jumpLimit == 0 || previousCoord && previousCoord->GetLengthTo(TGeoCoord(pointData[0].AsDouble(), pointData[1].AsDouble())) < jumpLimit))
                            {
                                continue;
                            }
                        }
                        previousCoord = TGeoCoord(pointData[0].AsDouble(), pointData[1].AsDouble());
                        ytCoordinates.emplace_back(*previousCoord, TInstant::Seconds(pointData[2].AsInt64()), speed);
                    }
                    std::sort(ytCoordinates.begin(), ytCoordinates.end(), [](const NGraph::TRouter::TTimedGeoCoordinate& a, const NGraph::TRouter::TTimedGeoCoordinate& b) -> bool {
                        return a.Timestamp < b.Timestamp;
                    });
                    auto itCoord = ytCoordinates.begin();
                    for (const auto& state : sessionInfo.GetLocalEvents()) {
                        while (itCoord != ytCoordinates.end()) {
                            if (itCoord->Timestamp < state.GetStartTimestamp()) {
                                ++itCoord;
                                continue;
                            }
                            if (itCoord->Timestamp > state.GetFinishTimestamp()) {
                                break;
                            }
                            if (state.GetName() == "old_state_reservation" && trackTypes.contains(NDrive::ECarStatus::csReservation) ||
                                state.GetName() == "old_state_acceptance" && trackTypes.contains(NDrive::ECarStatus::csAcceptance) ||
                                state.GetName() == "old_state_riding" && trackTypes.contains(NDrive::ECarStatus::csRide) ||
                                state.GetName() == "old_state_parking" && trackTypes.contains(NDrive::ECarStatus::csParking))
                            {
                                allCoordinates.emplace_back(*itCoord);
                            }
                            ++itCoord;
                        }
                    }
                    if (itCoord != ytCoordinates.end() && trackTypes.contains(NDrive::ECarStatus::csPost)) {
                        allCoordinates.insert(allCoordinates.end(), itCoord, ytCoordinates.end());
                    }
                    hasRoute = true;
                    break;
                }
            }
        } catch (const std::exception& e) {
            errors.AddMessage("YTError", FormatExc(e));
            ERROR_LOG << "YTError " << FormatExc(e) << Endl;
            return false;
        }
        if (!hasRoute) {
            errors.AddMessage("ui_errors", "Трек для сессии с идентификатором " + sessionId + " не найден");
            return false;
        }
    }

    std::sort(allCoordinates.begin(), allCoordinates.end(), [](const NGraph::TRouter::TTimedGeoCoordinate& a, const NGraph::TRouter::TTimedGeoCoordinate& b) -> bool {
        return a.Timestamp < b.Timestamp;
    });

    for (const auto& coordinate : allCoordinates) {
        if (std::abs(coordinate.Speed) < 0.001) {
            continue;
        }
        if (!coordinates.empty() && fabs(coordinates.back().GetLengthTo(coordinate)) < server.GetDocumentsManager()->GetConfig().GetPointMinDistance()) {
            continue;
        }
        coordinates.emplace_back(coordinate);
    }
    if (coordinates.empty() && !allCoordinates.empty()) {
        coordinates.emplace_back(allCoordinates.front());
    }
    return true;
}

bool GetFullSessionTrack(const TString& sessionId, const TRange<TInstant> cut, const TSet<NDrive::ECarStatus>& trackTypes, NGraph::TRouter::TTimedGeoCoordinates& coordinates, const NDrive::IServer& server, TMessagesCollector& errors) {
    TSimpleSessionInfo sessionInfo(sessionId);
    if (!sessionInfo.InitSimple(server, errors)) {
        return false;
    }
    sessionInfo.SetTrackCutPeriod(cut);
    return GetFullSessionTrack(sessionInfo, trackTypes, coordinates, server, errors);
}

bool TSessionTrackTemplate::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) {
    TString sessionId;
    if (!NJson::ParseField(json, ToString(EInput::SessionId), sessionId, true, errors)) {
        return false;
    }

    TSet<NDrive::ECarStatus> trackTypes = { NDrive::ECarStatus::csAcceptance, NDrive::ECarStatus::csParking, NDrive::ECarStatus::csReservation, NDrive::ECarStatus::csRide };
    if (json.Has(TrackTypesKey) && json[TrackTypesKey].IsArray() && json[TrackTypesKey].GetArray().size()) {
        trackTypes.clear();
        for (const auto& jsonType : json[TrackTypesKey].GetArraySafe()) {
            NDrive::ECarStatus type;
            if (TryFromString(jsonType.GetStringRobust(), type)) {
                trackTypes.emplace(type);
            }
        }
    }

    TRange<TInstant> cut;
    if (json[UseCustomStartTimestampKey].GetBoolean()) {
        TInstant customTimestamp = TInstant::Seconds(json[CustomStartTimestampKey].GetUInteger());
        if (customTimestamp != TInstant::Zero()) {
            cut.From = customTimestamp;
        }
    }
    if (json[UseCustomFinishTimestampKey].GetBoolean()) {
        TInstant customTimestamp = TInstant::Seconds(json[CustomFinishTimestampKey].GetUInteger());
        if (customTimestamp != TInstant::Zero()) {
            cut.To = customTimestamp;
        }
    }

    NGraph::TRouter::TTimedGeoCoordinates coordinates;
    if (!GetFullSessionTrack(sessionId, cut, trackTypes, coordinates, server, errors)) {
        return false;
    }

    if (cut.From) {
        coordinates = CutStart(*cut.From, coordinates);
    }
    if (cut.To) {
        coordinates = CutFinish(*cut.To, coordinates);
    }

    if (coordinates.empty()) {
        errors.AddMessage("session_tracks", "empty track");
        return false;
    }

    Storage.AddParameter(EOutput::StartPoint, GetCoordinateString(coordinates.front()));
    Storage.AddParameter(EOutput::StartTrack, GetTrackString(FilterNearestPoints(coordinates.front(), 1000, coordinates)));
    Storage.AddParameter(EOutput::EndPoint, GetCoordinateString(coordinates.back()));
    Storage.AddParameter(EOutput::EndTrack, GetTrackString(FilterNearestPoints(coordinates.back(), 1000, coordinates)));

    Storage.AddParameter(EOutput::Track, GetTrackString(coordinates));

    {
        Storage.AddParameter(EOutput::IllegalParkingPoint, Storage.GetParameter(EOutput::EndPoint));
        Storage.AddParameter(EOutput::IllegalParkingTrackPoint, Storage.GetParameter(EOutput::EndTrack));
        if (json[TSessionIllegalParkingTemplate::UseViolationTimeKey].GetBoolean()) {
            TSessionIllegalParkingTemplate illegalParkingTemplate;
            if (!illegalParkingTemplate.Fetch(json, server, errors)) {
                return false;
            }
            auto illegalParkingCoordinates = CutFinish(illegalParkingTemplate.GetViolationTime(), coordinates);
            if (illegalParkingCoordinates.empty()) {
                errors.AddMessage("illegal parking session_tracks", "empty track");
                return false;
            }
            Storage.AddParameter(EOutput::IllegalParkingPoint, GetCoordinateString(illegalParkingCoordinates.back()));
            Storage.AddParameter(EOutput::IllegalParkingTrackPoint, GetTrackString(FilterNearestPoints(illegalParkingCoordinates.back(), 1000, coordinates)));
        }
    }

    {
        Storage.AddParameter(EOutput::EvacuationPoint, Storage.GetParameter(EOutput::EndPoint));
        Storage.AddParameter(EOutput::EvacuationTrack, Storage.GetParameter(EOutput::Track));
        Storage.AddParameter(EOutput::EvacuationTrackPoint, Storage.GetParameter(EOutput::EndTrack));
        if (json[TSessionEvacuationTemplate::UseViolationTimeKey].GetBoolean()) {
            TSessionEvacuationTemplate evacuationTemplate;
            if (!evacuationTemplate.Fetch(json, server, errors)) {
                return false;
            }
            auto evacuationCoordinates = CutFinish(evacuationTemplate.GetViolationTime(), coordinates);
            if (evacuationCoordinates.empty()) {
                errors.AddMessage("evacuation session_tracks", "empty track");
                return false;
            }
            Storage.AddParameter(EOutput::EvacuationPoint, GetCoordinateString(evacuationCoordinates.back()));
            Storage.AddParameter(EOutput::EvacuationTrackPoint, GetTrackString(FilterNearestPoints(evacuationCoordinates.back(), 1000, evacuationCoordinates)));
            Storage.AddParameter(EOutput::EvacuationTrack, GetTrackString(evacuationCoordinates));
        }
    }
    return true;
}

void TSessionTemplate::Substitude(TString& inputData, const TDataEscape escape) const {
    ITemplateData::Substitude(inputData, escape);
    CarTemplate.Substitude(inputData, escape);
    UserTemplate.Substitude(inputData, escape);
    UserPhotoTemplate.Substitude(inputData, escape);
}

void TSessionTemplate::FetchOverride(const NJson::TJsonValue& json) {
    ITemplateData::FetchOverride(json);
    CarTemplate.FetchOverride(json);
    UserTemplate.FetchOverride(json);
    UserPhotoTemplate.FetchOverride(json);
}

void TSessionTemplate::AddInputsToScheme(const IServerBase& server, NDrive::TScheme& scheme) const {
    TBase::AddInputsToScheme(server, scheme);
    UserPhotoTemplate.AddInputsToScheme(server, scheme);
}

TString ReplaceName(const TString& name) {
    const TString badName = "Потратить на поездку";
    if (name == badName) {
        return "Бонусные баллы Плюса";
    }
    return name;
}

void TSessionHeavyTemplate::AddInputsToScheme(const IServerBase& server, NDrive::TScheme& scheme) const {
    TBase::AddInputsToScheme(server, scheme);
    if (!scheme.HasField("need_device_info")) {
        scheme.Add<TFSBoolean>("need_device_info", "Выполнить поиск данных о девайсе").SetDefault(false);
    }
}

bool TSessionHeavyTemplate::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) {
    TString sessionId = json[ToString(EInput::SessionId)].GetString();

    if (!server.GetDriveAPI()->HasBillingManager()) {
        errors.AddMessage("ui_errors", "Внутренняя ошибка: отсутствует менеджер");
        return false;
    }
    TSimpleSessionInfo sessionInfo(sessionId);
    if (!sessionInfo.Init(server, errors)) {
        return false;
    }

    TString deviceId;
    TString userAgent;
    if (json["need_device_info"].GetBoolean()
        && !NDocumentManager::GetDeviceInfo(sessionId, deviceId, userAgent, server, errors))
    {
        return false;
    }
    Storage.AddParameter(EOutput::DeviceId, deviceId);
    Storage.AddParameter(EOutput::UserAgent, userAgent);

    if (sessionInfo.GetSumPrice() == 0) {
        return DiskTemplate.Fetch(sessionInfo.GetStartTime(), server, errors);
    }
    {
        auto session = server.GetDriveAPI()->GetBillingManager().BuildSession(true);
        auto compiledBill = server.GetDriveAPI()->GetBillingManager().GetCompiledBills().GetFullBillFromDB(sessionId, session);
        if (!compiledBill) {
            errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить сессию из compiled_bills:" + sessionId + " " + session.GetStringReport());
            return false;
        }

        if (compiledBill->GetFinal()) {
            TStringBuilder walletTexTableBuilder;
            walletTexTableBuilder << "\\begin{tabular}{ll}" << Endl;
            for (const auto& wallet : compiledBill->GetDetails().GetItems()) {
                if (wallet.GetType() == NDrive::NBilling::EAccount::Trust) {
                    continue;
                }
                walletTexTableBuilder << NTexBuilder::TTextStyle::Quote(ReplaceName(wallet.GetName())) << " & " << wallet.GetSum() / 100. << " rub \\\\" << Endl;
            }
            walletTexTableBuilder << "\\end{tabular} \\\\" << Endl;
            Storage.AddParameter(EOutput::WalletsTexTable, walletTexTableBuilder);
            return DiskTemplate.Fetch(sessionInfo.GetStartTime(), server, errors);
        } else if (!compiledBill->GetDetails().GetItems().empty()) {
            errors.AddMessage("ui_errors", "Сессия еще не завершена");
            return false;
        }
    }

    try {
        auto reader = server.GetDocumentsManager()->GetPaymentsTableReaderBySession(sessionId);
        for (; reader->IsValid(); reader->Next()) {
            NYT::TNode inputRow = reader->GetRow();
            if (inputRow["session_id"].AsString() == sessionId) {
                TInstant startTime = TInstant::Seconds(inputRow["start_ts"].AsUint64());

                if (inputRow.HasKey("payment_details") && inputRow["payment_details"].IsMap()) {
                    auto wallets = inputRow["payment_details"].AsMap()["wallets"].AsList();
                    if (!wallets.empty()) {
                        TStringBuilder walletTexTableBuilder;
                        walletTexTableBuilder << "\\begin{tabular}{ll}" << Endl;
                        for (const auto& wallet : wallets) {
                            if (!wallet.HasKey("name") || wallet["name"].AsString() == "За свои") {
                                continue;
                            }
                            walletTexTableBuilder << NTexBuilder::TTextStyle::Quote(ReplaceName(wallet["name"].AsString())) << " & " << wallet["sum"].AsUint64() / 100. << " rub \\\\" << Endl;
                        }
                        walletTexTableBuilder << "\\end{tabular} \\\\" << Endl;
                        Storage.AddParameter(EOutput::WalletsTexTable, walletTexTableBuilder);
                    }
                }
                return DiskTemplate.Fetch(startTime, server, errors);
            }
        }
    } catch (const std::exception& e) {
        errors.AddMessage("YTError", FormatExc(e));
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        return false;
    }
    errors.AddMessage("ui_errors", "Счет для сессии с идентификатором " + sessionId + " не найден");
    return false;
}

void TSessionHeavyTemplate::Substitude(TString& inputData, const TDataEscape escape) const {
    ITemplateData::Substitude(inputData, escape);
    DiskTemplate.Substitude(inputData, escape);
}

void TSessionHeavyTemplate::FetchOverride(const NJson::TJsonValue& json) {
    ITemplateData::FetchOverride(json);
    DiskTemplate.FetchOverride(json);
}

bool TSessionDiskTemplate::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) {
    const TString sessionId = json[ToString(EInput::SessionId)].GetString();
    TSimpleSessionInfo sessionInfo(sessionId);
    if (!sessionInfo.Init(server, errors)) {
        return false;
    }
    return DiskTemplate.Fetch(sessionInfo.GetStartTime(), server, errors);
}

void TSessionDiskTemplate::Substitude(TString& inputData, const TDataEscape escape) const {
    ITemplateData::Substitude(inputData, escape);
    DiskTemplate.Substitude(inputData, escape);
}

void TSessionDiskTemplate::FetchOverride(const NJson::TJsonValue& json) {
    ITemplateData::FetchOverride(json);
    DiskTemplate.FetchOverride(json);
}

template <class EOutput>
bool ISessionInfoFromStartrekTemplate<EOutput>::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) {
    if (!server.GetDriveAPI()->HasStartrekClient()) {
        errors.AddMessage(__LOCATION__, "startrek client undefined");
        return false;
    }

    TString sessionId = json[ToString(TBase::EInput::SessionId)].GetString();

    TSimpleSessionInfo sessionInfo(sessionId);
    if (!sessionInfo.InitSimple(server, errors)) {
        return false;
    }
    TString userId = sessionInfo.GetUserId();

    if (!userId) {
        errors.AddMessage("fetch_session", "session not found");
        return false;
    }
    auto user = server.GetDriveAPI()->GetUsersData()->GetCachedObject(userId);
    if (!user) {
        errors.AddMessage("fetch_user", "user " + userId + " not found");
        return false;
    }

    const auto& userTagManager = server.GetDriveAPI()->GetTagsManager().GetUserTags();
    auto tx = userTagManager.BuildSession(true);
    auto optionalTaggedUser = userTagManager.RestoreObject(userId, tx);
    if (!optionalTaggedUser) {
        errors.MergeMessages(tx.GetMessages());
        return false;
    }

    bool found = false;
    for (const auto& tag : optionalTaggedUser->GetTags()) {
        const TUserProblemTag* userProblemTag = tag.GetTagAs<TUserProblemTag>();
        if (!userProblemTag) {
            continue;
        }
        if (userProblemTag->GetSessionId() != sessionInfo.GetId()) {
            continue;
        }
        auto issues = userProblemTag->GetStartrekIssues(GetIssueName());
        if (!issues) {
            continue;
        }
        if (issues.size() > 1) {
            errors.AddMessage("fetch_issue", "there are several st links");
            return false;
        }
        found = true;
        TStartrekClient::TTicket ticket;
        if (!server.GetDriveAPI()->GetStartrekClient().GetIssueInfo(issues.front(), ticket, errors)) {
            return false;
        }
        NJson::TJsonValue result = ticket.SerializeToJson();
        for (const auto& parameter : TBase::Storage.GetEnumOutputs()) {
            NJson::TJsonValue* value = result.GetValueByPath(ToString(parameter));
            if (value) {
                TBase::Storage.AddParameter(parameter, value->GetStringRobust());
            }
        }
    }
    if (!found) {
        errors.AddMessage("fetch_issue", "cannot find st " + GetIssueName() + " link");
        return false;
    }
    return true;
}

template class ISessionInfoFromStartrekTemplate<ESessionIllegalParkingOutput>;

TMaybe<TVector<TSimpleSessionInfo>> GetNextSessions(const NDrive::IServer& server, TMessagesCollector& errors, const TSimpleSessionInfo& sessionInfo, const TDuration after = TDuration::Days(5), const ui32 limit = 1) {
    if (!sessionInfo.GetCarId()) {
        return TVector<TSimpleSessionInfo>();
    }
    TVector<TMinimalCompiledRiding> localSessions;
    if (!NDocumentManager::GetCompiledRidingsByCar(sessionInfo.GetCarId(), sessionInfo.GetFinishTime(), sessionInfo.GetFinishTime() + after, localSessions, server, errors)) {
        return {};
    }
    std::sort(localSessions.begin(), localSessions.end(), [](const TMinimalCompiledRiding& a, const TMinimalCompiledRiding& b) -> bool {
        return a.GetStartInstant() < b.GetStartInstant();
    });
    TVector<TSimpleSessionInfo> nextCarSessions;
    for (const auto& session : localSessions) {
        if (nextCarSessions.size() >= limit) {
            break;
        }
        if (session.GetSessionId() == sessionInfo.GetSessionId()) {
            continue;
        }
        TSimpleSessionInfo ride(session.GetSessionId());
        if (!ride.InitSimple(server, errors)) {
            return {};
        }
        nextCarSessions.emplace_back(std::move(ride));
    }
    return nextCarSessions;
}

bool TSessionIllegalParkingTemplate::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) {
    if (json.Has(ViolationTimeKey)) {
        ViolationTime = TInstant::Seconds(json[ViolationTimeKey].GetUInteger());
    }
    if (json[UseViolationTimeKey].GetBoolean() && ViolationTime == TInstant::Zero()) {
        errors.AddMessage("input_parameter", "incorrect violation time");
        return false;
    }
    TString tableData;
    if (json[UseViolationTimeKey].GetBoolean()) {
        auto table = TStringBuilder()
            << server.GetSettings().GetValueDef<TString>("document_manager.illegal_parking_template.next_session_title", "Следующий пользователь: \\\\") << Endl
            << "\\setlength\\LTleft{0pt}" << Endl
            << "\\setlength\\LTright{0pt}" << Endl
            << "\\begin{longtable}{@{\\extracolsep{\\fill}}|l|r|r|}" << Endl
            << "\\hline" << Endl
            << " \\textbf{Операция} & \\textbf{Начало} & \\textbf{Конец} \\\\ \\hline" << Endl;
        TString sessionId = json[ToString(EInput::SessionId)].GetString();
        TSimpleSessionInfo sessionInfo(sessionId);
        if (!sessionInfo.Init(server, errors)) {
            return false;
        }
        const auto nearSessions = GetNextSessions(server, errors, sessionInfo);
        if (!nearSessions) {
            return false;
        }
        ui32 rowsCount = 0;
        for (const auto& nearSession : *nearSessions) {
            if (nearSession.GetStartTime() < GetViolationTime()) {
                for (const auto& event : nearSession.GetLocalEvents()) {
                    if (event.GetName() == "old_state_acceptance" && event.GetFinishTimestamp() > GetViolationTime()) {
                        table << "\\textcolor{red}{" << NTexBuilder::TTextStyle::Quote(::ToString(event.GetAction())) << " (" << NTexBuilder::TTextStyle::Quote(event.GetName()) << ")" << "}"
                            << " & " << "\\textcolor{red}{" << THRTime(event.GetStartTimestamp()).ToString() << "}"
                            << " & " << "\\textcolor{red}{" << THRTime(event.GetFinishTimestamp()).ToString() << "}";
                    } else {
                        table << NTexBuilder::TTextStyle::Quote(::ToString(event.GetAction())) << " (" << NTexBuilder::TTextStyle::Quote(event.GetName()) << ")"
                            << " & " << THRTime(event.GetStartTimestamp()).ToString()
                            << " & " << THRTime(event.GetFinishTimestamp()).ToString();
                    }
                    table << " \\\\" << Endl;
                    table << "\\hline" << Endl;
                }
            ++rowsCount;
            }
        }
        table << "\\end{longtable}" << Endl;
        if (rowsCount) {
            tableData = table;
        }
    }
    Storage.AddParameter(EOutput::NextSessionsEventsTexTable, tableData);
    return TBase::Fetch(json, server, errors);
}

TString TSessionIllegalParkingTemplate::GetTagName(const NDrive::IServer& server) const {
    CHECK_WITH_LOG(server.GetDocumentsManager());
    return server.GetDocumentsManager()->GetConfig().GetIllegalParkingTag();
}

TString TSessionIllegalParkingTemplate::GetIssueName() const {
    return "ILLEGALPARKING";
}

TString TSessionIllegalParkingTemplate::UseViolationTimeKey = "use_illegal_parking_time";
TString TSessionIllegalParkingTemplate::ViolationTimeKey = "illegal_parking_time";

template class ISessionInfoFromStartrekTemplate<ESessionEvacuationOutput>;

bool TSessionEvacuationTemplate::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& /*server*/, TMessagesCollector& errors) {
    if (json.Has(ViolationTimeKey)) {
        ViolationTime = TInstant::Seconds(json[ViolationTimeKey].GetUInteger());
    }
    if (json[UseViolationTimeKey].GetBoolean() && ViolationTime == TInstant::Zero()) {
        errors.AddMessage("input_parameter", "incorrect violation time");
        return false;
    }
    return true;
}

TString TSessionEvacuationTemplate::GetTagName(const NDrive::IServer& server) const {
    CHECK_WITH_LOG(server.GetDocumentsManager());
    return server.GetDocumentsManager()->GetConfig().GetEvacuationTag();
}
TString TSessionEvacuationTemplate::GetIssueName() const {
    return "DRIVEEVACUATION";
}

TString TSessionEvacuationTemplate::UseViolationTimeKey = "use_evacuation_time";
TString TSessionEvacuationTemplate::ViolationTimeKey = "evacuation_time";
