#pragma once

#include "car_template.h"
#include "disk_templates.h"
#include "user_templates.h"

#include <drive/backend/doc_packages/template.h>

#include <drive/library/cpp/common/status.h>

#include <rtline/util/types/accessor.h>

enum class ESessionInput {
    SessionId /* "session_id" */,
};

template <class EOutput>
class ISessionCommonTemplate : public ITemplateDataByStorage<EOutput> {
public:
    using EInput = ESessionInput;

public:
    virtual TVector<TString> GetInputs() const override {
        TVector<TString> result;
        for (auto&& var : GetEnumAllValues<EInput>()) {
            result.push_back(ToString(var));
        }
        return result;
    }

    virtual void AddInputsToScheme(const IServerBase& /* server */, NDrive::TScheme& scheme) const override {
        if (!scheme.HasField(ToString(EInput::SessionId))) {
            scheme.Add<TFSString>(ToString(EInput::SessionId), "Идентификатор сессии").SetVisual(TFSString::EVisualType::GUID).SetRequired(true);
        }
    }
};

enum class ESessionOutput {
    Id /* "id" */,
    StartTime /* "start_time" */,
    StartTimestamp /* "start_timestamp" */,
    FinishTime /* "finish_time" */,
    FinishTimestamp /* "finish_timestamp" */,
    FinishHrTime /* "finish_hr_time" */,
    SimpleHrFinishTime /* "simple_hr_finish_time" */,
    Price /* "total_price" */,
    PriceRaw /* "total_price_raw" */,
    TotalDuration /* "total_duration" */,
    Bonus /* "bonus" */,
    BonusRaw /* "bonus_raw" */,
    SessionDept /* "session_dept" */,
    SessionDeptRaw /* "session_dept_raw" */,
    AcceptanceTime /* "acceptance_time" */,
    AcceptanceTimestamp /* "acceptance_timestamp" */,
    AcceptanceTimeDate /* "acceptance_time_date" */,
    AcceptanceTimeMinutes /* "acceptance_time_minutes" */,
    EventsTexTable /* "events_tex_table" */,
    Mileage /* "mileage" */,
    MileageRaw /* "mileage_raw" */,
    OfferName /* "offer_name" */,
    OfferFullDetails /* "offer_full_details" */,
    StartAddress /* "start_address" */,
    LastAddress /* "last_address" */,
    StartGeoTags /* "start_geo_tags" */,
    LastGeoTags /* "last_geo_tags" */,
    StartPoint /* "start_point" */,
    LastPoint /* "last_point" */,
    Closed /* "closed" */,
    Agreement /* "agreement" */,
    CompanyName /* "company_name" */,
    RidingResult /* "riding_result" */,
};

class TSessionTemplate : public ISessionCommonTemplate<ESessionOutput> {
    using TBase = ISessionCommonTemplate<ESessionOutput>;
public:
    virtual bool Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) override;
    virtual void FetchOverride(const NJson::TJsonValue& json) override;
    virtual void Substitude(TString& inputData, const TDataEscape escape) const override;
    virtual void AddInputsToScheme(const IServerBase& server, NDrive::TScheme& scheme) const override;

    virtual const TString& GetName() const override {
        return Name;
    }
private:
    static TString Name;
    TCarTemplate CarTemplate;
    TUserTemplate UserTemplate;
    TUserPhotoTemplate UserPhotoTemplate;
    static TFactory::TRegistrator<TSessionTemplate> Registrator;
};

enum class ESessionTrackOutput {
    Track /* "track" */,
    StartTrack /* "start_track" */,
    EndTrack /* "end_track" */,
    StartPoint /* "start_point" */,
    EndPoint /* "end_point" */,
    IllegalParkingPoint /* "illegal_parking_point" */,
    IllegalParkingTrackPoint /* "illegal_parking_track_point" */,
    EvacuationTrack /* "evacuation_track" */,
    EvacuationPoint /* "evacuation_point" */,
    EvacuationTrackPoint /* "evacuation_track_point" */,
};

class TSessionTrackTemplate : public ISessionCommonTemplate<ESessionTrackOutput> {
    using TBase = ISessionCommonTemplate<ESessionTrackOutput>;
public:
    virtual TVector<TString> GetInputs() const override {
        TVector<TString> result = TBase::GetInputs();
        result.push_back(UseCustomStartTimestampKey);
        result.push_back(CustomStartTimestampKey);
        result.push_back(UseCustomFinishTimestampKey);
        result.push_back(CustomFinishTimestampKey);
        result.push_back(TrackTypesKey);
        return result;
    }

    virtual void AddInputsToScheme(const IServerBase& server, NDrive::TScheme& scheme) const override {
        TBase::AddInputsToScheme(server, scheme);
        if (!scheme.HasField(UseCustomStartTimestampKey)) {
            scheme.Add<TFSBoolean>(UseCustomStartTimestampKey, "Указать время начала сессии(для трека)").SetDefault(false);
        }

        if (!scheme.HasField(CustomStartTimestampKey)) {
            scheme.Add<TFSNumeric>(CustomStartTimestampKey, "Время начала сессии(для трека)").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(false);
        }

        if (!scheme.HasField(UseCustomFinishTimestampKey)) {
            scheme.Add<TFSBoolean>(UseCustomFinishTimestampKey, "Указать время завершения сессии(для трека)").SetDefault(false);
        }

        if (!scheme.HasField(CustomFinishTimestampKey)) {
            scheme.Add<TFSNumeric>(CustomFinishTimestampKey, "Время окончания сессии(для трека)").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(false);
        }

        if (!scheme.HasField(TrackTypesKey)) {
            scheme.Add<TFSVariants>(TrackTypesKey, "Типы интервалов трека").SetVariants({ NDrive::ECarStatus::csAcceptance, NDrive::ECarStatus::csParking, NDrive::ECarStatus::csPost, NDrive::ECarStatus::csReservation, NDrive::ECarStatus::csRide}).SetMultiSelect(true);
        }
    }

    virtual bool Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) override;

    virtual const TString& GetName() const override {
        return Name;
    }

private:
    static TString Name;
    static TFactory::TRegistrator<TSessionTrackTemplate> Registrator;

    static TString UseCustomStartTimestampKey;
    static TString CustomStartTimestampKey;
    static TString UseCustomFinishTimestampKey;
    static TString CustomFinishTimestampKey;
    static TString TrackTypesKey;
};

enum class ESessionHeavyOutput {
    WalletsTexTable /* "wallets_tex_table" */,
    DeviceId /* "device_id" */,
    UserAgent /* "user_agent" */,
};

class TSessionHeavyTemplate : public ISessionCommonTemplate<ESessionHeavyOutput> {
    using TBase = ISessionCommonTemplate<ESessionHeavyOutput>;
public:
    virtual bool Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) override;
    virtual void FetchOverride(const NJson::TJsonValue& json) override;
    virtual void Substitude(TString& inputData, TDataEscape escape) const override;
    virtual void AddInputsToScheme(const IServerBase& server, NDrive::TScheme& scheme) const override;

    virtual const TString& GetName() const override {
        return Name;
    }

private:
    static TString Name;
    TDiskTemplate DiskTemplate;
    static TFactory::TRegistrator<TSessionHeavyTemplate> Registrator;
};

enum class ESessionDiskTemplate {
};

class TSessionDiskTemplate : public ISessionCommonTemplate<ESessionDiskTemplate> {
public:
    virtual bool Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) override;
    virtual void FetchOverride(const NJson::TJsonValue& json) override;
    virtual void Substitude(TString& inputData, TDataEscape escape) const override;
    virtual const TString& GetName() const override {
        return Name;
    }

private:
    static TString Name;
    TDiskTemplate DiskTemplate;
    static TFactory::TRegistrator<TSessionDiskTemplate> Registrator;
};

template <class EOutput>
class ISessionInfoFromStartrekTemplate : public ISessionCommonTemplate<EOutput> {
    using TBase = ISessionCommonTemplate<EOutput>;
public:
    virtual bool Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) override;

private:
    virtual TString GetTagName(const NDrive::IServer& server) const = 0;
    virtual TString GetIssueName() const = 0;
};

enum class ESessionIllegalParkingOutput {
    ViolationPlace /* "violationPlace" */,
    RulingNumber /* "rulingNumber" */,
    NextSessionsEventsTexTable /* "next_sessions_events_tex_table" */,
};

class TSessionIllegalParkingTemplate : public ISessionInfoFromStartrekTemplate<ESessionIllegalParkingOutput> {
    R_READONLY(TInstant, ViolationTime, TInstant::Zero());

    using TBase = ISessionInfoFromStartrekTemplate<ESessionIllegalParkingOutput>;
public:
    virtual TVector<TString> GetInputs() const override {
        TVector<TString> result = TBase::GetInputs();
        result.push_back(UseViolationTimeKey);
        result.push_back(ViolationTimeKey);
        return result;
    }

    virtual void AddInputsToScheme(const IServerBase& server, NDrive::TScheme& scheme) const override {
        TBase::AddInputsToScheme(server, scheme);
        if (!scheme.HasField(UseViolationTimeKey)) {
            scheme.Add<TFSBoolean>(UseViolationTimeKey, "Использовать время нарушения парковки на газоне").SetDefault(false);
        }

        if (!scheme.HasField(ViolationTimeKey)) {
            scheme.Add<TFSNumeric>(ViolationTimeKey, "Время нарушения парковки на газоне").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(false);
        }
    }

    virtual bool Fetch(const NJson::TJsonValue& json, const NDrive::IServer& /*server*/, TMessagesCollector& errors) override;

    virtual const TString& GetName() const override {
        return Name;
    }

private:
    virtual TString GetTagName(const NDrive::IServer& server) const override;
    virtual TString GetIssueName() const override;

private:
    static TString Name;
    static TFactory::TRegistrator<TSessionIllegalParkingTemplate> Registrator;

public:
    static TString UseViolationTimeKey;
    static TString ViolationTimeKey;
};

enum class ESessionEvacuationOutput {
};

class TSessionEvacuationTemplate : public ISessionInfoFromStartrekTemplate<ESessionEvacuationOutput> {
    R_READONLY(TInstant, ViolationTime, TInstant::Zero());

    using TBase = ISessionInfoFromStartrekTemplate<ESessionEvacuationOutput>;
public:
    virtual TVector<TString> GetInputs() const override {
        TVector<TString> result = TBase::GetInputs();
        result.push_back(UseViolationTimeKey);
        result.push_back(ViolationTimeKey);
        return result;
    }

    virtual void AddInputsToScheme(const IServerBase& server, NDrive::TScheme& scheme) const override {
        TBase::AddInputsToScheme(server, scheme);
        if (!scheme.HasField(UseViolationTimeKey)) {
            scheme.Add<TFSBoolean>(UseViolationTimeKey, "Использовать время эвакуации").SetDefault(false);
        }

        if (!scheme.HasField(ViolationTimeKey)) {
            scheme.Add<TFSNumeric>(ViolationTimeKey, "Время эвакуации").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(false);
        }
    }

    virtual bool Fetch(const NJson::TJsonValue& json, const NDrive::IServer& /*server*/, TMessagesCollector& errors) override;

    virtual const TString& GetName() const override {
        return Name;
    }

private:
    virtual TString GetTagName(const NDrive::IServer& server) const override;
    virtual TString GetIssueName() const override;

private:
    static TString Name;
    static TFactory::TRegistrator<TSessionEvacuationTemplate> Registrator;

public:
    static TString UseViolationTimeKey;
    static TString ViolationTimeKey;
};
