#include "static_tools.h"

#include <drive/backend/compiled_riding/manager.h>

namespace {
    const TDuration CompiledRidesHistoryDepth = TDuration::Days(90);

    bool ForEachYTPayment(const auto& readerGetter, auto& collector, const NDrive::IServer& server, TMessagesCollector& errors, const TString fetchKey) {
        try {
            if (!server.GetDocumentsManager()) {
                errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось найти менеджер");
                return false;
            }
            NYT::TTableReaderPtr<NYT::TNode> reader = readerGetter(*server.GetDocumentsManager());
            for (; reader->IsValid(); reader->Next()) {
                NYT::TNode inputRow = reader->GetRow();
                TStringStream stream;
                NYT::NodeToYsonStream(inputRow, &stream, NYson::EYsonFormat::Pretty);

                NJson::TJsonValue jsonRow;
                NJson2Yson::DeserializeYsonAsJsonValue(&stream, &jsonRow, true);
                TPaymentTask payment;
                if (!payment.DeserializeFromJson(jsonRow, errors)) {
                    errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить payment из таблицы:" + fetchKey);
                    return false;
                }
                collector(std::move(payment));
            }
        } catch (const std::exception& e) {
            errors.AddMessage("YTError", FormatExc(e));
            ERROR_LOG << "YTError " << FormatExc(e) << Endl;
            return false;
        }
        return true;
    }

    template <class T>
    bool GetCompiledRiding(const TString& sessionId, TObjectEvent<T>& ride, const NDrive::IServer& server, TMessagesCollector& errors) {
        TVector<THistoryRideObject> rides;

        if (!NDocumentManager::GetHistoryRides(sessionId, "", "", TInstant::Zero(), TInstant::Max(), rides, server, errors)) {
            errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить сессию из compiled_rides:" + sessionId);
            return false;
        }

        if (!rides.empty()) {
            const bool useYDB = server.GetSettings().GetValueDef<bool>("ydb.document_manager.use_ydb", false);
            if constexpr (std::is_same<T, TMinimalCompiledRiding>::value) {
                if (rides.front().GetRiding()) {
                    ride = *rides.front().GetRiding();
                    return true;
                }
            } else {
                THistoryRideObject::FetchFullRiding(&server, rides, useYDB);
            }
            auto fullRiding = rides.front().GetFullCompiledRiding(useYDB);
            if (!fullRiding) {
                errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить информацию о сессии:" + sessionId);
                return false;
            }
            ride = TObjectEvent<T>(*fullRiding, EObjectHistoryAction::Add, rides.front().GetLastTS(), rides.front().GetUserId(), "frontend", "");
            return true;
        }
        return NDocumentManager::GetCompiledRidingFromYT(sessionId, ride, server, errors);
    }
}

namespace NDocumentManager {
    bool GetHistoryRides(const TString& sessionId, const TString& userId, const TString& carId, const TInstant start, const TInstant finish, TVector<THistoryRideObject>& rides, const NDrive::IServer& server, TMessagesCollector& errors) {
        THistoryRidesContext ridesContext(server, start);
        auto tx = Yensured(server.GetDriveAPI())->template BuildTx<NSQL::ReadOnly>();
        auto ydbTx = Yensured(server.GetDriveAPI())->template BuildYdbTx<NSQL::ReadOnly>("document_manager", &server);
        if (!ridesContext.Initialize(sessionId, userId, { carId }, tx, ydbTx)) {
            errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось инициализировать сессию из compiled_rides(" + sessionId + "): " + tx.GetStringReport());
            return false;
        }

        bool hasMore = true;
        rides = ridesContext.GetSessions(finish, 100000, &hasMore);
        if (hasMore) {
            errors.AddMessage("ui_errors", "Внутренняя ошибка: слишком много сессий в указанном интервале");
            return false;
        }
        return true;
    }

    bool GetFullCompiledRiding(const TString& sessionId, TObjectEvent<TFullCompiledRiding>& ride, const NDrive::IServer& server, TMessagesCollector& errors) {
        return GetCompiledRiding(sessionId, ride, server, errors);
    }

    bool GetMinimalCompiledRiding(const TString& sessionId, TObjectEvent<TMinimalCompiledRiding>& ride, const NDrive::IServer& server, TMessagesCollector& errors) {
        return GetCompiledRiding(sessionId, ride, server, errors);
    }

    bool GetDeviceInfo(const TString& sessionId, TString& deviceId, TString& userAgent, const NDrive::IServer& server, TMessagesCollector& errors) {
        try {
            if (!server.GetDocumentsManager()) {
                errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось найти менеджер");
                return false;
            }
            NYT::TTableReaderPtr<NYT::TNode> reader = server.GetDocumentsManager()->GetDeviceTableReaderBySession(sessionId);
            for (; reader->IsValid(); reader->Next()) {
                NYT::TNode inputRow = reader->GetRow();
                TStringStream stream;
                NYT::NodeToYsonStream(inputRow, &stream, NYson::EYsonFormat::Pretty);

                NJson::TJsonValue jsonRow;
                NJson2Yson::DeserializeYsonAsJsonValue(&stream, &jsonRow, true);
                if (!NJson::ParseField(jsonRow, "device_id_accept", deviceId, /* required = */ true, errors) || !NJson::ParseField(jsonRow, "user_agent", userAgent, /* required = */ true, errors)) {
                    errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить device_info из таблицы:" + sessionId);
                    return false;
                }
            }
        } catch (const std::exception& e) {
            errors.AddMessage("YTError", FormatExc(e));
            ERROR_LOG << "YTError " << FormatExc(e) << Endl;
            return false;
        }
        return true;
    }

    bool GetCompiledRidingsByObject(const TString& objectId, const TInstant start, const TInstant finish, const NEntityTagsManager::EEntityType type, TVector<TMinimalCompiledRiding>& rides, const NDrive::IServer& server, TMessagesCollector& errors) {
        if (start > ModelingNow()) {
            return true;
        }

        TSet<TString> sessionIds;
        bool useYDB = server.GetYDB() && server.GetSettings().GetValueDef<bool>("ydb.document_manager.use_ydb", false);
        if (useYDB) {
            const ui64 ydbHistoryLimit = server.GetSettings().GetValueDef("ydb.document_manager.ydb_history_limit", 0);
            useYDB &= ydbHistoryLimit && start > TInstant::Seconds(ydbHistoryLimit);
        }
        if (!useYDB && start + CompiledRidesHistoryDepth < ModelingNow()) {
            try {
                if (!server.GetDocumentsManager()) {
                    errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось найти менеджер");
                    return false;
                }
                NYT::TTableReaderPtr<NYT::TNode> reader = server.GetDocumentsManager()->GetTableReaderByObject(type, objectId, start, finish + server.GetDocumentsManager()->GetConfig().GetSessionMaxDuration());
                for (; reader->IsValid(); reader->Next()) {
                    NYT::TNode inputRow = reader->GetRow();
                    TStringStream stream;
                    NYT::NodeToYsonStream(inputRow, &stream, NYson::EYsonFormat::Pretty);

                    NJson::TJsonValue jsonRow;
                    NJson2Yson::DeserializeYsonAsJsonValue(&stream, &jsonRow, true);
                    TMinimalCompiledRiding minimalCompiledRiding;
                    if (!TBaseDecoder::DeserializeFromJson(minimalCompiledRiding, jsonRow)) {
                        errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить minimal_compiled_ride из таблицы:" + objectId);
                        return false;
                    }
                    if (minimalCompiledRiding.GetStartInstant() > finish) {
                        continue;
                    }
                    sessionIds.emplace(minimalCompiledRiding.GetSessionId());
                    rides.emplace_back(std::move(minimalCompiledRiding));
                }
            } catch (const std::exception& e) {
                errors.AddMessage("YTError", FormatExc(e));
                ERROR_LOG << "YTError " << FormatExc(e) << Endl;
                return false;
            }
        }

        TVector<THistoryRideObject> historyRides;
        switch (type) {
        case NEntityTagsManager::EEntityType::Car:
            if (!GetHistoryRides("", "", objectId, start, finish, historyRides, server, errors)) {
                errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить сессии для авто из compiled_rides:" + objectId);
                return false;
            }
            break;
        case NEntityTagsManager::EEntityType::User:
            if (!GetHistoryRides("", objectId, "", start, finish, historyRides, server, errors)) {
                errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить сессии для пользователя из compiled_rides:" + objectId);
                return false;
            }
            break;
        default:
            errors.AddMessage("ui_errors", "Внутренняя ошибка: некорректный тип " + ::ToString(type));
            return false;
        }

        if (historyRides.empty()) {
            return true;
        }
        std::reverse(historyRides.begin(), historyRides.end());
        for (const auto& historyRide : historyRides) {
            if (sessionIds.contains(historyRide.GetSessionId())) {
                continue;
            }
            rides.emplace_back(std::move(*historyRide.GetMinimalCompiledRiding()));
        }
        return true;
    }

    bool GetCompiledRidingsByCar(const TString& carId, const TInstant start, const TInstant finish, TVector<TMinimalCompiledRiding>& rides, const NDrive::IServer& server, TMessagesCollector& errors) {
        return GetCompiledRidingsByObject(carId, start, finish, NEntityTagsManager::EEntityType::Car, rides, server, errors);
    }

    bool GetCompiledRidingsByUser(const TString& userId, const TInstant start, const TInstant finish, TVector<TMinimalCompiledRiding>& rides, const NDrive::IServer& server, TMessagesCollector& errors) {
        return GetCompiledRidingsByObject(userId, start, finish, NEntityTagsManager::EEntityType::User, rides, server, errors);
    }

    bool GetCompiledRidingsByInterval(const TInstant start, const TInstant finish, TVector<TMinimalCompiledRiding>& rides, const NDrive::IServer& server, TMessagesCollector& errors) {
        if (start > ModelingNow()) {
            return true;
        }

        TSet<TString> sessionIds;
        NDrive::TEntitySession ydbTx;
        const ui64 ydbHistoryLimit = server.GetSettings().GetValueDef("ydb.document_manager.ydb_history_limit", 0);
        if (ydbHistoryLimit && start > TInstant::Seconds(ydbHistoryLimit)) {
            ydbTx = server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("document_manager", &server);
        }
        if (!ydbTx && start + CompiledRidesHistoryDepth < ModelingNow()) {
            try {
                if (!server.GetDocumentsManager()) {
                    errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось найти менеджер");
                    return false;
                }
                NYT::TTableReaderPtr<NYT::TNode> reader = server.GetDocumentsManager()->GetSessionsTableReaderByInterval(start, finish + server.GetDocumentsManager()->GetConfig().GetSessionMaxDuration());
                for (; reader->IsValid(); reader->Next()) {
                    NYT::TNode inputRow = reader->GetRow();
                    TStringStream stream;
                    NYT::NodeToYsonStream(inputRow, &stream, NYson::EYsonFormat::Pretty);

                    NJson::TJsonValue jsonRow;
                    NJson2Yson::DeserializeYsonAsJsonValue(&stream, &jsonRow, true);
                    TMinimalCompiledRiding minimalCompiledRiding;
                    if (!TBaseDecoder::DeserializeFromJson(minimalCompiledRiding, jsonRow)) {
                        errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить minimal_compiled_ride из таблицы interval");
                        return false;
                    }
                    if (minimalCompiledRiding.GetStartInstant() > finish) {
                        continue;
                    }
                    sessionIds.emplace(minimalCompiledRiding.GetSessionId());
                    rides.emplace_back(std::move(minimalCompiledRiding));
                }
            } catch (const std::exception& e) {
                errors.AddMessage("YTError", FormatExc(e));
                ERROR_LOG << "YTError " << FormatExc(e) << Endl;
                return false;
            }
        }

        auto tx = server.GetDriveAPI()->GetMinimalCompiledRides().BuildSession(true);
        THistoryRidesContext ridesContext(server, start);
        if (!ridesContext.Initialize(tx, ydbTx, finish)) {
            errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось инициализировать сессию из compiled_rides: " + tx.GetStringReport());
            return false;
        }

        bool hasMore = true;
        TVector<THistoryRideObject> historyRides = ridesContext.GetSessions(finish, 10000000, &hasMore);
        if (hasMore) {
            errors.AddMessage("ui_errors", "Внутренняя ошибка: слишком много сессий в указанном интервале");
            return false;
        }

        if (historyRides.empty()) {
            return true;
        }
        std::reverse(historyRides.begin(), historyRides.end());
        for (const auto& historyRide : historyRides) {
            if (sessionIds.contains(historyRide.GetSessionId())) {
                continue;
            }
            rides.emplace_back(std::move(*historyRide.GetMinimalCompiledRiding()));
        }
        return true;
    }

    bool GetSessionPaymentsFromYT(const TString& sessionId, TCachedPayments& payments, const NDrive::IServer& server, TMessagesCollector& errors) {
        auto getter = [sessionId] (const TDocumentsManager& mgr) { return mgr.GetPaymentTasksTableReaderBySession(sessionId); };
        auto collector = [&payments] (TPaymentTask&& task) { payments.Update(std::move(task)); };
        return ForEachYTPayment(getter, collector, server, errors, sessionId);
    }

    TOptionalPayments GetPaymentByRRN(const TString& rrn, const NDrive::IServer& server, TMessagesCollector& errors) {
        TVector<TPaymentTask> payments;
        auto getter = [rrn] (const TDocumentsManager& mgr) { return mgr.GetPaymentTasksTableReaderByRRN(rrn); };
        auto collector = [&payments] (TPaymentTask&& task) { payments.emplace_back(std::move(task)); };
        if (!ForEachYTPayment(getter, collector, server, errors, rrn)) {
            return {};
        }
        return payments;
    }

    TOptionalPayments GetPaymentsByCard(const TString& mask, const ui64 sum, const TRange<TInstant>& range, const NDrive::IServer& server, TMessagesCollector& errors) {
        TVector<TPaymentTask> payments;
        auto getter = [&mask, sum, &range] (const TDocumentsManager& mgr) { return mgr.GetPaymentTasksTableReaderByCard(mask, sum, range); };
        auto collector = [&payments] (TPaymentTask&& task) { payments.emplace_back(std::move(task)); };
        if (!ForEachYTPayment(getter, collector, server, errors, mask)) {
            return {};
        }
        return payments;
    }
}
