#pragma once

#include <drive/backend/doc_packages/manager.h>

#include <drive/backend/billing/interfaces/payments.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/history_iterator/history_iterator.h>

#include <library/cpp/json/yson/json2yson.h>
#include <library/cpp/yson/node/node_io.h>

#include <mapreduce/yt/interface/io.h>
#include <mapreduce/yt/interface/node.h>

namespace NDocumentManager {

    template <class TRiding>
    bool GetCompiledRidingFromYT(const TString& sessionId, TRiding& ride, const NDrive::IServer& server, TMessagesCollector& errors) {
        try {
            if (!server.GetDocumentsManager()) {
                errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось найти менеджер");
                return false;
            }
            NYT::TTableReaderPtr<NYT::TNode> reader = server.GetDocumentsManager()->GetTableReaderBySession(sessionId);
            for (; reader->IsValid(); reader->Next()) {
                NYT::TNode inputRow = reader->GetRow();
                TStringStream stream;
                NYT::NodeToYsonStream(inputRow, &stream, NYson::EYsonFormat::Pretty);

                NJson::TJsonValue jsonRow;
                NJson2Yson::DeserializeYsonAsJsonValue(&stream, &jsonRow, true);

                if (!TBaseDecoder::DeserializeFromJson(ride, jsonRow)) {
                    errors.AddMessage("ui_errors", "Внутренняя ошибка: не удалось восстановить сессию из таблицы:" + sessionId);
                    return false;
                }
                return true;
            }
        } catch (const std::exception& e) {
            errors.AddMessage("YTError", FormatExc(e));
            ERROR_LOG << "YTError " << FormatExc(e) << Endl;
            return false;
        }
        errors.AddMessage("ui_errors", "Сессия с идентификатором " + sessionId + " не найдена");
        return false;
    }

    bool GetHistoryRides(const TString& sessionId, const TString& userId, const TString& carId, const TInstant start, const TInstant finish, TVector<THistoryRideObject>& rides, const NDrive::IServer& server, TMessagesCollector& errors);

    bool GetFullCompiledRiding(const TString& sessionId, TObjectEvent<TFullCompiledRiding>& ride, const NDrive::IServer& server, TMessagesCollector& errors);
    bool GetMinimalCompiledRiding(const TString& sessionId, TObjectEvent<TMinimalCompiledRiding>& ride, const NDrive::IServer& server, TMessagesCollector& errors);

    bool GetDeviceInfo(const TString& sessionId, TString& deviceId, TString& userAgent, const NDrive::IServer& server, TMessagesCollector& errors);

    bool GetCompiledRidingsByObject(const TString& objectId, const TInstant start, const TInstant finish, const NEntityTagsManager::EEntityType type, TVector<TMinimalCompiledRiding>& rides, const NDrive::IServer& server, TMessagesCollector& errors);

    bool GetCompiledRidingsByCar(const TString& carId, const TInstant start, const TInstant finish, TVector<TMinimalCompiledRiding>& rides, const NDrive::IServer& server, TMessagesCollector& errors);
    bool GetCompiledRidingsByUser(const TString& userId, const TInstant start, const TInstant finish, TVector<TMinimalCompiledRiding>& rides, const NDrive::IServer& server, TMessagesCollector& errors);
    bool GetCompiledRidingsByInterval(const TInstant start, const TInstant finish, TVector<TMinimalCompiledRiding>& rides, const NDrive::IServer& server, TMessagesCollector& errors);

    bool GetSessionPaymentsFromYT(const TString& sessionId, TCachedPayments& payments, const NDrive::IServer& server, TMessagesCollector& errors);
    TOptionalPayments GetPaymentByRRN(const TString& rrn, const NDrive::IServer& server, TMessagesCollector& errors);
    TOptionalPayments GetPaymentsByCard(const TString& mask, const ui64 sum, const TRange<TInstant>& range, const NDrive::IServer& server, TMessagesCollector& errors);
};
