#include "user_templates.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/user_document_photos/manager.h>

using namespace NTemplateData;

TString TUserTemplate::Name("user");
TString TUserPhotoTemplate::Name("user_photo");

TSet<TUserTemplate::EOutput> TUserTemplate::DateParameters = { TUserTemplate::EOutput::RegistrationDate, TUserTemplate::EOutput::PassportIssueDate, TUserTemplate::EOutput::PassportBirthDate, TUserTemplate::EOutput::DrivingLicenseExperienceFrom, TUserTemplate::EOutput::DrivingLicenseIssueDate, TUserTemplate::EOutput::DrivingLicenseValidTo };

ITemplateData::TFactory::TRegistrator<TUserTemplate> TUserTemplate::Registrator(TUserTemplate::Name);
ITemplateData::TFactory::TRegistrator<TUserPhotoTemplate> TUserPhotoTemplate::Registrator(TUserPhotoTemplate::Name);

void TUserTemplate::AddParameter(TUserTemplate::EOutput parameter, const NJson::TJsonValue& json) {
    THRDate date;
    if (DateParameters.contains(parameter) && IHRInstant::TryFromString(json.GetStringRobust(), date)) {
        Storage.AddParameter(parameter, date.ToString());
    } else {
        Storage.AddParameter(parameter, json.GetStringRobust());
    }
}

bool TUserTemplate::Fetch(const TString& userId, const NDrive::IServer& server, TMessagesCollector& errors) {
    auto fetchResult = server.GetDriveAPI()->GetUsersData()->FetchInfo(userId);
    auto dataPtr = fetchResult.GetResultPtr(userId);
    if (!dataPtr) {
        errors.AddMessage("fetch_user_template", "unknown user_id");
        return false;
    }

    NJson::TJsonValue userInfo = dataPtr->GetFullReportSync(NUserReport::ReportAll, server);
    for (const auto& parameter : Storage.GetEnumOutputs()) {
        NJson::TJsonValue* value = userInfo.GetValueByPath(ToString(parameter));
        if (value) {
            AddParameter(parameter, *value);
        } else if (auto defaultVal = server.GetSettings().GetValue<TString>("document_manager.user_template.default." + ToString(parameter))) {
            Storage.AddParameter(parameter, *defaultVal);
        }
    }
    Storage.AddParameter(EOutput::UserName, dataPtr->GetFullName());
    Storage.AddParameter(EOutput::Id, userId);
    return true;
}

TSet<NUserDocument::EVerificationStatus> ParseVerificationStatuses(const TString& statusParam) {
    TSet<NUserDocument::EVerificationStatus> verificationStatuses;
    TSet<TString> statuses = StringSplitter(statusParam).Split(',').SkipEmpty();
    for (auto&& status : statuses) {
        NUserDocument::EVerificationStatus verificationStatus;
        if (!TryFromString(status, verificationStatus)) {
            ALERT_LOG << "Fail to parse GVar(document_manager.user_template.photo_ver_status): " << statusParam << Endl;
            continue;
        }
        verificationStatuses.insert(verificationStatus);
    }
    return verificationStatuses;
}

bool TUserPhotoTemplate::Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) {
    const auto userId = json[ToString(EUserInput::UserId)].GetString();
    auto fetchResult = server.GetDriveAPI()->GetUsersData()->FetchInfo(userId);
    auto dataPtr = fetchResult.GetResultPtr(userId);
    if (!dataPtr) {
        errors.AddMessage("fetch_user_template", "unknown user_id");
        return false;
    }

    if (!server.GetDriveAPI()->HasDocumentPhotosManager()) {
        errors.AddMessage("fetch_photos", "user photo manager not defined");
        return false;
    }
    TSet<NUserDocument::EVerificationStatus> verificationStatuses;
    if (!NJson::ParseField(json, "photo_statuses", verificationStatuses)) {
        errors.AddMessage("fetch_photos", "fail to parse verification statuses");
        return false;
    }
    if (auto statusParam = server.GetSettings().GetValue<TString>("document_manager.user_template.photo_ver_status"); statusParam && verificationStatuses.empty()) {
        verificationStatuses = ParseVerificationStatuses(*statusParam);
    }
    auto userPhotos = server.GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(userId, MakeVector(GetEnumAllValues<NUserDocument::EType>()), verificationStatuses);
    for (const auto& type : GetEnumAllValues<NUserDocument::EType>()) {
        if (type == NUserDocument::EType::Unknown) {
            continue;
        }

        auto itDocument = userPhotos.find(type);
        if (itDocument != userPhotos.end()) {
            EOutput documentType;
            EOutput photoType;
            if (!TryFromString(ToString(type) + "_document_id", documentType)) {
                ERROR_LOG << "unknown type of user photo " << ToString(type) << Endl;
                continue;
            }
            if (!TryFromString(ToString(type) + "_photo_id", photoType)) {
                ERROR_LOG << "unknown type of user photo " << ToString(type) << Endl;
                continue;
            }

            Storage.AddParameter(documentType, itDocument->second.GetDocumentId().empty() ? "null" : itDocument->second.GetDocumentId());
            Storage.AddParameter(photoType, itDocument->second.GetId());
        }
    }
    return true;
}

void TUserPhotoTemplate::AddInputsToScheme(const IServerBase& server, NDrive::TScheme& scheme) const {
    if (!scheme.HasField(ToString(EUserInput::UserId)) && !scheme.HasField("session_id")) {
        scheme.Add<TFSString>(ToString(EUserInput::UserId), "Идентификатор пользователя").SetVisual(TFSString::EVisualType::GUID).SetRequired(true);
    }
    if (!scheme.HasField("photo_statuses")) {
        TVector<TFSVariants::TCompoundVariant> variants;
        for (auto&& [key, value] : TUserDocumentPhoto::GetVerificationStatusMap()) {
            variants.emplace_back(ToString(value), key);
        }
        if (!variants.empty()) {
            auto& statuses = scheme.Add<TFSVariants>("photo_statuses", "Допустимые статусы фотографий пользователя").SetCompoundVariants(std::move(variants)).SetMultiSelect(true);
            if (auto statusParam = server.GetSettings().GetValue<TString>("document_manager.user_template.photo_ver_status")) {
                TVector<TString> values;
                TSet<NUserDocument::EVerificationStatus> verificationStatuses = ParseVerificationStatuses(*statusParam);
                Transform(verificationStatuses.begin(), verificationStatuses.end(), std::back_inserter(values), [](auto& v) { return ToString(v); });
                statuses.SetDefaults(values);
            }
        }
    }
}
