#include "manager.h"

#include <rtline/util/algorithm/container.h>

void TDocumentsManagerConfig::Init(const TYandexConfig::Section* section) {
    TDBEntitiesManagerConfig::Init(section);
    auto sections = section->GetAllChildren();
    auto itStorages = sections.find("Storages");
    if (itStorages != sections.end()) {
        auto storages = itStorages->second->GetAllChildren();
        for (const auto& storage : storages) {
            TDocumentExternalStorageOptions options;
            if (!options.Init(storage.second)) {
                continue;
            }
            StorageOptions.emplace(storage.first, options);
        }
    }
    SessionYtCluster = section->GetDirectives().Value("SessionYtCluster", SessionYtCluster);
    CompiledRidesYtDataPath = section->GetDirectives().Value("CompiledRidesYtDataPath", CompiledRidesYtDataPath);
    AssertCorrectConfig(!!CompiledRidesYtDataPath, "empty yt path");
    PaymentsYtDataPath = section->GetDirectives().Value("PaymentsYtDataPath", PaymentsYtDataPath);
    AssertCorrectConfig(!!PaymentsYtDataPath, "empty yt path");
    PaymentsTasksYtDataPath = section->GetDirectives().Value("PaymentsTasksYtDataPath", PaymentsTasksYtDataPath);
    AssertCorrectConfig(!!PaymentsTasksYtDataPath, "empty yt path");
    TracksService = section->GetDirectives().Value("TracksService", TracksService);
    AssertCorrectConfig(!!TracksService, "empty track service");
    TracksApiTimeout = section->GetDirectives().Value("TracksApiTimeout", TracksApiTimeout);
    RentTermsPath = section->GetDirectives().Value("RentTermsPath", RentTermsPath);
    AssertCorrectConfig(!!RentTermsPath, "RentTermsPath undefined");
    RentCertifiedTermsPath = section->GetDirectives().Value("RentCertifiedTermsPath", RentCertifiedTermsPath);
    AssertCorrectConfig(!!RentCertifiedTermsPath, "RentCertifiedTermsPath undefined");
    ServiceUsingTermsPath = section->GetDirectives().Value("ServiceUsingTermsPath", ServiceUsingTermsPath);
    AssertCorrectConfig(!!ServiceUsingTermsPath, "ServiceUsingTermsPath undefined");
    ServiceUsingCertifiedTermsPath = section->GetDirectives().Value("ServiceUsingCertifiedTermsPath", ServiceUsingCertifiedTermsPath);
    AssertCorrectConfig(!!ServiceUsingCertifiedTermsPath, "ServiceUsingCertifiedTermsPath undefined");
    IllegalParkingTag = section->GetDirectives().Value("IllegalParkingTag", IllegalParkingTag);
    AssertCorrectConfig(!!IllegalParkingTag, "IllegalParkingTag undefined");
    EvacuationTag = section->GetDirectives().Value("EvacuationTag", EvacuationTag);
    AssertCorrectConfig(!!EvacuationTag, "EvacuationTag undefined");
    TracksApiLifetime = section->GetDirectives().Value("TracksApiLifetime", TracksApiLifetime);
    TracksDataYtPath = section->GetDirectives().Value("TracksDataYtPath", TracksDataYtPath);
    AssertCorrectConfig(!!TracksDataYtPath, "empty yt path");
    PointMinDistance = section->GetDirectives().Value("PointMinDistance", PointMinDistance);
    SessionMaxDuration = section->GetDirectives().Value("SessionMaxDuration", SessionMaxDuration);
    OrdersYtDataPath = section->GetDirectives().Value("OrdersYtDataPath", OrdersYtDataPath);
    DevicesYtDataPath = section->GetDirectives().Value("DevicesYtDataPath", DevicesYtDataPath);
    AssertCorrectConfig(!!DevicesYtDataPath, "empty yt path");
}
void TDocumentsManagerConfig::ToString(IOutputStream& os) const {
    TDBEntitiesManagerConfig::ToString(os);
    os << "<Storages>" << Endl;
    for (const auto& storage : StorageOptions) {
        os << "<" + storage.first + ">" << Endl;
        storage.second.ToString(os);
        os << "</" + storage.first + ">" << Endl;
    }
    os << "</Storages>" << Endl;
    os << "SessionYtCluster: " << SessionYtCluster << Endl;
    os << "CompiledRidesYtDataPath: " << CompiledRidesYtDataPath << Endl;
    os << "PaymentsYtDataPath: " << PaymentsYtDataPath << Endl;
    os << "PaymentsTasksYtDataPath: " << PaymentsTasksYtDataPath << Endl;
    os << "TracksService: " << TracksService << Endl;
    os << "TracksApiTimeout: " << TracksApiTimeout << Endl;
    os << "TracksApiLifetime: " << TracksApiLifetime << Endl;
    os << "TracksDataYtPath: " << TracksDataYtPath << Endl;
    os << "RentTermsPath: " << RentTermsPath << Endl;
    os << "RentCertifiedTermsPath: " << RentCertifiedTermsPath << Endl;
    os << "ServiceUsingTermsPath: " << ServiceUsingTermsPath << Endl;
    os << "ServiceUsingCertifiedTermsPath: " << ServiceUsingCertifiedTermsPath << Endl;
    os << "IllegalParkingTag: " << IllegalParkingTag << Endl;
    os << "EvacuationTag: " << EvacuationTag << Endl;
    os << "PointMinDistance: " << PointMinDistance << Endl;
    os << "SessionMaxDuration: " << SessionMaxDuration << Endl;
    os << "OrdersYtDataPath: " << OrdersYtDataPath << Endl;
    os << "DevicesYtDataPath: " << DevicesYtDataPath << Endl;
}

TDocumentsManager::TDocumentsManager(const IHistoryContext& context, const TDocumentsManagerConfig& config, const NDrive::IServer& server)
    : DocumentsDB(context, THistoryConfig(config.GetHistoryConfig()).SetDeep(TDuration::Max()))
    , QueuedDocumentsDB(context, THistoryConfig(config.GetHistoryConfig()).SetDeep(TDuration::Days(0)))
    , YtClient(NYT::CreateClient(config.GetSessionYtCluster()))
    , Config(config)
    , Server(server)
{
    for (const auto& storage : config.GetStorageOptions()) {
        FileStorages.emplace(storage.first, storage.second.ConstructStorage());
    }
    Y_ENSURE_BT(DocumentsDB.Start());
    Y_ENSURE_BT(QueuedDocumentsDB.Start());
}

TDocumentsManager::~TDocumentsManager() {
    if (!DocumentsDB.Stop()) {
        ERROR_LOG << "cannot stop DocumentsDB manager" << Endl;
    }
    if (!QueuedDocumentsDB.Stop())  {
        ERROR_LOG << "cannot stop QueuedDocumentsDB manager" << Endl;
    }
}

bool TDocumentsManager::HasStorage(const TString& name) const {
    return FileStorages.contains(name);
}

IDocumentExternalStorage::TPtr TDocumentsManager::GetStorage(const TString& name) const {
    auto it = FileStorages.find(name);
    Y_ENSURE_BT(it != FileStorages.end());
    return it->second;
}

TBlob TDocumentsManager::BuildDocumentPreview(const NJson::TJsonValue& json, const NDrive::IServer& server, TUserPermissions::TPtr permissions, TMessagesCollector& errors) const {
    TRawDocBuilder rawBuilder;
    return BuildDocument(json, server, rawBuilder, permissions, errors);
}

bool TDocumentsManager::InitializeTexBuilder(TTexServerDocBuilder& pdfBuilder) const {
    if (!pdfBuilder.Init()) {
        return false;
    }
    TString settingsHeader;
    if (Server.GetSettings().GetValue("document_manager.tex_builder.header", settingsHeader)) {
        pdfBuilder.SetHeader(settingsHeader);
    }
    TString signature;
    if (Server.GetSettings().GetValue("document_manager.tex_builder.signature", signature)) {
        pdfBuilder.SetSignature(signature);
    }
    return true;
}

TBlob TDocumentsManager::BuildDocument(const NJson::TJsonValue& json, const NDrive::IServer& server, IDocumentAssembler& builder, TUserPermissions::TPtr permissions, TMessagesCollector& errors) const {
    TString docName = json["document_name"].GetString();
    if (!docName) {
        errors.AddMessage("parse_parameters", "incorrect document_name");
        return Default<TBlob>();
    }
    if (!!permissions && !permissions->GetAvailableDocuments().contains(docName)) {
        errors.AddMessage("permissions", "no permissions to build this type of documents " + docName);
        return Default<TBlob>();
    }

    TVector<TDocumentDescriptionPtr> documents;
    if (!GetRegisteredDocuments(documents, { docName }, false)) {
        errors.AddMessage("fetch documents", "cannot fetch document from cache " + docName);
        return Default<TBlob>();
    }

    if (documents.size() != 1) {
        errors.AddMessage("parse_parameters", "incorrect document_name " + docName);
        return Default<TBlob>();
    }

    if (!documents.front()->CreateContentFromJson(json, server, builder, errors)) {
        return Default<TBlob>();
    }

    return builder.BuildFinalDocument(errors);
}

bool TDocumentsManager::UpsertDocumentDescription(const TDocumentDescription::TPtr description, const TString& userId, NDrive::TEntitySession& session) const {
    TDocumentDescriptionPtr update(description);
    return DocumentsDB.Upsert(update, userId, session);
}

bool TDocumentsManager::RemoveDocumentDescription(const TString& name, const TString& userId, NDrive::TEntitySession& session) const {
    return DocumentsDB.Remove({ name }, userId, session);
}

bool TDocumentsManager::UpsertQueuedDocument(const TQueuedDocument& document, const TString& userId, NDrive::TEntitySession& session) const {
    TQueuedDocument newDocument = document;
    newDocument.SetLastUpdate(Now());
    return QueuedDocumentsDB.Upsert(newDocument, userId, session);
}

bool TDocumentsManager::RemoveQueuedDocument(const TString& documentId, const TString& userId, NDrive::TEntitySession& session) const {
    return QueuedDocumentsDB.Remove({ documentId }, userId, session);
}

bool TDocumentsManager::GetRegisteredDocuments(TVector<TDocumentDescriptionPtr>& documents, bool isActive /*= false*/, TInstant reqInstant /* = TInstant::Zero() */) const {
    const auto action = [&documents, isActive](const TDocumentDescriptionPtr& entity) {
        if (!isActive || entity->GetIsActive()) {
            documents.push_back(entity);
        }
    };
    return DocumentsDB.ForObjectsList(action, reqInstant);
}

bool TDocumentsManager::GetRegisteredDocuments(TVector<TDocumentDescriptionPtr>& documents, const TSet<TString>& names, bool isActive /*= false*/, TInstant reqInstant /* = TInstant::Zero() */) const {
    const auto action = [&documents, isActive](const TDocumentDescriptionPtr& entity) {
        if (!isActive || entity->GetIsActive()) {
            documents.push_back(entity);
        }
    };
    return DocumentsDB.ForObjectsList(action, reqInstant, &names);
}

NJson::TJsonValue TDocumentsManager::GetScheme(const IServerBase& server) const {
    NJson::TJsonValue resultJson(NJson::JSON_MAP);

    NDrive::TScheme contentTypeScheme;
    TSet<TString> contentTypes;
    TDocumentDescription::TFactory::GetRegisteredKeys(contentTypes);
    contentTypeScheme.Add<TFSVariants>("content_type").SetVariants(contentTypes);

    resultJson.InsertValue("content_types", contentTypeScheme.SerializeToJson());

    NJson::TJsonValue& documents = resultJson.InsertValue("documents", NJson::JSON_MAP);
    for (auto&& i : contentTypes) {
        TDocumentDescription::TPtr dd = TDocumentDescription::TFactory::Construct(i);
        if (!!dd) {
            documents.InsertValue(i, dd->GetScheme(Server).SerializeToJson());
        }
    }

    TVector<TDocumentDescriptionPtr> documentDescriptions;
    GetRegisteredDocuments(documentDescriptions);
    NJson::TJsonValue& documentInput = resultJson.InsertValue("document_input", NJson::JSON_MAP);
    for (const auto& doc : documentDescriptions) {
        auto scheme = doc->GetFullTemplatesSchemeResult(Server);
        documentInput.InsertValue(doc->GetName(), scheme.SerializeToJson());
    }

    NJson::TJsonValue& templates = resultJson.InsertValue("templates", NJson::JSON_MAP);
    TSet<TString> keys;
    ITemplateData::TFactory::GetRegisteredKeys(keys);
    for (auto&& temp : keys) {
        THolder<ITemplateData> templateImpl(ITemplateData::TFactory::Construct(temp));
        if (!templateImpl) {
            continue;
        }
        templates.AppendValue(templateImpl->GetSchemeJson());
    }

    resultJson.InsertValue("templates_scheme", ITemplateData::GetFullScheme(server));
    return resultJson;
}

TString TDocumentsManager::GetTypeName() {
    return "documents_manager";
}

bool TDocumentsManager::GetQueuedDocuments(TVector<TQueuedDocument>& result, TMaybe<EAssemblerStage> status, const TString& userId) const {
    const auto action = [&](const TQueuedDocument& doc) {
        if ((!status.Defined() || doc.GetStatus() == status.GetRef()) && (!userId || doc.GetAuthor() == userId)) {
            result.push_back(doc);
        }
    };
    return QueuedDocumentsDB.ForObjectsList(action, TInstant::Zero());
}

TMaybe<TQueuedDocument> TDocumentsManager::GetQueuedDocument(const TString& id, const TInstant reqActuality) const {
    return QueuedDocumentsDB.GetObject(id, reqActuality);
}

TInstant TDocumentsManager::GetLastTracksTimestamp() const {
    DEBUG_LOG << "Get table info: " << Endl;
    auto info = YtClient->Get(Config.GetTracksDataYtPath() + "/@row_count");
    if (!info.IsInt64() || info.AsInt64() == 0) {
        ythrow yexception() << "Cannot get table info for " << Config.GetTracksDataYtPath() << info;
    }
    DEBUG_LOG << "Create table reader by last row: " << info.AsInt64() << Endl;
    TString path = Config.GetTracksDataYtPath() + "[#" + ToString(info.AsInt64() - 1) + "]";
    auto reader = YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(path));
    if (!reader->IsValid()) {
        ythrow yexception() << "Cannot get table row by " << path;
    }
    DEBUG_LOG << "Got last table timestamp: " << reader->GetRow()["finish_timestamp"].AsInt64() << Endl;
    return TInstant::Seconds(reader->GetRow()["finish_timestamp"].AsInt64());
}
