#pragma once

#include "builder.h"
#include "document.h"
#include "storage.h"
#include "template.h"

#include <drive/backend/compiled_riding/compiled_riding.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/yt/common/writer.h>

#include <mapreduce/yt/interface/init.h>

#include <rtline/util/types/accessor.h>

class TDocumentsManagerConfig : public TDBEntitiesManagerConfig {
    using TStorageOptionMap = TMap<TString, TDocumentExternalStorageOptions>;
    R_READONLY(TStorageOptionMap, StorageOptions, {});
    R_READONLY(TString, SessionYtCluster, "hahn");
    R_READONLY(TString, CompiledRidesYtDataPath);
    R_READONLY(TString, PaymentsYtDataPath);
    R_READONLY(TString, PaymentsTasksYtDataPath, "//home/carsharing/production/cache/billing_payments");
    R_READONLY(TString, TracksService, "drive_graph");
    R_READONLY(ui64, PointMinDistance, 10);
    R_READONLY(TDuration, TracksApiTimeout, TDuration::MilliSeconds(100));
    R_READONLY(TDuration, TracksApiLifetime, TDuration::Days(179));
    R_READONLY(TString, TracksDataYtPath, "//home/carsharing/production/data/orders/raw_routes");
    R_READONLY(TString, RentTermsPath, "disk:/Документы_для_газонов_и_эвакуаций/Общие документы - незаверенные/Договор аренды ТС");
    R_READONLY(TString, RentCertifiedTermsPath, "disk:/Документы_для_газонов_и_эвакуаций/Общие документы - заверенные/Договор аренды ТС");
    R_READONLY(TString, ServiceUsingTermsPath, "disk:/Документы_для_газонов_и_эвакуаций/Общие документы - незаверенные/Условия использования сервиса Яндекс.Драйв");
    R_READONLY(TString, ServiceUsingCertifiedTermsPath, "disk:/Документы_для_газонов_и_эвакуаций/Общие документы - заверенные/Условия использования сервиса Яндекс.Драйв");
    R_READONLY(TString, IllegalParkingTag, "violation_hardpdd_parking");
    R_READONLY(TString, EvacuationTag, "fine_eva_msc");
    R_READONLY(TDuration, SessionMaxDuration, TDuration::Days(20));
    R_READONLY(TString, OrdersYtDataPath, "//home/carsharing/production/orders");
    R_READONLY(TString, OrdersLatestYtDataPath, "//home/carsharing/production/data/orders/latest");
    R_READONLY(TString, DevicesYtDataPath, "//home/carsharing/production/data/offers/accepted/session_user_agent");
public:
    void Init(const TYandexConfig::Section* section);
    void ToString(IOutputStream& os) const;
};

class TDocumentsManager {
public:
    TDocumentsManager(const IHistoryContext& context, const TDocumentsManagerConfig& config, const NDrive::IServer& server);
    ~TDocumentsManager();

    bool HasStorage(const TString& name) const;
    IDocumentExternalStorage::TPtr GetStorage(const TString& name) const;
    TVector<TString> GetStoragesNames() const {
        return MakeVector(NContainer::Keys(FileStorages));
    }

    const NDrive::IServer& GetServer() const {
        return Server;
    }

    TBlob BuildDocumentPreview(const NJson::TJsonValue& json, const NDrive::IServer& server, TUserPermissions::TPtr permissions, TMessagesCollector& errors) const;
    TBlob BuildDocument(const NJson::TJsonValue& json, const NDrive::IServer& server, IDocumentAssembler& builder, TUserPermissions::TPtr permissions, TMessagesCollector& errors) const;
    bool InitializeTexBuilder(TTexServerDocBuilder& pdfBuilder) const;

    NJson::TJsonValue GetScheme(const IServerBase& server) const;
    bool GetRegisteredDocuments(TVector<TDocumentDescriptionPtr>& result, bool isActive = false, TInstant reqInstant = TInstant::Zero()) const;
    bool GetRegisteredDocuments(TVector<TDocumentDescriptionPtr>& result, const TSet<TString>& names, bool isActive = false, TInstant reqInstant = TInstant::Zero()) const;
    bool UpsertDocumentDescription(const TDocumentDescription::TPtr description, const TString& userId, NDrive::TEntitySession& session) const;
    bool RemoveDocumentDescription(const TString& name, const TString& userId, NDrive::TEntitySession& session) const;

    bool GetQueuedDocuments(TVector<TQueuedDocument>& result, TMaybe<EAssemblerStage> status, const TString& userId = "") const;
    TMaybe<TQueuedDocument> GetQueuedDocument(const TString& id, const TInstant reqActuality = TInstant::Zero()) const;
    bool UpsertQueuedDocument(const TQueuedDocument& document, const TString& userId, NDrive::TEntitySession& session) const;
    bool RemoveQueuedDocument(const TString& documentId, const TString& userId, NDrive::TEntitySession& session) const;

    static TString GetTypeName();

    NYT::TTableReaderPtr<NYT::TNode> GetTableReaderBySession(const TString& sessionId) const {
        DEBUG_LOG << "Create table reader by session: " << sessionId << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetCompiledRidesYtDataPath() + "[\"" + sessionId + "\"]"));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetPaymentsTableReaderBySession(const TString& sessionId) const {
        DEBUG_LOG << "Create table reader by session: " << sessionId << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetPaymentsYtDataPath() + "[\"" + sessionId + "\"]"));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetDeviceTableReaderBySession(const TString& sessionId) const {
        DEBUG_LOG << "Create table reader by session: " << sessionId << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetDevicesYtDataPath() + "[\"" + sessionId + "\"]"));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetTableReaderByObject(const NEntityTagsManager::EEntityType& type, const TString& objectId, const TInstant start, const TInstant finish) const {
        DEBUG_LOG << "Create table reader by object: " << objectId << Endl;
        TString startKey = "\"" + objectId + "\"";
        if (start != TInstant::Zero()) {
            startKey = "(" + startKey + ",\"" + ToString(start.Seconds()) + "\")";
        }
        TString finishKey = "\"" + objectId + "\"";
        if (finish != TInstant::Max()) {
            finishKey = "(" + finishKey + ",\"" + ToString(finish.Seconds()) + "\")";
        }
        TString path = Config.GetCompiledRidesYtDataPath() + "__by_" + ::ToString(type) + "[" + startKey + ":" + finishKey + "]";
        INFO_LOG << path << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(path));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetSessionsTableReaderByInterval(const TInstant since, const TInstant until) const {
        DEBUG_LOG << "Create table reader by interval: " << since << "-" << until << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetCompiledRidesYtDataPath() + "__by_finish[" + ::ToString(since.Seconds()) + ":" + ::ToString(until.Seconds()) + "]"));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetTracksTableReaderByTimestamp(const TInstant sessionFinish) const {
        DEBUG_LOG << "Create table reader by timestamp: " << sessionFinish << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetTracksDataYtPath() + "[" + ::ToString(sessionFinish.Seconds()) + "]"));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetTracksTableReaderByTimestamp(const TString& condition) const {
        DEBUG_LOG << "Create table reader by timestamp: " << condition << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetTracksDataYtPath() + "[" + condition + "]"));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetTracksTableReaderByTimestamp(const TInstant since, const TInstant until) const {
        return GetTracksTableReaderByTimestamp(::ToString(since.Seconds()) + ":" + ::ToString(until.Seconds()));
    }

    TInstant GetLastTracksTimestamp() const;

    NYT::TTableReaderPtr<NYT::TNode> GetOrderTableReader(const TInstant sessionFinish) const {
        DEBUG_LOG << "Create table reader: " << TTableSelectorDay::GetTable(sessionFinish) << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetOrdersYtDataPath() + "/" + TTableSelectorDay::GetTable(sessionFinish)));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetOrdersByTimestamp(const TInstant sessionFinish) const {
        DEBUG_LOG << "Create table reader by timestamp: " << sessionFinish << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetOrdersLatestYtDataPath() + "[" + ::ToString(sessionFinish.Seconds()) + "]"));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetPaymentTasksTableReaderBySession(const TString& sessionId) const {
        DEBUG_LOG << "Create table reader by session: " << sessionId << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetPaymentsTasksYtDataPath() + "__by_session[\"" + sessionId + "\"]"));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetPaymentTasksTableReaderByRRN(const TString& rrn) const {
        DEBUG_LOG << "Create table reader by rrn: " << rrn << Endl;
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(Config.GetPaymentsTasksYtDataPath() + "__by_rrn[\"" + rrn + "\"]"));
    }

    NYT::TTableReaderPtr<NYT::TNode> GetPaymentTasksTableReaderByCard(const TString& mask, const ui64 sum, const TRange<TInstant>& range) const {
        DEBUG_LOG << "Create table reader by card: " << mask << " sum:" << sum << Endl;
        if (!range.From) {
            ythrow yexception() << "since is undefined";
        }
        const TString since = TStringBuilder() << "\"" << mask << "\"," << sum << "," << range.From->Seconds();
        auto until = TStringBuilder() << "\"" << mask << "\"," << sum + 1;
        if (range.To && *range.To != TInstant::Zero() && *range.To != TInstant::Max()) {
            until << "," << range.To->Seconds();
        }
        auto path = TStringBuilder()
            << Config.GetPaymentsTasksYtDataPath() << "__by_card"
            << "[(" << since << "):(" << until << ")]";
        return YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(path));
    }

    const TDocumentsDescriptionHistoryManager& GetDescriptionsHistoryManager() const {
        return DocumentsDB.GetHistoryManager();
    }

    const TDocumentsQueueHistoryManager& GetQueueHistoryManager() const {
        return QueuedDocumentsDB.GetHistoryManager();
    }

    const TDocumentsManagerConfig& GetConfig() const {
        return Config;
    }

private:
    mutable TDocumentsDescriptionDB DocumentsDB;
    mutable TQueuedDocumentDB QueuedDocumentsDB;
    TMap<TString, IDocumentExternalStorage::TPtr> FileStorages;
    NYT::IClientPtr YtClient;
    TDocumentsManagerConfig Config;
    const NDrive::IServer& Server;
};
