#pragma once

#include <drive/backend/abstract/base.h>
#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/object_factory/object_factory.h>

#include <rtline/util/types/messages_collector.h>

#include <util/generic/serialized_enum.h>
#include <util/string/builder.h>
#include <util/string/subst.h>

namespace NDrive {
    class IServer;
}

namespace NTemplateData {
    class IHRInstant {
        R_FIELD(TInstant, Time);
    public:
        IHRInstant() = default;

        IHRInstant(TInstant time)
            : Time(time)
        {}

        virtual ~IHRInstant() = default;

        bool operator<(const IHRInstant& other) const {
            return Time < other.GetTime();
        }

        static bool TryFromString(const TString& str, IHRInstant& result) {
            ui64 timestamp = 0;
            if (::TryFromString(str, timestamp)) {
                result.SetTime(TInstant::Seconds(timestamp));
                return true;
            }
            TInstant instant;
            if (TInstant::TryParseIso8601(str, instant)) {
                result.SetTime(instant);
                return true;
            }
            {
                // yadisk format
                TString format = "%Y/%m.%d";
                tm x;
                memset(&x, 0, sizeof(x));
                strptime(str.c_str(), format.c_str(), &x);
                result.SetTime(TInstant::Seconds(timegm(&x)));
                return true;
            }
            return false;
        }

        virtual TString ToString() const = 0;
    };

    class THRDate : public IHRInstant {
    public:
        using IHRInstant::IHRInstant;
        virtual TString ToString() const override {
            return GetTime().FormatLocalTime("%d.%m.%Y");
        }
    };

    class THRTime : public IHRInstant {
    public:
        using IHRInstant::IHRInstant;
        virtual TString ToString() const override {
            return GetTime().FormatLocalTime("%d %b %Y %R");
        }
    };

    class THRDiskDate : public IHRInstant {
    public:
        using IHRInstant::IHRInstant;
        virtual TString ToString() const override {
            return GetTime().FormatLocalTime("%Y/%m.%d");
        }
    };

    TString GetDurationString(TDuration d);
}

template <class EOutputs>
class TParametersStorage {
public:
    TVector<TString> GetOutputs() const {
        TVector<TString> result;
        for (auto&& var : GetEnumAllValues<EOutputs>()) {
            result.push_back(ToString(var));
        }
        return result;
    }

    TVector<EOutputs> GetEnumOutputs() const {
        TVector<EOutputs> result;
        for (auto&& var : GetEnumAllValues<EOutputs>()) {
            result.push_back(var);
        }
        return result;
    }

    void AddParameter(EOutputs name, const TString& value) {
        ParsedParameters[name] = value;
    }

    const TString& GetParameter(EOutputs name) const {
        auto it = ParsedParameters.find(name);
        if (it != ParsedParameters.end()) {
            return it->second;
        }
        return Default<TString>();
    }

private:
    TMap<EOutputs, TString> ParsedParameters;
};

class ITemplateData {
public:
    using TPtr = TAtomicSharedPtr<ITemplateData>;
    using TDataEscape = std::function<TString(const TString&)>;

    virtual ~ITemplateData() {}

    virtual void FetchOverride(const NJson::TJsonValue& json) {
        for (const auto& parameter : json["override_parameters"][GetName()].GetMap()) {
            auto value = parameter.second.GetStringRobust();
            if (value) {
                OverrideParameters.emplace(parameter.first, std::move(value));
            }
        }
    }
    virtual bool Fetch(const NJson::TJsonValue& json, const NDrive::IServer& server, TMessagesCollector& errors) = 0;

    virtual void Substitude(TString& inputData, const TDataEscape escape) const {
        for (auto parameter : GetOutputs()) {
            const auto itOverride = OverrideParameters.find(parameter);
            const TString value = itOverride != OverrideParameters.end() ? itOverride->second : GetParameter(parameter);
            SubstGlobal(inputData, "<" + GetName() + ":" + parameter + ">", escape ? escape(value) : value);
        }
    }

    virtual const TString& GetName() const = 0;

    NDrive::TScheme GetInputsScheme(const IServerBase& server) const {
        NDrive::TScheme scheme;
        AddInputsToScheme(server, scheme);
        return scheme;
    }

    virtual void AddInputsToScheme(const IServerBase& server, NDrive::TScheme& scheme) const = 0;

    virtual TVector<TString> GetInputs() const = 0;
    virtual TVector<TString> GetOutputs() const = 0;
    virtual const TString& GetParameter(const TString&) const = 0;

    // old version
    NJson::TJsonValue GetSchemeJson() const;
    NJson::TJsonValue GetScheme(const IServerBase&  server) const;
    static NJson::TJsonValue GetFullScheme(const IServerBase& server);

    static bool BuildFromPostData(TMap<TString, TPtr>& result, const NJson::TJsonValue& json, const TSet<TString>& templates, const NDrive::IServer& server, TMessagesCollector& errors);

public:
    using TFactory = NObjectFactory::TObjectFactory<ITemplateData, TString>;

private:
    TMap<TString, TString> OverrideParameters;
};

template <class TOutput>
class ITemplateDataByStorage : public ITemplateData {
protected:
    using EOutput = TOutput;
public:
    const TString& GetParameter(const TString& name) const override final {
        EOutput param;
        if (TryFromString(name, param)) {
            return Storage.GetParameter(param);
        }
        return Default<TString>();
    }

    virtual TVector<TString> GetOutputs() const override final {
        return Storage.GetOutputs();
    }

protected:
    TParametersStorage<EOutput> Storage;
};
