#include <drive/backend/doc_packages/manager.h>

#include <drive/backend/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>


Y_UNIT_TEST_SUITE(DocumentsSuite) {

    class TLocalConfigGenerator: public NDrive::TServerConfigGenerator {
    private:
        virtual void GetApiConfigString(IOutputStream& os) const override {
            Y_UNUSED(os);
        }
        virtual void GetBackgroundConfigString(IOutputStream& os) const override {
            Y_UNUSED(os);
        }
    };

    const TString DefaultUserId = "12789938-63f9-406d-8407-c4f7cf5d741c";

    void Clear(TDatabasePtr database) {
        auto transaction = database->CreateTransaction();
        transaction->Exec("DELETE from service_documents");
        UNIT_ASSERT(transaction->Commit());
    }

    void Prepare(TDatabasePtr db) {
        Clear(db);
    }

    bool WaitDocuments(ui32 count, const TDocumentsManager& manager) {
        TInstant start = TInstant::Now();
        while (start + TDuration::Seconds(60) > TInstant::Now()) {
            TVector<TDocumentDescriptionPtr> docs;
            UNIT_ASSERT(manager.GetRegisteredDocuments(docs));
            if (docs.size() == count) {
                return true;
            }
            Sleep(TDuration::Seconds(1));
        }
        return false;
    }

    Y_UNIT_TEST(BaseOperations) {
        DoInitGlobalLog("console", 6, false, false);
        TLocalConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TDatabasePtr db = driveApi.GetDatabase();
        Prepare(db);
        UNIT_ASSERT(server->GetDocumentsManager());
        const TDocumentsManager& manager = *(server->GetDocumentsManager());
        ui32 startDescriptionsCount = 0;
        {
            TVector<TDocumentDescriptionPtr> docs;
            UNIT_ASSERT(manager.GetRegisteredDocuments(docs));
            INFO_LOG << "docs size: " << docs.size() << Endl;
            startDescriptionsCount = docs.size();
        }
        {
            NJson::TJsonValue docJson;
            docJson["content_type"] = "raw";
            docJson["document_name"] = "simple_doc";
            docJson["document_meta"]["content"] = "Test user name is <user:user_name>";
            docJson["document_meta"]["templates"].AppendValue("user");
            docJson["comment"] = "simple_doc";
            docJson["active"] = true;
            auto description = TDocumentDescription::BuildFromJson(docJson, *server, nullptr);
            UNIT_ASSERT(description);
            NDrive::TEntitySession session(db->CreateTransaction());
            UNIT_ASSERT(manager.UpsertDocumentDescription(description, "test", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            NJson::TJsonValue docJson;
            docJson["content_type"] = "raw";
            docJson["document_name"] = "simple_post_doc";
            docJson["document_meta"]["content"] = "Test user name is <post:user_name>";
            docJson["document_meta"]["templates"].AppendValue("post");
            docJson["comment"] = "simple_doc";
            docJson["active"] = true;
            {
                auto description = TDocumentDescription::BuildFromJson(docJson, *server, nullptr);
                UNIT_ASSERT(description);
                UNIT_ASSERT(description->GetFullTemplatesScheme(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 0);
                UNIT_ASSERT(description->GetFullTemplatesSchemeResult(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 3);
                UNIT_ASSERT(description->GetFullTemplatesSchemeResult(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer)["document_format"]["default"].GetString() == "pdf");
                UNIT_ASSERT(description->GetFullTemplatesSchemeResult(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer)["queue"]["default"].GetString() == "default");
            }
            docJson["document_meta"]["document_format"] = "csv";
            docJson["document_meta"]["queue"] = "prestable";
            auto description = TDocumentDescription::BuildFromJson(docJson, *server, nullptr);
            UNIT_ASSERT(description);
            UNIT_ASSERT(description->GetFullTemplatesScheme(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 0);
            UNIT_ASSERT(description->GetFullTemplatesSchemeResult(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer)["document_format"]["default"].GetString() == "csv");
            UNIT_ASSERT(description->GetFullTemplatesSchemeResult(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer)["queue"]["default"].GetString() == "prestable");
            NDrive::TEntitySession session(db->CreateTransaction());
            UNIT_ASSERT(manager.UpsertDocumentDescription(description, "test", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            NJson::TJsonValue docJson;
            docJson["content_type"] = "package";
            docJson["document_name"] = "package_doc";
            docJson["document_meta"]["documents"].AppendValue("simple_doc");
            docJson["document_meta"]["documents"].AppendValue("simple_post_doc");
            docJson["comment"] = "package_doc";
            docJson["active"] = true;

            UNIT_ASSERT(WaitDocuments(startDescriptionsCount + 2, manager));
            auto description = TDocumentDescription::BuildFromJson(docJson, *server, nullptr);
            UNIT_ASSERT(description);
            INFO_LOG << description->GetTemplates().size() << Endl;
            UNIT_ASSERT(description->GetTemplates().size() == 0);
            UNIT_ASSERT(description->GetFullTemplatesScheme(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 1);
            NDrive::TEntitySession session(db->CreateTransaction());
            UNIT_ASSERT(manager.UpsertDocumentDescription(description, "test", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            NJson::TJsonValue docJson;
            docJson["content_type"] = "raw";
            docJson["document_name"] = "simple_post_doc";
            docJson["document_meta"]["content"] = "Test user name is <post:user_name>";
            docJson["document_meta"]["templates"].AppendValue("post");
            docJson["document_meta"]["additional_parameters"].AppendValue("user_name");
            docJson["comment"] = "simple_doc";
            docJson["active"] = true;
            auto description = TDocumentDescription::BuildFromJson(docJson, *server, nullptr);
            UNIT_ASSERT(description);
            UNIT_ASSERT(description->GetFullTemplatesScheme(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 1);
            NDrive::TEntitySession session(db->CreateTransaction());
            UNIT_ASSERT(manager.UpsertDocumentDescription(description, "test", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            TVector<TDocumentDescriptionPtr> doc;
            UNIT_ASSERT(manager.GetRegisteredDocuments(doc, { "package_doc" }, false, TInstant::Now()));
            UNIT_ASSERT(doc.size() == 1);
            UNIT_ASSERT(doc.front()->GetFullTemplatesScheme(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 2);
        }
        {
            NJson::TJsonValue docJson;
            docJson["content_type"] = "package";
            docJson["document_name"] = "package_doc_unused_parameters";
            docJson["document_meta"]["documents"].AppendValue("simple_doc");
            docJson["document_meta"]["documents"].AppendValue("package_doc");
            docJson["document_meta"]["templates"].AppendValue("user");
            docJson["document_meta"]["templates"].AppendValue("post");
            docJson["document_meta"]["templates"].AppendValue("common");
            docJson["comment"] = "package_doc";
            docJson["active"] = true;
            auto description = TDocumentDescription::BuildFromJson(docJson, *server, nullptr);
            UNIT_ASSERT(description);
            INFO_LOG << description->GetTemplates().size() << Endl;
            UNIT_ASSERT(description->GetTemplates().size() == 3);
            UNIT_ASSERT(description->GetFullTemplatesScheme(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 3);
            NDrive::TEntitySession session(db->CreateTransaction());
            UNIT_ASSERT(manager.UpsertDocumentDescription(description, "test", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            NJson::TJsonValue docJson;
            docJson["content_type"] = "repeated";
            docJson["document_name"] = "repeated";
            docJson["document_meta"]["document"] = "simple_doc";
            docJson["comment"] = "repeated";
            docJson["active"] = true;
            auto description = TDocumentDescription::BuildFromJson(docJson, *server, nullptr);
            UNIT_ASSERT(description);
            INFO_LOG << description->GetTemplates().size() << Endl;
            UNIT_ASSERT(description->GetTemplates().size() == 0);
            UNIT_ASSERT(description->GetFullTemplatesScheme(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 1);
            NDrive::TEntitySession session(db->CreateTransaction());
            UNIT_ASSERT(manager.UpsertDocumentDescription(description, "test", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            NJson::TJsonValue docJson;
            docJson["content_type"] = "repeated";
            docJson["document_name"] = "repeated_package";
            docJson["document_meta"]["document"] = "package_doc";
            docJson["comment"] = "repeated";
            docJson["active"] = true;
            auto description = TDocumentDescription::BuildFromJson(docJson, *server, nullptr);
            UNIT_ASSERT(description);
            INFO_LOG << description->GetTemplates().size() << Endl;
            UNIT_ASSERT(description->GetTemplates().size() == 0);
            UNIT_ASSERT(description->GetFullTemplatesScheme(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 1);
            NDrive::TEntitySession session(db->CreateTransaction());
            UNIT_ASSERT(manager.UpsertDocumentDescription(description, "test", session));
            UNIT_ASSERT(session.Commit());
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("service_documents");
        TVector<TDocumentDescriptionPtr> documents;
        UNIT_ASSERT(manager.GetRegisteredDocuments(documents, false, Now()));
        UNIT_ASSERT_VALUES_EQUAL(documents.size(), startDescriptionsCount + 6);

        {
            TMessagesCollector errors;
            NJson::TJsonValue json;
            json.InsertValue("user_id", DefaultUserId);
            json.InsertValue("document_name", "simple_doc");

            TBlob result = manager.BuildDocumentPreview(json, *server, nullptr, errors);
            if (result.Empty()) {
                ERROR_LOG << errors.GetStringReport() << Endl;
                UNIT_ASSERT(false);
            }
            TString resultStr(result.AsCharPtr(), result.Size());
            INFO_LOG << resultStr << Endl;
            UNIT_ASSERT_VALUES_EQUAL(result.Size(), 22);
            UNIT_ASSERT(resultStr.EndsWith("Ivan"));
        }
        {
            NJson::TJsonValue parameterJson;
            parameterJson.InsertValue("user_id", DefaultUserId);

            ui16 n = 5;
            NJson::TJsonValue parametersArray;
            for (ui16 i = 0; i < n; ++i) {
                parametersArray.AppendValue(parameterJson);
            }

            TMessagesCollector errors;
            NJson::TJsonValue json;
            json.InsertValue("document_parameters_repeated", parametersArray);
            json.InsertValue("document_name", "repeated");

            TBlob result = manager.BuildDocumentPreview(json, *server, nullptr, errors);
            if (result.Empty()) {
                ERROR_LOG << errors.GetStringReport() << Endl;
                UNIT_ASSERT(false);
            }
            TString resultStr(result.AsCharPtr(), result.Size());
            INFO_LOG << resultStr << Endl;
            UNIT_ASSERT_VALUES_EQUAL(result.Size(), 22*n);
            UNIT_ASSERT(resultStr.EndsWith("Ivan"));
        }
        {
            TMessagesCollector errors;
            NJson::TJsonValue json;
            json.InsertValue("user_name", "Ivan");
            json.InsertValue("document_name", "simple_post_doc");

            TBlob result = manager.BuildDocumentPreview(json, *server, nullptr, errors);
            if (result.Empty()) {
                ERROR_LOG << errors.GetStringReport() << Endl;
                UNIT_ASSERT(false);
            }
            TString resultStr(result.AsCharPtr(), result.Size());
            INFO_LOG << resultStr << Endl;
            UNIT_ASSERT_VALUES_EQUAL(result.Size(), 22);
            UNIT_ASSERT(resultStr.EndsWith("Ivan"));
        }
        {
            TMessagesCollector errors;
            NJson::TJsonValue json;
            json.InsertValue("document_name", "package_doc");
            json.InsertValue("user_id", DefaultUserId);
            json.InsertValue("user_name", "Ivan");

            TBlob result = manager.BuildDocumentPreview(json, *server, nullptr, errors);
            if (result.Empty()) {
                ERROR_LOG << errors.GetStringReport() << Endl;
                UNIT_ASSERT(false);
            }
            TString resultStr(result.AsCharPtr(), result.Size());
            INFO_LOG << resultStr << Endl;
            UNIT_ASSERT_VALUES_EQUAL(result.Size(), 44);
        }
        {
            TMessagesCollector errors;
            NJson::TJsonValue json;
            json.InsertValue("document_name", "package_doc_unused_parameters");
            json.InsertValue("user_id", DefaultUserId);
            json.InsertValue("user_name", "Ivan");

            TBlob result = manager.BuildDocumentPreview(json, *server, nullptr, errors);
            if (result.Empty()) {
                ERROR_LOG << errors.GetStringReport() << Endl;
                UNIT_ASSERT(false);
            }
            TString resultStr(result.AsCharPtr(), result.Size());
            INFO_LOG << resultStr << Endl;
            UNIT_ASSERT_VALUES_EQUAL(result.Size(), 66);
        }
        {
            TMessagesCollector errors;
            NJson::TJsonValue json;
            json.InsertValue("document_name", "package_doc_unused_parameters");
            json.InsertValue("user_id", DefaultUserId);
            json.InsertValue("user_name", "Ivan");

            TUserPermissions::TPtr userPermissions = driveApi.GetUserPermissions(DefaultUserId, TUserPermissionsFeatures());
            UNIT_ASSERT(!!userPermissions);

            TBlob result = manager.BuildDocumentPreview(json, *server, userPermissions, errors);
            UNIT_ASSERT(result.Empty());
        }
    }

    void BuildDescription(const NDrive::TServerGuard& server, const NJson::TJsonValue& docJson) {
        const TDocumentsManager& manager = *(server->GetDocumentsManager());
        TVector<TDocumentDescriptionPtr> docs;
        UNIT_ASSERT(manager.GetRegisteredDocuments(docs));
        INFO_LOG << "docs size: " << docs.size() << Endl;
        NJson::TJsonValue json = docJson;
        json["content_type"] = "raw";
        json["document_meta"]["content"] = "<post:post_param>";
        json["document_meta"]["templates"].AppendValue("post");
        json["document_meta"]["additional_parameters"].AppendValue("post_param");
        json["active"] = true;
        auto description = TDocumentDescription::BuildFromJson(json, *server, nullptr);
        UNIT_ASSERT(description);
        UNIT_ASSERT(description->GetFullTemplatesScheme(*server).SerializeToJson(NDrive::ISchemeElement::ReportServer).GetMap().size() == 1);
        NDrive::TEntitySession session(server->GetDriveAPI()->GetDatabase()->CreateTransaction());
        UNIT_ASSERT(manager.UpsertDocumentDescription(description, "test", session));
        UNIT_ASSERT(session.Commit());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("service_documents");
        UNIT_ASSERT(WaitDocuments(docs.size() + 1, manager));
    }

    Y_UNIT_TEST(EscapeByFormat) {
        DoInitGlobalLog("console", 6, false, false);
        TLocalConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        Prepare(server->GetDriveAPI()->GetDatabase());
        UNIT_ASSERT(server->GetDocumentsManager());
        const TDocumentsManager& manager = *(server->GetDocumentsManager());
        const TString contolStr = "!@#$%^&*()\\\"";
        {
            NJson::TJsonValue descJson;
            descJson["document_name"] = "simple_post_doc_text";
            BuildDescription(server, descJson);
            TMessagesCollector errors;
            NJson::TJsonValue json;
            json.InsertValue("post_param", contolStr);
            json.InsertValue("document_name", "simple_post_doc_text");
            TBlob result = manager.BuildDocumentPreview(json, *server, nullptr, errors);
            UNIT_ASSERT_C(!result.Empty(), errors.GetStringReport());
            const TString resultStr(result.AsCharPtr(), result.Size());
            UNIT_ASSERT_VALUES_EQUAL(resultStr, contolStr);
        }
        {
            NJson::TJsonValue descJson;
            descJson["document_name"] = "simple_post_doc_epdf";
            descJson["document_meta"]["document_format"] = "epdf";
            BuildDescription(server, descJson);
            TMessagesCollector errors;
            NJson::TJsonValue json;
            json.InsertValue("post_param", contolStr);
            json.InsertValue("document_name", "simple_post_doc_epdf");
            TBlob result = manager.BuildDocumentPreview(json, *server, nullptr, errors);
            UNIT_ASSERT_C(!result.Empty(), errors.GetStringReport());
            const TString resultStr(result.AsCharPtr(), result.Size());
            const TString expectStr = "!@\\#\\$\\% \\textasciicircum \\&*() \\textbackslash \"";
            UNIT_ASSERT_VALUES_EQUAL(resultStr, expectStr);
        }
        {
            NJson::TJsonValue descJson;
            descJson["document_name"] = "simple_post_doc_json";
            descJson["document_meta"]["document_format"] = "json";
            BuildDescription(server, descJson);
            TMessagesCollector errors;
            NJson::TJsonValue json;
            json.InsertValue("post_param", contolStr);
            json.InsertValue("document_name", "simple_post_doc_json");
            TBlob result = manager.BuildDocumentPreview(json, *server, nullptr, errors);
            UNIT_ASSERT_C(!result.Empty(), errors.GetStringReport());
            const TString resultStr(result.AsCharPtr(), result.Size());
            const TString expectStr = "!@#$%^&*()\\\\\\\"";
            UNIT_ASSERT_VALUES_EQUAL(resultStr, expectStr);
        }
    }
}
