#include "manager.h"

#include <drive/backend/abstract/drive_database.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/user.h>

TDocumentsVerificationManager::TDocumentsVerificationManager(const TDocumentsVerificationConfig& config, TDocumentPhotosManager& userDocumentPhotosManager)
    : Config(config)
    , UserDocumentPhotosManager(userDocumentPhotosManager)
{
    YangClient = MakeHolder<TYangYTClient<TYangDocumentVerificationAssignment>>(Config.GetYtClusters());
    YangScreencapClient = MakeHolder<TYangYTClient<TYangVideoScreencapAssignment>>(Config.GetYtClusters());
}

void TDocumentsVerificationManager::FilterAssignments(const TVector<TYangDocumentVerificationAssignment>& assignments, TVector<TYangDocumentVerificationAssignment>& result, const NDrive::IServer* server) const {
    result.clear();
    TSet<TString> allUserIds;
    for (auto&& assignment : assignments) {
        allUserIds.emplace(assignment.GetUserId());
    }

    auto users = server->GetDriveDatabase().GetUsersData()->FetchInfo(allUserIds);
    TSet<TString> filteredUserIds;
    bool getAll = Config.GetGeo().contains("*");
    for (auto&& it : users) {
        if (getAll || Config.GetGeo().contains(it.second.GetRegistrationGeo())) {
            filteredUserIds.insert(it.first);
        }
    }

    for (auto&& assignment : assignments) {
        if (filteredUserIds.contains(assignment.GetUserId())) {
            result.push_back(assignment);
        }
    }
}

bool TDocumentsVerificationManager::CleanupTempPools(NDrive::TEntitySession& session) const {
    if (Config.GetTempRequestsPath().empty()) {
        return true;
    }
    auto idsByPools = YangClient->FetchAssignmentIds(Config.GetTempRequestsPath());
    TSet<TString> poolsToMove;
    for (auto&& [poolId, ids] : idsByPools) {
        auto fetchResult = UserDocumentPhotosManager.GetDocumentVerificationAssignments().FetchInfo(ids, session);
        if (!fetchResult) {
            return false;
        }
        if (fetchResult.size() == ids.size()) {
            poolsToMove.insert(poolId);
        } else {
            YangClient->MarkPoolProcessed(YangClient->BuildPoolPath(Config.GetTempRequestsPath(), poolId));
        }
    }
    if (!poolsToMove.empty()) {
        YangClient->MovePools(poolsToMove, Config.GetTempRequestsPath(), Config.GetRequestsPath());
    }
    return true;
}

bool TDocumentsVerificationManager::CreatePools(TVector<TYangDocumentVerificationAssignment>& assignments, const TString& rootPath, const TString& tempPath, NDrive::TEntitySession& session) const {
    TYangPool<TYangDocumentVerificationAssignment> currentPool;
    auto requestPath = tempPath.empty() ? rootPath : tempPath;
    bool success = true;
    TSet<TString> poolIds;
    for (size_t i = 0; i < assignments.size(); ++i) {
        currentPool.AddAssignment(assignments[i]);
        if (currentPool.Size() == Config.GetMaxPoolSize() || i == assignments.size() - 1) {
            if (YangClient->CreatePool(requestPath, currentPool)) {
                for (auto&& assignment : currentPool.GetAssignments()) {
                    if (!UserDocumentPhotosManager.GetDocumentVerificationAssignments().Upsert(assignment, session)) {
                        return false;
                    }
                }
            } else {
                success = false;
            }
            poolIds.emplace(currentPool.GetId());
            currentPool = TYangPool<TYangDocumentVerificationAssignment>();
        }
    }
    if (success) {
        NJson::TJsonValue ev = NJson::TMapBuilder
            ("assignments", NJson::ToJson(assignments))
            ("root_path", requestPath)
        ;
        session.Committed().Subscribe([
            ytClusters = Config.GetYtClusters(),
            ev = std::move(ev),
            tempPath,
            rootPath,
            poolIds
        ](const NThreading::TFuture<void>& c) {
            if (c.HasValue()) {
                THolder<TYangYTClient<TYangDocumentVerificationAssignment>> client = MakeHolder<TYangYTClient<TYangDocumentVerificationAssignment>>(ytClusters);
                if (!tempPath.empty()) {
                    client->MovePools(poolIds, tempPath, rootPath);
                }
                NDrive::TEventLog::Log("YangRegisterAssignments", ev);
            }
        });
    }
    return success;
}

bool TDocumentsVerificationManager::RegisterNewAssignments(const TVector<TYangDocumentVerificationAssignment>& assignments, const TString& poolsPath, const TString& tempPath, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    TVector<TYangDocumentVerificationAssignment> filteredAssignments;
    TDocumentsVerificationManager::FilterAssignments(assignments, filteredAssignments, server);
    return TDocumentsVerificationManager::CreatePools(filteredAssignments, poolsPath, tempPath, session);
}

bool TDocumentsVerificationManager::CreateNewAssignments(const TInstant since, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!CleanupTempPools(session)) {
        return false;
    }
    switch (Config.GetTaskType()) {
    case TDocumentsVerificationConfig::ETaskType::CustomPackage:
        return CreateNewCustomAssignments(since, server, session);
    case TDocumentsVerificationConfig::ETaskType::DocumentsPackage:
        return CreateNewRegistrationAssignments(since, server, session);
    case TDocumentsVerificationConfig::ETaskType::FaceMatching:
        return CreateNewSelfieAssignments(server, session);
    }
    return false;
}

bool TDocumentsVerificationManager::CreateNewCustomAssignments(const TInstant since, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto& requiredOriginChats = Config.GetRequiredOriginChats();
    const auto& requiredChatNodes = Config.GetRequiredChatNodes();
    const auto& types = Config.GetDocumentTypes();
    const bool requireRecognizerMeta = Config.GetRequireRecognizerMeta();
    auto presentAssignments = UserDocumentPhotosManager.GetDocumentVerificationAssignments().GetNotVerifiedCustomAssignments(types);
    INFO_LOG << "looking after photos since " << since.ToString() << Endl;
    INFO_LOG << "there are " << presentAssignments.size() << " assignments in the queue" << Endl;
    TSet<ui64> presentAssignmentHashes;
    for (auto&& [_, assignment] : presentAssignments) {
        auto assignmentHash = assignment.CalcHash();
        presentAssignmentHashes.insert(assignmentHash);
    }

    TSet<TString> recentlyActiveUsers;
    {
        auto usersWithRecentPhotos = UserDocumentPhotosManager.GetUserPhotosDB().GetRecentlyActiveUsers(since);
        auto usersWithRecentVideos = UserDocumentPhotosManager.GetUserBackgroundVideosDB().GetRecentlyActiveUsers(since);

        usersWithRecentPhotos.insert(usersWithRecentVideos.begin(), usersWithRecentVideos.end());
        recentlyActiveUsers = std::move(usersWithRecentPhotos);
    }


    TMap<TString, TMap<NUserDocument::EType, TUserDocumentPhoto>> lastUploadedPhotos;
    {
        TVector<TUserDocumentPhoto> uploadedPhotos = UserDocumentPhotosManager.GetUserPhotosDB().GetAllForUsers(recentlyActiveUsers);
        INFO_LOG << "there are " << uploadedPhotos.size() << " photos which can possibly form an assignment" << Endl;
        Sort(uploadedPhotos.begin(), uploadedPhotos.end(), [](const TUserDocumentPhoto& lhs, const TUserDocumentPhoto& rhs) {
            return lhs.GetSubmittedAt() < rhs.GetSubmittedAt();
        });

        for (auto&& photo : uploadedPhotos) {
            TString chatId, topic;
            IChatRobot::ParseTopicLink(photo.GetOriginChat(), chatId, topic);
            if (!requiredOriginChats.empty()) {
                if (!requiredOriginChats.contains(chatId)) {
                    continue;
                }
            }
            TString chatNode;
            if (!requiredChatNodes.empty()) {
                TChatRobotScriptItem currentScriptItem;
                auto chatRobot = server->GetChatRobot(chatId);
                if (chatRobot &&
                    chatRobot->GetCurrentScriptItem(photo.GetUserId(), topic, currentScriptItem, TInstant::Zero()) &&
                    currentScriptItem.GetBareId(chatNode) &&
                    !requiredChatNodes.contains(chatNode)
                ) {
                    continue;
                }
            }
            if (requireRecognizerMeta && !photo.HasRecognizerMeta()) {
                continue;
            }
            if (!types.contains(photo.GetType())) {
                continue;
            }
            lastUploadedPhotos[photo.GetUserId()][photo.GetType()] = photo;
        }
        INFO_LOG << "there are " << lastUploadedPhotos.size() << " users which uploaded photos recently" << Endl;
    }

    TMap<TString, TMap<NUserDocument::EType, TUserDocumentVideo>> lastUploadedVideos;
    {
        TVector<TUserDocumentVideo> uploadedVideos = UserDocumentPhotosManager.GetUserBackgroundVideosDB().GetAllForUsers(recentlyActiveUsers);
        INFO_LOG << "there are " << uploadedVideos.size() << " videos which can possibly form an assignment" << Endl;
        Sort(uploadedVideos.begin(), uploadedVideos.end(), [](const TUserDocumentVideo& lhs, const TUserDocumentVideo& rhs) {
            return lhs.GetSubmittedAt() < rhs.GetSubmittedAt();
        });

        for (auto&& video : uploadedVideos) {
            TString chatId, topic;
            IChatRobot::ParseTopicLink(video.GetOriginChat(), chatId, topic);
            if (!requiredOriginChats.empty()) {
                if (!requiredOriginChats.contains(chatId)) {
                    continue;
                }
            }
            if (!requiredChatNodes.empty()) {
                TChatRobotScriptItem currentScriptItem;
                auto chatRobot = server->GetChatRobot(chatId);
                if (chatRobot && chatRobot->GetCurrentScriptItem(video.GetUserId(), topic, currentScriptItem, TInstant::Zero()) && !requiredChatNodes.contains(currentScriptItem.GetId())) {
                    continue;
                }
            }
            if (video.IsBackground()) {
                continue;
            }
            if (!types.contains(video.GetType())) {
                continue;
            }
            lastUploadedVideos[video.GetUserId()][video.GetType()] = video;
        }
        INFO_LOG << "there are " << lastUploadedVideos.size() << " users which uploaded videos recently" << Endl;
    }

    TMap<TString, NJson::TJsonValue> metaMap;
    GetAssignmentsMeta(recentlyActiveUsers, metaMap);

    TVector<TYangDocumentVerificationAssignment> assignmentsToCreate;
    for (const auto& userId : recentlyActiveUsers) {
        auto documentsIds = TMap<NUserDocument::EType, TString>();

        const auto& photos = lastUploadedPhotos[userId];
        const auto& videos = lastUploadedVideos[userId];
        if ((photos.size() + videos.size()) != types.size()) {
            continue;
        }

        {
            bool hasNotVerifiedPhoto = false;
            bool formsValidAssignment = true;
            for (auto&& [type, photo] : photos) {
                auto photoStatus = photo.GetVerificationStatus();
                if (photoStatus == NUserDocument::EVerificationStatus::NotYetProcessed) {
                    hasNotVerifiedPhoto = true;
                } else if (photoStatus != NUserDocument::EVerificationStatus::Ok) {
                    formsValidAssignment = false;
                }
            }

            if (!hasNotVerifiedPhoto || !formsValidAssignment) {
                continue;
            }

            for (auto&& [type, photo] : photos) {
                documentsIds.emplace(type, photo.GetId());
            }
        }

        {
            for (auto&& [type, video] : videos) {
                documentsIds.emplace(type, video.GetId());
            }
        }

        auto assignment = TYangDocumentVerificationAssignment(documentsIds, userId);
        assignment.SetMeta(metaMap[userId]);
        assignment.SetIsExperimental(Config.GetAreAssignmentsExperimental());
        for (auto&& [type, photo] : photos) {
            switch (type) {
                case NUserDocument::EType::LicenseBack:
                    assignment.SetLicenseBackConfidence(photo.OptionalRecognizerMeta());
                    break;
                case NUserDocument::EType::LicenseFront:
                    assignment.SetLicenseFrontConfidence(photo.OptionalRecognizerMeta());
                    break;
                case NUserDocument::EType::PassportBiographical:
                    assignment.SetPassportBiographicalConfidence(photo.OptionalRecognizerMeta());
                    break;
                default:
                    break;
            }
        }

        auto assignmentHash = assignment.CalcHash();
        if (presentAssignmentHashes.contains(assignmentHash)) {
            continue;
        }
        INFO_LOG << "new assignment: " << assignment.GetId() << ':' << assignmentHash << Endl;
        assignmentsToCreate.push_back(std::move(assignment));
    }

    INFO_LOG << "there are " << assignmentsToCreate.size() << " assignments to create" << Endl;
    return RegisterNewAssignments(assignmentsToCreate, Config.GetRequestsPath(), Config.GetTempRequestsPath(), server, session);
}

namespace {
    const TSet<NUserDocument::EType> NewRegistrationAssignmentPhotoTypes = {
        NUserDocument::EType::LicenseBack,
        NUserDocument::EType::LicenseFront,
        NUserDocument::EType::PassportBiographical,
        NUserDocument::EType::PassportRegistration,
        NUserDocument::EType::PassportSelfie,
    };
}

bool TDocumentsVerificationManager::CreateNewRegistrationAssignments(const TInstant since, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto presentAssignments = UserDocumentPhotosManager.GetDocumentVerificationAssignments().GetNotVerifiedRegistrationAssignments();
    INFO_LOG << "looking after photos since " << since.ToString() << Endl;
    INFO_LOG << "there are " << presentAssignments.size() << " assignments in the queue" << Endl;
    TSet<TString> presentAssignmentTraces;
    for (auto&& it : presentAssignments) {
        TStringStream ss;
        ss << it.second.GetLicenseBackId();
        ss << it.second.GetLicenseFrontId();
        ss << it.second.GetPassportBiographicalId();
        ss << it.second.GetPassportRegistrationId();
        ss << it.second.GetPassportSelfieId();
        presentAssignmentTraces.insert(ss.Str());
    }

    TMap<TString, TMap<NUserDocument::EType, TUserDocumentPhoto>> lastUploadedPhotos;
    TVector<TUserDocumentPhoto> uploadedPhotos = UserDocumentPhotosManager.GetUserPhotosDB().GetPhotosOfUsersWhichAddedPhotosRecently(since);
    INFO_LOG << "there are " << uploadedPhotos.size() << " photos which can possibly form an assignment" << Endl;

    Sort(uploadedPhotos.begin(), uploadedPhotos.end(), [](const TUserDocumentPhoto& lhs, const TUserDocumentPhoto& rhs){return lhs.GetSubmittedAt() < rhs.GetSubmittedAt();});

    for (auto&& photo : uploadedPhotos) {
        if (!NewRegistrationAssignmentPhotoTypes.contains(photo.GetType())) {
            continue;
        }
        auto userId = photo.GetUserId();
        auto photoType = photo.GetType();
        lastUploadedPhotos[userId][photoType] = photo;
    }
    INFO_LOG << "there are " << lastUploadedPhotos.size() << " users which uploaded photos recently" << Endl;

    TMap<TString, NJson::TJsonValue> metaMap;
    GetAssignmentsMeta(
        MakeSet(NContainer::Keys(lastUploadedPhotos)),
        metaMap
    );

    TVector<TYangDocumentVerificationAssignment> assignmentsToCreate;
    for (auto&& userIt : lastUploadedPhotos) {
        const TString& userId = userIt.first;
        if (userIt.second.size() != NewRegistrationAssignmentPhotoTypes.size()) {
            INFO_LOG << "CreateNewRegistrationAssignments: skip " << userId << " due to incorrect number of photos: " << userIt.second.size() << Endl;
            continue;
        }

        bool hasNotVerifiedPhoto = false;
        bool formsValidAssignment = true;
        for (auto&& photoTypeIt : userIt.second) {
            auto photoStatus = photoTypeIt.second.GetVerificationStatus();
            if (photoStatus == NUserDocument::EVerificationStatus::NotYetProcessed) {
                hasNotVerifiedPhoto = true;
            } else if (photoStatus != NUserDocument::EVerificationStatus::Ok) {
                formsValidAssignment = false;
                INFO_LOG << "CreateNewRegistrationAssignments: " << userId << " has bad photo " << photoTypeIt.first << ": " << photoStatus << Endl;
            }
        }

        if (!hasNotVerifiedPhoto || !formsValidAssignment) {
            INFO_LOG << "CreateNewRegistrationAssignments: skip " << userId << " due to photo statuses: " << userIt.second.size() << Endl;
            continue;
        }
        TString newAssignmentTrace;
        {
            TStringStream ss;
            ss << userIt.second[NUserDocument::EType::LicenseBack].GetId();
            ss << userIt.second[NUserDocument::EType::LicenseFront].GetId();
            ss << userIt.second[NUserDocument::EType::PassportBiographical].GetId();
            ss << userIt.second[NUserDocument::EType::PassportRegistration].GetId();
            ss << userIt.second[NUserDocument::EType::PassportSelfie].GetId();
            newAssignmentTrace = ss.Str();
        }
        if (presentAssignmentTraces.contains(newAssignmentTrace)) {
            continue;
        }

        TYangDocumentVerificationAssignment newAssignment(
            userIt.second[NUserDocument::EType::LicenseBack].GetId(),
            userIt.second[NUserDocument::EType::LicenseFront].GetId(),
            /*licenseSelfieId=*/{},
            userIt.second[NUserDocument::EType::PassportBiographical].GetId(),
            userIt.second[NUserDocument::EType::PassportRegistration].GetId(),
            userIt.second[NUserDocument::EType::PassportSelfie].GetId(),
            /*selfieId=*/{},
            /*licenseVideoId=*/{},
            /*passportVideoId=*/{},
            /*videoSelfieId=*/{},
            userIt.first
        );
        newAssignment.SetMeta(metaMap[userIt.first]);

        newAssignment.SetIsExperimental(Config.GetAreAssignmentsExperimental());
        assignmentsToCreate.emplace_back(std::move(newAssignment));
        DEBUG_LOG << "creating new assignment. UserId: " << userIt.first << " trace:" << newAssignmentTrace << Endl;
    }
    INFO_LOG << "there are " << assignmentsToCreate.size() << " assignments to create" << Endl;
    return RegisterNewAssignments(assignmentsToCreate, Config.GetRequestsPath(), Config.GetTempRequestsPath(), server, session);
}

bool TDocumentsVerificationManager::CreateNewSelfieAssignments(const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    auto presentAssignments = UserDocumentPhotosManager.GetDocumentVerificationAssignments().GetNotVerifiedSelfieAssignments();
    TSet<TString> selfieWithAssignments;
    for (auto&& it : presentAssignments) {
        selfieWithAssignments.insert(it.second.GetSelfieId());
    }
    auto preUnverifiedSelfies = UserDocumentPhotosManager.GetUserPhotosDB().GetUnverifiedPhotosOfType(NUserDocument::EType::Selfie);

    // Create assignments with selfie only
    TSet<TString> userIds;
    TVector<TYangDocumentVerificationAssignment> queuedAssignments;
    for (auto&& selfie : preUnverifiedSelfies) {
        if (selfieWithAssignments.contains(selfie.GetId())) {
            continue;
        }
        auto newAssignment = TYangDocumentVerificationAssignment(
            /*licenseBackId=*/{},
            /*licenseFrontId=*/{},
            /*licenseSelfieId=*/{},
            /*passportBiographicalId=*/{},
            /*passportRegistrationId=*/{},
            /*passportSelfieId=*/{},
            selfie.GetId(),
            /*licenseVideoId=*/{},
            /*passportVideoId=*/{},
            /*videoSelfieId=*/{},
            selfie.GetUserId()
        );
        queuedAssignments.push_back(std::move(newAssignment));
        userIds.insert(selfie.GetUserId());
    }

    if (userIds.empty()) {
        return true;
    }

    // Fetch passport selfie photos
    TMap<TString, TString> psForUser;
    auto psPhotos = UserDocumentPhotosManager.GetUserPhotosDB().GetTypeToRecentPhotoMapping(userIds, {NUserDocument::PassportSelfie});
    for (auto&& [userId, photoMapping] : psPhotos) {
        auto it = photoMapping.find(NUserDocument::PassportSelfie);
        if (it == photoMapping.end()) {
            continue;
        }
        psForUser[userId] = it->second.GetId();
    }

    TVector<TYangDocumentVerificationAssignment> assignmentsToCreate;
    for (auto&& as : queuedAssignments) {
        auto it = psForUser.find(as.GetUserId());
        if (it == psForUser.end()) {
            continue;
        }
        as.SetPassportSelfieId(it->second);
        assignmentsToCreate.emplace_back(std::move(as));
    }

    return RegisterNewAssignments(assignmentsToCreate, Config.GetRequestsPath(), Config.GetTempRequestsPath(), server, session);
}

bool TDocumentsVerificationManager::ResendToYang(const TSet<TString>& userIds, const TString& tagToRemove, const TString& operatorUserId, const NDrive::IServer* server) const {
    const TSet<NUserDocument::EType>& includeDocumentTypes = Config.GetDocumentTypes();
    TVector<TYangDocumentVerificationAssignment> assignmentsToCreate;
    TVector<NUserDocument::EType> soughtTypes = MakeVector(includeDocumentTypes);
    auto mapping = UserDocumentPhotosManager.GetUserPhotosDB().GetTypeToRecentPhotoMapping(MakeSet(userIds), soughtTypes);
    for (auto&& it : mapping) {
        if (it.second.size() != includeDocumentTypes.size()) {
            continue;
        }
        TMap<NUserDocument::EType, TString> typeIds;
        for (const auto& type : includeDocumentTypes) {
            auto photoIt = it.second.find(type);
            if (photoIt != it.second.end()) {
                typeIds[type] = photoIt->second.GetId();
            }
        }

        assignmentsToCreate.emplace_back(TYangDocumentVerificationAssignment(typeIds, it.first));
    }

    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    if (RegisterNewAssignments(assignmentsToCreate, Config.GetRequestsPath(), Config.GetTempRequestsPath(), server, session)) {
        if (tagToRemove && !RemoveAssignmentTags(assignmentsToCreate, tagToRemove, operatorUserId, server, session)) {
            return false;
        }
        return session.Commit();
    }
    return false;
}

bool TDocumentsVerificationManager::ResendToYangByPhotoIds(const TSet<TString>& photoIds, const TString& tagToRemove, const TString& operatorUserId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    TVector<TYangDocumentVerificationAssignment> assignmentsToCreate;
    auto mapping = UserDocumentPhotosManager.GetUserPhotosDB().GetUsersWithPhotoIds(photoIds, session);
    if (!mapping) {
        return false;
    }
    for (auto&& [userId, photos] : *mapping) {
        TMap<NUserDocument::EType, TString> typeIds;
        for (const auto& photo : photos) {
            typeIds[photo.GetType()] = photo.GetId();
        }
        assignmentsToCreate.emplace_back(TYangDocumentVerificationAssignment(typeIds, userId));
    }
    if (RegisterNewAssignments(assignmentsToCreate, Config.GetRequestsPath(), Config.GetTempRequestsPath(), server, session)) {
        if (tagToRemove && !RemoveAssignmentTags(assignmentsToCreate, tagToRemove, operatorUserId, server, session)) {
            return false;
        }
    }
    return true;
}

bool TDocumentsVerificationManager::RemoveAssignmentTags(const TVector<TYangDocumentVerificationAssignment>& assignments, const TString& tagToRemove, const TString& operatorUserId, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto& tagsManager = server->GetDriveAPI()->GetTagsManager();
    TSet<TString> userIds;
    for (auto&& assignment : assignments) {
        userIds.insert(assignment.GetUserId());
    }
    auto optionalTags = tagsManager.GetUserTags().RestoreTags(userIds, { tagToRemove }, session);
    if (!optionalTags) {
        return false;
    }
    if (!tagsManager.GetUserTags().RemoveTags(*optionalTags, operatorUserId, server, session)) {
        return false;
    }
    return true;
}

bool TDocumentsVerificationManager::GetAssignmentsMeta(const TSet<TString>& userIds, TMap<TString, NJson::TJsonValue>& metaMap) const {
    TMap<TString, TVector<TInstant>> userToVerificationTimes;

    auto assignments = UserDocumentPhotosManager.GetDocumentVerificationAssignments().GetAllAssignmentsForUsers(userIds);
    for (auto&& assignment : assignments) {
        auto userId = assignment.GetUserId();
        userToVerificationTimes[userId].push_back(assignment.GetCreatedAt());
    }

    for (auto&& it : userToVerificationTimes) {
        Sort(it.second.begin(), it.second.end());
        NJson::TJsonValue userResult = NJson::JSON_ARRAY;
        for (auto&& elem : it.second) {
            NJson::TJsonValue currentElement;
            currentElement["created_at"] = elem.ToString();
            userResult.AppendValue(std::move(currentElement));
        }
        metaMap[it.first] = std::move(userResult);
    }

    return true;
}

void TDocumentsVerificationManager::UpdatePhotoVerificationInfo(TUserDocumentPhoto& photo, const TString& receivedStatus) const {
    photo.SetVerifiedAt(TInstant::Now());
    NUserDocument::EVerificationStatus status;
    if (TryFromString(receivedStatus, status)
        && (status == NUserDocument::EVerificationStatus::Ok
        || status == NUserDocument::EVerificationStatus::Foreign
        || status == NUserDocument::EVerificationStatus::NeedInfo
        || status == NUserDocument::EVerificationStatus::Unrecognizable
        || status == NUserDocument::EVerificationStatus::NonLatin
        || status == NUserDocument::EVerificationStatus::Discarded)
    ) {
        photo.SetVerificationStatus(status);
    }
}
