#pragma once

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/tags/abstract.h>

#include <rtline/util/types/uuid.h>

namespace NDrivematics {
    struct TSignal {
    public:
        TUuid Id;
        TString Name;
        NEntityTagsManager::EEntityType Type;

        TUuid ObjectId;
        TUuid SessionId;
        TUuid UserId;

        TString Resolution;
        TInstant Start;
        TInstant Finish;
        TMaybe<bool> Visible;

    public:
        class TDecoder: public TBaseDecoder {
            R_FIELD(i32, Id, -1);
            R_FIELD(i32, Name, -1);
            R_FIELD(i32, Type, -1);
            R_FIELD(i32, ObjectId, -1);
            R_FIELD(i32, SessionId, -1);
            R_FIELD(i32, UserId, -1);
            R_FIELD(i32, Resolution, -1);
            R_FIELD(i32, Start, -1);
            R_FIELD(i32, Finish, -1);
            R_FIELD(i32, Visible, -1);

        public:
            TDecoder() = default;
            TDecoder(const TMap<TString, ui32>& decoderBase);
        };

    public:
        NSQL::TTableRecord SerializeToTableRecord() const;
        bool DeserializeWithDecoder(const TDecoder& decoder, TConstArrayRef<TStringBuf> values, const IHistoryContext* historyContext);
    };

    class TSignalManager: protected TDatabaseEntityManager<TSignal, true, TUuid> {
    private:
        using TBase = TDatabaseEntityManager<TSignal, true, TUuid>;

    public:
        struct TQueryOptions: public TBase::TQueryOptions {
            R_OPTIONAL(TSet<TString>, Ids);
            R_OPTIONAL(TSet<TString>, Names);
            R_OPTIONAL(TInstant, Since);
            R_OPTIONAL(TInstant, Until);
            R_OPTIONAL(bool, Actual);
            R_OPTIONAL(bool, Visible);

            R_OPTIONAL(TSet<TString>, ObjectIds);
            R_OPTIONAL(TSet<TString>, SessionIds);
            R_OPTIONAL(TSet<TString>, UserIds);

            R_READONLY(bool, IsOrObjectIds, false);
            R_READONLY(bool, IsOrSessionIds, false);
            R_READONLY(bool, IsOrUserIds, false);

        public:
            TQueryOptions& SetObjectIdsOr(auto&& value) {
                IsOrObjectIds = true;
                SetObjectIds(value);
                return *this;
            }
            TQueryOptions& SetSessionIdsOr(auto&& value) {
                IsOrSessionIds = true;
                SetSessionIds(value);
                return *this;
            }
            TQueryOptions& SetUserIdsOr(auto&& value) {
                IsOrUserIds = true;
                SetUserIds(value);
                return *this;
            }
        };

    public:
        TSignalManager(NSQL::IDatabase::TPtr database = nullptr)
            : TBase(std::move(database))
        {
        }

        TOptionalEntities Fetch(NDrive::TEntitySession& tx, TQueryOptions&& queryOptions) const;

        using TBase::Remove;
        using TBase::Upsert;

    protected:
        TUuid GetMainId(const TSignal& entity) const override {
            return entity.Id;
        }
        TString GetTableName() const override {
            return "signals";
        }
    };
}
