#include "tag.h"
#include "manager.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/drivematics/signal/proto/signal.pb.h>
#include <drive/backend/sessions/manager/billing.h>

#include <rtline/library/json/field.h>

#include <ranges>

bool NDrivematics::TSignalTagTraits::Fill(const TUuid& entityId, TInstant timestamp, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
    if (!Start) {
        Start = timestamp;
    }
    auto type = GetEntityType();
    if (!ObjectId && type == NEntityTagsManager::EEntityType::Car) {
        ObjectId = entityId;
    }
    if (!SessionId && type == NEntityTagsManager::EEntityType::Trace) {
        SessionId = entityId;
    }
    if (!UserId && type == NEntityTagsManager::EEntityType::User) {
        UserId = entityId;
        if (UserId == ObjectId) {
            ObjectId = TUuid();
        }

    }
    bool fillSessionId = ShouldFillSessionId();
    if (!SessionId && ObjectId && fillSessionId) {
        auto optionalSessions = server.GetDriveDatabase().GetSessionManager().GetObjectSessions(ObjectId, tx);
        if (!optionalSessions) {
            return false;
        }
        for (auto&& session : *optionalSessions) {
            if (!session) {
                continue;
            }
            if (Start < session->GetStartTS()) {
                continue;
            }

            if (!SetSessionId(TUuid::Parse(session->GetSessionId()), tx)) {
                return false;
            }
            if (!SetUserId(TUuid::Parse(session->GetUserId()), tx)) {
                return false;
            }
            break;
        }
    }
    if (!SessionId && ObjectId && fillSessionId) {
        const TString& objectId = ObjectId;
        auto ydbTx = NDrive::TEntitySession();
        auto optionalSessions = server.GetDriveDatabase().GetCompiledSessionManager().GetObjects<TMinimalCompiledRiding>({ objectId }, tx, ydbTx);
        if (!optionalSessions) {
            return false;
        }
        for (auto&& session : *optionalSessions | std::views::reverse) {
            if (Start < session.GetStartInstant().Get()) {
                continue;
            }

            if (!SetSessionId(TUuid::Parse(session.GetSessionId()), tx)) {
                return false;
            }
            if (!SetUserId(TUuid::Parse(session.GetHistoryUserId()), tx)) {
                return false;
            }
            break;
        }
    }
    if (!UserId && SessionId) {
        auto optionalSession = server.GetDriveDatabase().GetSessionManager().GetSession(SessionId, tx);
        if (!optionalSession) {
            return false;
        }
        auto session = *optionalSession;
        if (session) {
            if (!SetObjectId(TUuid::Parse(session->GetObjectId()), tx)) {
                return false;
            }
            if (!SetUserId(TUuid::Parse(session->GetUserId()), tx)) {
                return false;
            }
        }
    }
    if (!UserId && SessionId) {
        const TString& sessionId = SessionId;
        auto ydbTx = NDrive::TEntitySession();
        auto optionalSessions = server.GetDriveDatabase().GetCompiledSessionManager().Get<TMinimalCompiledRiding>({ sessionId }, tx, ydbTx);
        if (!optionalSessions) {
            return false;
        }
        for (auto&& session : *optionalSessions | std::views::reverse) {
            if (sessionId != session.GetSessionId()) {
                continue;
            }

            if (!SetObjectId(TUuid::Parse(session.GetObjectId()), tx)) {
                return false;
            }
            if (!SetUserId(TUuid::Parse(session.GetHistoryUserId()), tx)) {
                return false;
            }
            break;
        }
    }
    return true;
}

bool NDrivematics::TSignalTagTraits::SetFinish(TInstant value, NDrive::TEntitySession& tx) {
    if (Finish && Finish != value) {
        tx.SetErrorInfo("CommonSignalTag::SetFinish", TStringBuilder() << "mismatch: " << Finish << " " << value);
        return false;
    }
    Finish = value;
    return true;
}

bool NDrivematics::TSignalTagTraits::SetObjectId(TUuid&& value, NDrive::TEntitySession& tx) {
    if (ObjectId && ObjectId != value) {
        tx.SetErrorInfo("CommonSignalTag::SetObjectId", TStringBuilder() << "mismatch: " << ObjectId << " " << value);
        return false;
    }
    if (!ObjectId) {
        ObjectId = std::move(value);
    }
    return true;
}

bool NDrivematics::TSignalTagTraits::SetSessionId(TUuid&& value, NDrive::TEntitySession& tx) {
    if (SessionId && SessionId != value) {
        tx.SetErrorInfo("CommonSignalTag::SetSessionId", TStringBuilder() << "mismatch: " << SessionId << " " << value);
        return false;
    }
    if (!SessionId) {
        SessionId = std::move(value);
    }
    return true;
}

bool NDrivematics::TSignalTagTraits::SetUserId(TUuid&& value, NDrive::TEntitySession& tx) {
    if (UserId && UserId != value) {
        tx.SetErrorInfo("CommonSignalTag::SetUserId", TStringBuilder() << "mismatch: " << UserId << " " << value);
        return false;
    }
    if (!UserId) {
        UserId = std::move(value);
    }
    return true;
}

bool NDrivematics::TSignalTagTraits::RemoveSignal(const TConstDBTag& self, NDrive::TEntitySession& tx) const {
    TSignalManager manager;
    return manager.Remove(self.GetTagId(), tx);
}

bool NDrivematics::TSignalTagTraits::UpdateSignal(const TConstDBTag& self, NDrive::TEntitySession& tx) const {
    TSignal signal;
    signal.Id = TUuid::Parse(self.GetTagId());
    signal.Name = self->GetName();
    signal.Type = GetEntityType();

    signal.ObjectId = ObjectId;
    signal.SessionId = SessionId;
    signal.UserId = UserId;

    signal.Resolution = Resolution;
    signal.Start = Start;
    signal.Finish = Finish;
    signal.Visible = Visible;

    const auto& settings = NDrive::GetServer().GetSettings();
    const auto checkObjectId = settings.GetValue<bool>("signal.check_object_id").GetOrElse(true);
    if (signal.Type == NEntityTagsManager::EEntityType::User) {
        if (!signal.UserId && checkObjectId) {
            tx.SetErrorInfo("UpdateSignal", "user_id is not set");
            return false;
        }
    } else if (!signal.ObjectId && checkObjectId) {
        tx.SetErrorInfo("UpdateSignal", "object_id is not set");
        return false;
    }
    if (!signal.Start) {
        tx.SetErrorInfo("UpdateSignal", "start is not set");
        return false;
    }

    TSignalManager manager;
    return manager.Upsert(signal, tx);
}

bool NDrivematics::TSignalTagTraits::Deserialize(const NProto::TSignalTagTraits& proto) {
    if (proto.HasObjectId() && !ObjectId.Deserialize(proto.GetObjectId())) {
        return false;
    }
    if (proto.HasSessionId() && !SessionId.Deserialize(proto.GetSessionId())) {
        return false;
    }
    if (proto.HasUserId() && !UserId.Deserialize(proto.GetUserId())) {
        return false;
    }
    if (proto.HasResolution()) {
        Resolution = proto.GetResolution();
    }
    if (proto.HasStart()) {
        Start = TInstant::Seconds(proto.GetStart());
    }
    if (proto.HasFinish()) {
        Finish = TInstant::Seconds(proto.GetFinish());
    }
    if (proto.HasVisible()) {
        Visible = proto.GetVisible();
    }
    return true;
}

void NDrivematics::TSignalTagTraits::Serialize(NProto::TSignalTagTraits& proto) const {
    if (ObjectId) {
        ObjectId.Serialize(*proto.MutableObjectId());
    }
    if (SessionId) {
        SessionId.Serialize(*proto.MutableSessionId());
    }
    if (UserId) {
        UserId.Serialize(*proto.MutableUserId());
    }
    if (Resolution) {
        proto.SetResolution(Resolution);
    }
    if (Start) {
        proto.SetStart(Start.Seconds());
    }
    if (Finish) {
        proto.SetFinish(Finish.Seconds());
    }
    if (Visible) {
        proto.SetVisible(*Visible);
    }
}

bool NDrivematics::TCommonSignalTag::OnBeforeAdd(const TString& entityId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    Y_UNUSED(userId);
    auto timestamp = Now();
    auto entityUuid = TUuid::TryParse(entityId);
    if (!entityUuid) {
        tx.SetErrorInfo("CommonSignalTag::OnBeforeAdd", TStringBuilder() << "cannot parse " << entityId << " as UUID");
        return false;
    }
    return Fill(*entityUuid, timestamp, *server, tx);
}

bool NDrivematics::TCommonSignalTag::OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    Y_UNUSED(userId);
    Y_UNUSED(server);
    return UpdateSignal(self, tx);
}

bool NDrivematics::TCommonSignalTag::OnAfterUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const {
    Y_UNUSED(userId);
    Y_UNUSED(server);
    Y_UNUSED(to);
    return UpdateSignal(self, tx);
}

bool NDrivematics::TCommonSignalTag::OnBeforeEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, const TEvolutionContext* eContext) const {
    Y_UNUSED(to);
    Y_UNUSED(permissions);
    Y_UNUSED(server);
    Y_UNUSED(eContext);
    return RemoveSignal(from, tx);
}

bool NDrivematics::TCommonSignalTag::OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, const TEvolutionContext* eContext) const {
    Y_UNUSED(to);
    Y_UNUSED(permissions);
    Y_UNUSED(server);
    Y_UNUSED(eContext);
    auto tag = from;
    tag.SetData(to);
    return UpdateSignal(tag, tx);
}

bool NDrivematics::TCommonSignalTag::ProvideDataOnEvolve(const TDBTag& from, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    Y_UNUSED(permissions);
    Y_UNUSED(server);
    Y_UNUSED(tx);
    auto impl = from.GetTagAs<TCommonSignalTag>();
    if (impl) {
        MergeFields(*impl);
    }
    return true;
}

bool NDrivematics::TCommonSignalTag::OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) {
    Y_UNUSED(userId);
    Y_UNUSED(server);
    return RemoveSignal(self, tx);
}

void NDrivematics::TCommonSignalTag::SerializeSpecialDataToJson(NJson::TJsonValue& value) const {
    TBase::SerializeSpecialDataToJson(value);
    NJson::InsertNonNull(value, "object_id", GetObjectId());
    NJson::InsertField(value, "start", GetStart());
    NJson::InsertNonNull(value, "finish", GetFinish());
    NJson::InsertNonNull(value, "resolution", GetResolution());
    NJson::InsertNonNull(value, "session_id", GetSessionId());
    NJson::InsertNonNull(value, "user_id", GetUserId());
    NJson::InsertNonNull(value, "visible", GetVisible());
}

bool NDrivematics::TCommonSignalTag::DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
    return TBase::DoSpecialDataFromJson(value, errors)
        && NJson::TryFieldsFromJson(value, GetFields(), errors)
    ;
}
