#pragma once

#include <drive/backend/data/common/serializable.h>

#include <rtline/util/types/field.h>
#include <rtline/util/types/uuid.h>

namespace NDrivematics {
    namespace NProto {
        class TSignalTagTraits;
    }

    class TSignalTagTraits {
    public:
        virtual ~TSignalTagTraits() = default;

        virtual NJson::TJsonValue GetDetailReport() const {
            return NJson::JSON_NULL;
        }
        virtual TString GetSignalNameOverride() const {
            return {};
        }

        const TUuid& GetObjectId() const {
            return ObjectId;
        }
        const TUuid& GetSessionId() const {
            return SessionId;
        }
        const TUuid& GetUserId() const {
            return UserId;
        }
        const TString& GetResolution() const {
            return Resolution;
        }
        TInstant GetStart() const {
            return Start;
        }
        TInstant GetFinish() const {
            return Finish;
        }
        auto GetVisible() const {
            return Visible;
        }

        void SetResolution(const TString& value) {
            Resolution = value;
        }
        void SetVisible(bool value) {
            Visible = value;
        }

        bool RemoveSignal(const TConstDBTag& self, NDrive::TEntitySession& tx) const;
        bool UpdateSignal(const TConstDBTag& self, NDrive::TEntitySession& tx) const;

    protected:
        virtual NEntityTagsManager::EEntityType GetEntityType() const = 0;

        virtual bool ShouldFillSessionId() const {
            return false;
        }

        bool Fill(const TUuid& entityId, TInstant timestamp, const NDrive::IServer& server, NDrive::TEntitySession& tx);
        bool SetFinish(TInstant finish, NDrive::TEntitySession& tx);
        bool SetObjectId(TUuid&& value, NDrive::TEntitySession& tx);
        bool SetSessionId(TUuid&& value, NDrive::TEntitySession& tx);
        bool SetUserId(TUuid&& value, NDrive::TEntitySession& tx);

        bool Deserialize(const NProto::TSignalTagTraits& proto);
        void Serialize(NProto::TSignalTagTraits& proto) const;

    protected:
        DECLARE_FIELDS(
            Field(ObjectId, "object_id"),
            Field(SessionId, "session_id"),
            Field(UserId, "user_id"),
            Field(Resolution, "resolution"),
            Field(Start, "start"),
            Field(Finish, "finish"),
            Field(Visible, "visible")
        );

    private:
        TUuid ObjectId;
        TUuid SessionId;
        TUuid UserId;
        TString Resolution;
        TInstant Start;
        TInstant Finish;
        TMaybe<bool> Visible;
    };

    class TCommonSignalTag
        : public IJsonSerializableTag
        , public TSignalTagTraits
    {
    private:
        using TBase = IJsonSerializableTag;

    public:
        using TBase::TBase;

        TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return { GetEntityType() };
        }

        bool OnBeforeAdd(const TString& entityId, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) override;
        bool OnAfterAdd(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const override;
        bool OnAfterUpdate(const TDBTag& self, ITag::TPtr to, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) const override;

        bool OnBeforeEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, const TEvolutionContext* eContext) const override;
        bool OnAfterEvolve(const TDBTag& from, ITag::TPtr to, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx, const TEvolutionContext* eContext) const override;
        bool ProvideDataOnEvolve(const TDBTag& from, const TUserPermissions& permissions, const NDrive::IServer* server, NDrive::TEntitySession& tx) override;

        bool OnBeforeRemove(const TDBTag& self, const TString& userId, const NDrive::IServer* server, NDrive::TEntitySession& tx) override;

    protected:
        void SerializeSpecialDataToJson(NJson::TJsonValue& value) const override;
        bool DoSpecialDataFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) override;
    };
}
