#include "description.h"

#include <rtline/library/json/field.h>

template<>
NJson::TJsonValue NJson::ToJson(const NDrivematics::TSignalDescription::ESource& object) {
    return NJson::TMapBuilder("source", ToString(object));
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& json, NDrivematics::TSignalDescription::ESource& object) {
    return NJson::ParseField(json["source"], NJson::Stringify(object));
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrivematics::TSignalDescription::TSignalParams& object) {
    return NJson::TMapBuilder("parameters", object.SerializeToJson());
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& json, NDrivematics::TSignalDescription::TSignalParams& object) {
    return object.DeserializeFromJson(json["parameters"], nullptr);
}

DECLARE_FIELDS_JSON_SERIALIZER(NDrivematics::TSignalDescription);

NJson::TJsonValue NDrivematics::TSignalDescription::GetReport(ELocalization locale, const NDrive::IServer& server) const {
    auto localization = server.GetLocalization();
    auto NFDB = server.GetDriveAPI()->GetNamedFiltersDB();

    NJson::TJsonValue result;
    result["name"] = Name;
    result["type"] = Type;
    result["source"] = ToString(Source);
    result["display_name"] = localization ? localization->ApplyResources(DisplayName, locale) : DisplayName;
    result["is_editable"] = IsEditable;
    result["is_enabled"] = IsEnabled;
    result["is_permanent"] = IsPermanent;
    result["priority"] = ToString(Priority);

    auto fillNamedFilter = [&](const auto& namedFilterPath, const auto& namedFilterContainer) -> void {
        if (!namedFilterContainer.empty()) {
            auto& namedFilter = result["named_filters"];
            for (const auto& filterId : namedFilterContainer) {
                auto namedFilterOpt = NFDB->GetObject(filterId);
                namedFilter[namedFilterPath].InsertValue(
                    filterId,
                    namedFilterOpt ? namedFilterOpt->GetDisplayName() : ""
                );
            }
        }
    };
    if (HasCarsFilter()) {
        fillNamedFilter("include_dynamic_group", GetCarsFilterRef().GetIncludeDynamicFilters());
        fillNamedFilter("exclude_dynamic_group", GetCarsFilterRef().GetExcludeDynamicFilters());
    }
    if (HasParams()) {
        result["parameters"] = GetParamsRef().SerializeToJson();
    }
    return result;
}
