#pragma once

#include "signal_configurations.h"

namespace NDrivematics {
    class TSignalDescription {
    public:
        enum class ESource: ui8 {
            Telematics = 0 /* "telematics" */,
            SignalQ = 1 /* "signalQ" */,
            Driver = 2 /* "driver" */,
            Booking = 3 /* "booking" */,
        };

        struct TSignalParams {
        private:
            NJson::TJsonValue Values;

        public:
            TSignalParams() = default;

            void Add(NJson::TJsonValue newParam, TInstant since = TInstant::Zero()) {
                if (IsEmpty() || CheckIsNewValue(newParam)) {
                    if (since != TInstant::Zero()) {
                        newParam["since"] = NJson::ToJson(since.Seconds());
                    }
                    Values.AppendValue(newParam);
                } else {
                    auto& lastValue = Values.GetArraySafe().back();
                    if (since != TInstant::Zero() && (!lastValue.Has("since") || lastValue["since"].GetUInteger() > since.Seconds())) {
                        lastValue["since"] = NJson::ToJson(since.Seconds());
                    }
                }
            }

            bool IsEmpty() const {
                return Values.GetArray().empty();
            }

            bool CheckIsNewValue(const NJson::TJsonValue& newParam) const {
                auto lastValue = Values.GetArray().back();
                lastValue.GetMapSafe().erase("since");
                if (lastValue == newParam) {
                    return false;
                }
                return true;
            }

            const NJson::TJsonValue& SerializeToJson() const {
                return Values;
            }

            bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
                if (json.IsArray()) {
                    Values = json;
                    return true;
                }
                errors ? errors->AddMessage("DeserializeFromJson", "json must be array") : void();
                return false;
            }
        };

    public:
        R_FIELD(TString, Name);
        R_FIELD(TString, Type);
        R_FIELD(ESource, Source);
        R_FIELD(NDrivematics::ISignalConfiguration::ESignalPriority, Priority);
        R_FIELD(TString, DisplayName);
        R_FIELD(bool, IsEnabled, true);
        R_FIELD(bool, IsEditable, true);
        R_FIELD(bool, IsPermanent, false);
        R_OPTIONAL(TSignalParams, Params);
        R_OPTIONAL(TCarsFilter, CarsFilter);
        R_OPTIONAL(TDuration, ActivationThreshold);

    public:
        DECLARE_FIELDS(
            Field(Name, "name"),
            Field(Type, "type"),
            Field(NJson::Stringify(Source), "source"),
            Field(CarsFilter, "cars_filter"),
            Field(NJson::Stringify(Priority), "priority"),
            Field(DisplayName, "display_name"),
            Field(IsEnabled, "is_enabled"),
            Field(IsEditable, "is_editable"),
            Field(Params, "parameters"),
            Field(IsPermanent, "is_permanent"),
            Field(NJson::Seconds(ActivationThreshold), "activation_threshold")
        );

    public:
        NJson::TJsonValue GetReport(ELocalization locale, const NDrive::IServer& server) const;
    };
}
