#include "signal_configurations.h"
#include "tags.h"

#include <drive/backend/alerts/data/location.h>
#include <drive/backend/alerts/data/history_rides.h>
#include <drive/backend/alerts/types/alert_types.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/consistency/consistency.h>

#include <rtline/library/json/field.h>

namespace {
    ui64 MinimalDelayInAcceptanceDelayedSignal = 59;
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrivematics::ISignalConfiguration& conf) {
    return conf.SerializeToJson();
}

template<>
NJson::TJsonValue NJson::ToJson(const NDrivematics::ISignalConfiguration::TZoneContainer& zoneContainer) {
    return zoneContainer.SerializeToJson();
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrivematics::ISignalConfiguration::TZoneContainer& zoneContainer) {
    return zoneContainer.DeserializeFromJson(value);
}

TOptionalObjectEvents<TSignalConfigurationDB> TSignalsConfigurationsHistoryManager::GetEvents(
    TRange<TEventId>&& idRange,
    TRange<TInstant>&& timestampRange,
    NDrive::TEntitySession& session,
    TQueryOptions&& queryOptions
) const {
    auto transaction = session.GetTransaction();
    TQueryOptions::TBase options(queryOptions);
    TStringBuilder customCondition;
    customCondition << "True";
    if (queryOptions.HasActions()) {
        auto actions = TSet<TString>();
        for (auto&& action : queryOptions.GetActionsRef()) {
            actions.insert(ToString(action));
        }
        customCondition << " AND " << queryOptions.PrintCondition("history_action", std::move(actions), *transaction);
    }

    if (queryOptions.GetSignalIds()) {
        customCondition << " AND " << queryOptions.PrintCondition("id", queryOptions.GetSignalIds().BuildCondition(), *transaction);
    }
    if (queryOptions.GetOwner()) {
        customCondition << " AND owner = " << transaction->Quote(queryOptions.GetOwner());
    }
    if (queryOptions.GetTypes()) {
        customCondition << " AND " << queryOptions.PrintCondition("type", queryOptions.GetTypes().BuildCondition(), *transaction);
    }

    options.AddCustomCondition(customCondition);
    return TBase::GetEvents(idRange, timestampRange, session, options);
}

[[nodiscard]] TMaybe<TVector<TSignalConfigurationDB>> TSignalsConfigurationsStorage::GetObjects(const TTagDescription::TConstPtr companyTagDescription, NDrive::TEntitySession& session) const {
    if (!companyTagDescription) {
        return Default<TMaybe<TVector<TSignalConfigurationDB>>>();
    }
    return TBase::GetObjectsByField(companyTagDescription->GetName(), "owner", session);
}

[[nodiscard]] bool TSignalsConfigurationsStorage::UpsertObject(const TSignalConfigurationDB& object, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TSignalConfigurationDB>* /*containerExt*/) const {
    if (!object) {
        session.SetErrorInfo("incorrect user data", "upsert object into " + TSignalConfigurationDB::GetTableName(), EDriveSessionResult::DataCorrupted);
        return false;
    }

    NStorage::TTableRecord trUpdate = object.SerializeToTableRecord();
    NStorage::TTableRecord trCondition = TSignalsConfigurationsConditionConstructor::BuildCondition(object);

    auto table = HistoryManager->GetDatabase().GetTable(TSignalConfigurationDB::GetTableName());

    NStorage::TObjectRecordsSet<TSignalConfigurationDB> records;
    bool isUpdate = false;
    switch (table->UpsertWithRevision(trUpdate, session.GetTransaction(), trCondition, object.OptionalRevision(), "revision", &records)) {
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::IncorrectRevision:
            session.SetErrorInfo("upsert_object", "incorect_revision", EDriveSessionResult::InconsistencyUser);
            return false;
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::Failed:
            session.SetErrorInfo("upsert_object", "UpsertWithRevision failure", EDriveSessionResult::TransactionProblem);
            return false;
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::Updated:
            isUpdate = true;
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::Inserted:
            break;
    }

    if (!HistoryManager->AddHistory(records.GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
        return false;
    }

    auto* api = NDrive::GetServerAs<NDrive::IServer>().GetDriveAPI();
    TVector<TConsistency> references;
    for (const auto& object : records.GetObjects()) {
        auto temp = object.GetReferences();
        references.insert(references.end(), temp.begin(), temp.end());
    }
    if (!api->GetConsistencyDB().UpdateConnectivity(references, session)) {
        return false;
    }
    return true;
}


namespace NDrivematics {

    ISignalConfiguration::TPtr ISignalConfiguration::ConstructConfiguration(const NJson::TJsonValue& json, TMessagesCollector* errors,  const NDrive::IServer& server) {
        ISignalConfiguration::TPtr configuration = TFactory::Construct(json["type"].GetString());
        if (!configuration) {
            errors ? errors->AddMessage("ISignalConfiguration::ConstructConfiguration", TStringBuilder() << "can't create signal configuration of type '" << json["type"].GetString() << "'") : void();
            return nullptr;
        }
        if (!configuration->DeserializeFromJson(json, errors) || !configuration->Verify(server, errors)) {
            return nullptr;
        }
        return configuration;
    }

    bool ISignalConfiguration::DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
        auto result = NJson::TryFieldsFromJson(json, GetFields(), errors);
        if (!result) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue ISignalConfiguration::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::FieldsToJson(result, GetFields());
        return result;
    }

    NJson::TJsonValue IRobotSetting::GetBaseSettings(const TString bpName) const {
        NJson::TJsonValue containerJson;
        R_ENSURE(bpName, {}, "empty bpName");
        containerJson["bp_name"] = bpName;
        containerJson["bp_type"] = "common_alerts";
        return containerJson;
    }

    TRTBackgroundProcessContainer ISignalConfiguration::ConstructBackgroundProcess() const {
        const auto* settings = dynamic_cast<const IRobotSetting*>(this);
        Y_ENSURE(settings);
        auto containerJson = settings->ConstructBackgroundSetting();
        auto& settingsJson = containerJson["bp_settings"];
        Y_ENSURE(FilterProcess(settingsJson));
        TRTBackgroundProcessContainer container;
        Y_ENSURE(TBaseDecoder::DeserializeFromJson(container, containerJson));
        return container;
    }

    TVector<TRTBackgroundProcessContainer> ISignalConfiguration::ConstructBackgroundProcesses() const {
        return { ConstructBackgroundProcess() };
    }

    bool ISignalConfiguration::FilterProcess(NJson::TJsonValue& settings) const {
        {
            auto& fetchers = settings["fetchers"];
            if (!fetchers.IsArray() && fetchers.IsDefined()) {
                return false;
            }
            uint maxIndex = 0;
            for (uint i = 0; i < fetchers.GetArray().size(); i++) {
                if (NJson::TJsonValue temp; fetchers.GetArray()[i].GetValueByPath("config.index", temp)) {
                    if (temp.GetUInteger() > maxIndex) {
                        maxIndex = temp.GetUInteger();
                    }
                    continue;
                }
                return false;
            }
            auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
            fetcherConfig["filter"] = GetCarTagsFilters();
            fetcherConfig["index"] = maxIndex + 1;
            fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
        }
        {
            auto& checkers = settings["checkers"];
            auto& checker = checkers.AppendValue(NJson::JSON_MAP);
            checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ECarTags) + "." + ToString(NAlerts::EFetchedItems::Main);
            checker["min_value"] = ToString(1);
            checker["max_value"] = ToString(2);
            checker["skip_empty"] = false;
            checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Main);
        }
        return true;
    }

    bool ISignalConfiguration::CreateTags(const TString&  /*authorId*/, const NDrive::IServer&  /*server*/, NDrive::TEntitySession&  /*tx*/) const {
        return true;
    }

    TVector<TString> ISignalConfiguration::GetBackgroundProcessesNames() const {
        return { "robot_" + GetSignalId() };
    }

    TString ISignalConfiguration::GetSignalName() const {
        return "signal_" + GetSignalId();
    }

    NJson::TJsonValue ISignalConfiguration::GetCarTagsFilters() const {
        NJson::TJsonValue filter;
        TString includeTagsFilterString = THasTelematicsTag::TypeName;
        if (HasCarsFilter()) {
            filter = GetCarsFilterRef().SerializeToJson();
            if (GetCarsFilterRef().HasIncludeTagsFilter()) {
                includeTagsFilterString += "*" + GetCarsFilterRef().GetIncludeTagsFilterRef().ToString();
            }
        }
        filter["include_tags_filter"] = includeTagsFilterString;
        return filter;
    }

    TString ISignalConfiguration::GetDescription(TString typeName) const {
        return GetOwnerName() + ":" + typeName + ":" + GetDisplayName();
    }

    bool ISignalConfiguration::TZoneContainer::DeserializeFromJson(const NJson::TJsonValue& json) {
        if (!NJson::ParseField(json["included_group_ids"], OptionalIncluded())
            || !NJson::ParseField(json["excluded_group_ids"], OptionalExcluded())) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue ISignalConfiguration::TZoneContainer::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertNonNull(result, "included_group_ids", Included);
        NJson::InsertNonNull(result, "excluded_group_ids", Excluded);
        return result;
    }

    bool TSignalConfigurationDB::DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
        NStorage::TTableRecord record;
        if (!record.DeserializeFromJson(json)) {
            errors ? errors->AddMessage("DeserializeFromJson", "cannot deserialize json: " + json.GetStringRobust()) : void();
            return false;
        }
        return DeserializeFromTableRecord(record, nullptr);
    }

    NJson::TJsonValue TSignalConfigurationDB::SerializeToJson() const {
        return SerializeToTableRecord().SerializeToJson();
    }

    bool TSignalConfigurationDB::DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*context*/) {
        SetId(record.Get("id"));
        SetType(record.Get("type"));
        TryFromString(record.Get("revision"), MutableRevision());
        Data = TFactory::Construct(Type);

        NJson::TJsonValue meta;
        if (!NJson::ReadJsonFastTree(record.Get("meta"), &meta) || !Data->DeserializeFromJson(meta)) {
            return false;
        }
        return true;
    }

    NStorage::TTableRecord TSignalConfigurationDB::SerializeToTableRecord() const {
        NStorage::TTableRecord record;
        Y_ENSURE(Data);
        record.Set("id", GetId());
        record.Set("owner", Data->GetOwnerId());
        record.Set("type", Data->GetType());
        if (HasRevision()) {
            record.Set("revision", GetRevision());
        }
        record.Set("meta", Data->SerializeToJson());
        return record;
    }

    TSignalConfigurationDB::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
        Id = GetFieldDecodeIndex("id", decoderBase);
        Type = GetFieldDecodeIndex("type", decoderBase);
        Revision = GetFieldDecodeIndex("revision", decoderBase);
        Meta = GetFieldDecodeIndex("meta", decoderBase);
    }

    bool TSignalConfigurationDB::DeserializeWithDecoderVerbose(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, TMessagesCollector& errors, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, Type);
        Data = TFactory::Construct(Type);
        Y_ENSURE(Data);

        READ_DECODER_VALUE(decoder, values, Id);
        READ_DECODER_VALUE(decoder, values, Revision);

        NJson::TJsonValue meta;
        READ_DECODER_VALUE_JSON(decoder, values, meta, Meta);
        if (!Data->DeserializeFromJson(meta, &errors)) {
            return false;
        }

        if (GetId() == "uuid_generate_v4()" || !HasRevision()) {
            errors.AddMessage("DeserializeWithDecoderVerbose", "some value uninitialized");
            return false;
        }

        return true;
    }

    TVector<TConsistency> TSignalConfigurationDB::GetReferences() const {
        TVector<TConsistency> result;
        auto& data = Yensured(Data);
        ui32 maxSize = 0;

        TVector<TString> namedFilters;
        if (data->HasCarsFilter()) {
            auto& exclude = data->GetCarsFilterRef().GetExcludeDynamicFilters();
            namedFilters.insert(namedFilters.end(), exclude.begin(), exclude.end());
            auto& include = data->GetCarsFilterRef().GetIncludeDynamicFilters();
            namedFilters.insert(namedFilters.end(), include.begin(), include.end());
            maxSize = maxSize > namedFilters.size() ? maxSize : namedFilters.size();
        }
        TVector<TString> zones;
        if (data->HasZoneContainer()) {
            auto& zoneRef = data->GetZoneContainerRef();
            if (zoneRef.HasExcluded()) {
                auto& exclude = zoneRef.GetExcludedRef();
                zones.insert(zones.end(), exclude.begin(), exclude.end());
            }
            if (zoneRef.HasIncluded()) {
                auto& include = zoneRef.GetIncludedRef();
                zones.insert(zones.end(), include.begin(), include.end());
            }
            maxSize = maxSize > zones.size() ? maxSize : zones.size();
        }
        result.resize(maxSize);

        for (ui32 idx = 0; idx < result.size(); idx++) {
            auto& current = result[idx];
            current.MutableId().clear();
            current.SetEntityName(TSignalConfigurationDB::GetTableName());
            current.SetReferenceObjectId(GetId());
            if (namedFilters.size() > idx) {
                current.SetNamedFilterId(namedFilters[idx]);
            }
            if (zones.size() > idx) {
                current.SetZoneId(zones[idx]);
            }
        }
        return result;
    }


    class TSpeedingSignalConfiguration final : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "1m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Session);
            return settings;
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["cars_filter"] = GetCarTagsFilters();
                fetcherConfig["index"] = 0;
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ETraceTagsHistory);
                fetcherConfig["query"] = NJson::TMapBuilder
                    (
                        "tag_names", NJson::TArrayBuilder(SpeedingTraceTagName)
                    )
                    (
                        "actions", NJson::TArrayBuilder
                            (ToString(EObjectHistoryAction::Add))
                            (ToString(EObjectHistoryAction::UpdateData))
                    )
                ;
                fetcherConfig["trace_filter"] = TTagsFilter::BuildFromString(SpeedingTraceTagName).ToString();
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            auto& checker = checkers.AppendValue(NJson::JSON_MAP);
            checker["meta"] =
                NJson::TMapBuilder
                    ("tag_name", SpeedingTraceTagName)
                    ("field_path", SpeedingValueFieldName)
                    ("precision", 0)
                ;
            checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ETraceTagsHistory);
            checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ETraceTagsHistory) + "." + ToString(NAlerts::EFetchedItems::TagFieldValue);
            checker["min_value"] = ToString(SpeedThreshold);
            checker["max_value"] = ToString(100000);
            checker["skip_empty"] = false;
            checker["iterator_type"] = ToString(NAlerts::EFetchedItems::TagFieldValue);
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = false;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ETraceTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TScoringTraceTag::TDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TScoringTraceTag::TypeName);
            tagDescription->SetKind(IScoringBaseTag::EScoringKind::Speeding);
            tagDescription->SetSignalEnabled(true);
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        bool FilterProcess(NJson::TJsonValue&  /*settings*/) const override {
            return true;
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString SpeedingTraceTagName{"speeding_trace_tag"};
        static inline const TString SpeedingValueFieldName{"value"};
        static inline const TString TypeName{"speeding_signal"};
        static inline const TFactory::TRegistrator<TSpeedingSignalConfiguration> Registrator{TypeName};

        ui64 SpeedThreshold;

    public:
        DECLARE_FIELDS(
            Field(SpeedThreshold, "speed_threshold", true)
        );

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return TBase::DeserializeFromJson(json, errors) &&
                NJson::TryFieldsFromJson(json, GetFields(), errors);
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            NJson::MergeJson(SerializeParamsToJson(), result);
            return result;
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TTelematicsLagSignalConfiguration final : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "1m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }

        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
            fetcherConfig["filter"] = GetCarTagsFilters();
            fetcherConfig["index"] = 0;
            fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            auto& checker = checkers.AppendValue(NJson::JSON_MAP);
            checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ECarTags) + "." + ToString(NAlerts::EFetchedItems::Lag);
            checker["min_value"] = ToString(GetActivationThresholdRef().Seconds());
            checker["max_value"] = ToString(100000000000);
            checker["skip_empty"] = false;
            checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Lag);
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = true;
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue&  /*settings*/) const override {
            return true;
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"telematics_lag_signal"};
        static inline const TFactory::TRegistrator<TTelematicsLagSignalConfiguration> Registrator{TypeName};

    public:

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            if (!TBase::DeserializeFromJson(json, errors)) {
                return false;
            }
            if (!HasActivationThreshold()) {
                errors ? errors->AddMessage("TTelematicsLagSignalConfiguration::DeserializeFromJson", "missing required field: 'activation_threshold'") : void();
                return false;
            }
            return true;
        }

        NJson::TJsonValue SerializeToJson() const override {
            return TBase::SerializeToJson();
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TBatteryLowSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "1m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }
        TString GetSQLRequest() const {
            return "SELECT object_id FROM "
            "(SELECT object_id, arrayReduce('max',arraySlice(groupArray(value_float) as `values`, -10, 10)) as `value` "
            "FROM (SELECT object_id, timestamp, value_float FROM drive_sensor_history WHERE (id = " + ToString(VEGA_POWER_VOLTAGE) + ") AND timestamp > now() - toIntervalDay(30) "
            "ORDER BY timestamp ASC) GROUP BY object_id) WHERE value < " + ToString(MaxVoltage) +  " AND value > " + ToString(MinVoltage);
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
            fetcherConfig["index"] = 0;
            fetcherConfig["query"] = GetSQLRequest();
            fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            auto& checker = checkers.AppendValue(NJson::JSON_MAP);
            checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
            checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds) + "." + ToString(NAlerts::EFetchedItems::Main);
            checker["min_value"] = "1";
            checker["max_value"] = "2";
            checker["skip_empty"] = false;
            checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Main);
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = true;
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& settings) const override {
            return TBase::FilterProcess(settings);
        }

    private:
        virtual TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"car_battery_low_signal"};
        static inline const TFactory::TRegistrator<TBatteryLowSignalConfiguration> Registrator{TypeName};

        R_FIELD(double, MinVoltage, 4.);
        R_FIELD(double, MaxVoltage, 10.);

    public:
        DECLARE_FIELDS(
            Field(MinVoltage, "min_voltage", true),
            Field(MaxVoltage, "max_voltage", true)
        );

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            if (!TBase::DeserializeFromJson(json, errors) || !NJson::TryFieldsFromJson(json, GetFields(), errors)) {
                return false;
            }
            if (MinVoltage < 0.) {
                errors ? errors->AddMessage("DeserializeFromJson", "min_voltage < 0, too low") : void();
                return false;
            }
            if (MaxVoltage < 0.) {
                errors ? errors->AddMessage("DeserializeFromJson", "max_voltage < 0, too low") : void();
                return false;
            }
            if (MinVoltage >= MaxVoltage) {
                errors ? errors->AddMessage("DeserializeFromJson", "max_voltage < min_voltage") : void();
                return false;
            }
            return true;
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            NJson::MergeJson(SerializeParamsToJson(), result);
            return result;
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TBatteryDiedSignalConfiguration final : public TBatteryLowSignalConfiguration {
    private:
        using TBase = TBatteryLowSignalConfiguration;

    public:
        TBatteryDiedSignalConfiguration()
            : TBase()
        {
            SetMinVoltage(0.);
            SetMaxVoltage(4.);
        }
    private:
        TString GetTypeName() const override {
            return TypeName;
        }
        static inline const TString TypeName{"car_battery_died_signal"};
        static inline const TFactory::TRegistrator<TBatteryDiedSignalConfiguration> Registrator{TypeName};
    };

    class TInZoneSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    protected:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "1m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
            fetcherConfig["filter"] = GetCarTagsFilters();
            fetcherConfig["index"] = 0;
            fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            auto& checker = checkers.AppendValue(NJson::JSON_MAP);
            checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ECarTags) + "." + ToString(NAlerts::EFetchedItems::Location);
            checker["min_value"] = "1";
            checker["max_value"] = "2";
            checker["skip_empty"] = false;
            checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Location);
            R_ENSURE(HasZoneContainer(), HTTP_INTERNAL_SERVER_ERROR, "zone configs empty");
            checker["meta"] = GetZoneContainerRef().SerializeToJson();

            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = true;
            filter["exclude_tags_filter"] = filter.Has("exclude_tags_filter")
                ? filter["exclude_tags_filter"].GetString() + "*(installation,new_car)"
                : "*(installation,new_car)";
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& /*settings*/) const override {
            return true;
        }

    private:
        virtual TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"car_in_zone_signal"};
        static inline const TFactory::TRegistrator<TInZoneSignalConfiguration> Registrator{TypeName};

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            if (!TBase::DeserializeFromJson(json, errors)) {
                return false;
            }
            if (!HasZoneContainer() || GetZoneContainerRef().IsEmpty()) {
                errors ? errors->AddMessage("DeserializeFromJson", "missing required field: 'zone'") : void();
                return false;
            }
            return true;
        }

        NJson::TJsonValue SerializeToJson() const override {
            return TBase::SerializeToJson();
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TOutZoneSignalConfiguration final : public TInZoneSignalConfiguration {
    private:
        using TBase = TInZoneSignalConfiguration;
    public:
        using TBase::TBase;

    private:
        TString GetTypeName() const override {
            return TypeName;
        }
        static inline const TString TypeName{"car_out_zone_signal"};
        static inline const TFactory::TRegistrator<TOutZoneSignalConfiguration> Registrator{TypeName};
    };

    class TOverMileageSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "5m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }
        TString GetSQLRequest() const {
            return "SELECT object_id FROM "
                "(SELECT object_id, arrayElement(groupArray(value_float),-1) - arrayElement(groupArray(value_float),1) as mileage "
                "FROM (SELECT object_id, value_float FROM drive_sensor_history WHERE (id = " + ToString(CAN_ODOMETER_KM) + ") "
                "AND timestamp > now() - toIntervalDay(" + ToString(IntervalDays) + ") ORDER BY timestamp) "
                "GROUP BY object_id) WHERE mileage > " + ToString(Mileage);
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["query"] = GetSQLRequest();
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
            }
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 1;
                fetcherConfig["filter"] = GetCarTagsFilters();
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds) + "." + ToString(NAlerts::EFetchedItems::Main);
                checker["min_value"] = "1";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Main);
            }
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ECarTags) + "." + ToString(NAlerts::EFetchedItems::LastActionDelta);
                checker["min_value"] = ToString(IntervalDays) + "d";
                checker["max_value"] = "inf";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::LastActionDelta);
                auto& meta = checker["meta"];
                meta.InsertValue("history_tags", NJson::JSON_ARRAY).AppendValue(GetSignalName());
                meta.InsertValue("history_actions", NJson::JSON_ARRAY).AppendValue(ToString(EObjectHistoryAction::Add));
                meta.InsertValue("history_deep", ToString(IntervalDays + 1) + "d");

            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto addActions = [&actionsDesc, &signalName, timeZone = TimeZone](const EObjectHistoryAction& historyAction) -> void {
                auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
                NJson::TJsonValue value = NJson::TMapBuilder
                    ("tag", signalName)
                ;
                action["tag_data"] = value.GetStringRobust();
                action["max_tags"] = 500;
                action["auto_remove"] = false;
                action["save_fetched"] = false;
                action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
                action["tag_action"] = ToString(historyAction);
                action["tag_name"] = signalName;
                action["timezone"] = timeZone;
            };
            addActions(EObjectHistoryAction::Add);
            addActions(EObjectHistoryAction::Remove);
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& /*settings*/) const override {
            return true;
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"car_over_mileage_signal"};
        static inline const TFactory::TRegistrator<TOverMileageSignalConfiguration> Registrator{TypeName};

        R_FIELD(double, Mileage, 100);
        R_FIELD(ui32, IntervalDays, 12);

    public:
        DECLARE_FIELDS(
            Field(Mileage, "mileage", true),
            Field(IntervalDays, "interval_days", true)
        );

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            if (!TBase::DeserializeFromJson(json, errors) || !NJson::TryFieldsFromJson(json, GetFields(), errors)) {
                return false;
            }
            if (Mileage < 10) {
                errors ? errors->AddMessage("DeserializeFromJson", "mileage < 10, too low") : void();
                return false;
            }
            if (IntervalDays < 1) {
                errors ? errors->AddMessage("DeserializeFromJson", "interval_days < 1, too low") : void();
                return false;
            }
            return true;
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            NJson::MergeJson(SerializeParamsToJson(), result);
            return result;
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TChangedMileageSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "5m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }
        TString GetSQLRequest() const {
            return "SELECT object_id FROM (SELECT object_id, arrayReduce('min', arrayDifference(groupArray(value_float))) as `min_value_float`, "
                "arrayReduce('max', arrayDifference(groupArray(value_float))) as `max_value_float` "
                "FROM (SELECT object_id, value_float FROM(SELECT object_id, value_float, timestamp FROM  drive_sensor_history WHERE (id = " + ToString(CAN_ODOMETER_KM) + ") AND timestamp > now() - toIntervalDay(7)) "
                "ORDER BY timestamp) GROUP BY object_id) WHERE min_value_float < -10.0 AND max_value_float < 10000";
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["query"] = GetSQLRequest();
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds) + "." + ToString(NAlerts::EFetchedItems::Main);
                checker["min_value"] = "1";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Main);
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = false;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& settings) const override {
            return TBase::FilterProcess(settings);
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"car_changed_mileage_signal"};
        static inline const TFactory::TRegistrator<TChangedMileageSignalConfiguration> Registrator{TypeName};

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return TBase::DeserializeFromJson(json, errors);
        }

        NJson::TJsonValue SerializeToJson() const override {
            return TBase::SerializeToJson();
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TWarningDTCSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "5m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }
        TString GetSQLRequest() const {
            auto dtcInserter = [&dtc = DTCs]() -> TString {
                TStringStream ss;
                for (const auto& el : dtc) {
                    ss << '\'' << el << "\',";
                }
                return {ss.Data(), ss.Size() - 1};
            }();
            return "WITH sensors_data AS (SELECT object_id, splitByWhitespace(argMax(value_string, timestamp)) AS `error_array` "
                "FROM drive_sensor_history WHERE id = " + ToString(VEGA_ECM_DTC_LIST_SENSOR) + " AND timestamp > now() - toIntervalDay(30) GROUP BY object_id) "
                "SELECT object_id FROM sensors_data WHERE NOT empty(arrayIntersect(error_array, "
                "[" + dtcInserter + "]))";
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["query"] = GetSQLRequest();
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds) + "." + ToString(NAlerts::EFetchedItems::Main);
                checker["min_value"] = "1";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Main);
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& settings) const override {
            return TBase::FilterProcess(settings);
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"warning_dtc_error_signal"};
        static inline const TFactory::TRegistrator<TWarningDTCSignalConfiguration> Registrator{TypeName};

        R_FIELD(TVector<TString>, DTCs);

    public:
        DECLARE_FIELDS(
            Field(DTCs, "dtc", true)
        );

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return TBase::DeserializeFromJson(json, errors) &&
                NJson::TryFieldsFromJson(json, GetFields(), errors);
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            NJson::MergeJson(SerializeParamsToJson(), result);
            return result;
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }

    private:
        bool Verify(const NDrive::IServer& server, TMessagesCollector* errors) const override {
            auto allDTCs = MakeSet(
                StringSplitter(
                    server.GetSettings().GetValue<TString>("signals.params.dtc_error_verify").GetOrElse("")
                ).SplitBySet(",").SkipEmpty().ToList<TString>()
            );
            for (const auto& el : GetDTCs()) {
                if (!allDTCs.contains(el.c_str()) ) {
                    errors ? errors->AddMessage("Verify", "unknown code: " + el) : void();
                    return false;
                }
            }
            return true;
        }
    };

    class IDashboardSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    protected:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "5m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }

        virtual TString GetSQLRequest() const = 0;

        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["query"] = GetSQLRequest();
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds) + "." + ToString(NAlerts::EFetchedItems::Main);
                checker["min_value"] = "1";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Main);
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& settings) const override {
            return TBase::FilterProcess(settings);
        }

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return TBase::DeserializeFromJson(json, errors);
        }

        NJson::TJsonValue SerializeToJson() const override {
            return TBase::SerializeToJson();
        }

        virtual TString GetTypeName() const = 0;

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TDashboardCheckEngineSignalConfiguration : public IDashboardSignalConfiguration {
    protected:
        using TBase = IDashboardSignalConfiguration;

    private:
        TString GetSQLRequest() const override {
            return "SELECT object_id FROM (SELECT object_id, arrayReduce('median', arraySlice(groupArray(value_float),-10,10)) as `value` "
                "FROM (SELECT object_id, timestamp, value_float FROM drive_sensor_history WHERE (id = " + ToString(CAN_CHECK_ENGINE) + ") AND (subid = " + ToString(CAN_ENGINE_IS_ON) + ") AND timestamp > now() - toIntervalDay(30) "
                "ORDER BY timestamp) GROUP BY object_id) WHERE value != 0";
        }

    public:
        TString GetTypeName() const override {
            return TypeName;
        }

    private:
        static inline const TString TypeName{"dashboard_check_engine_signal"};
        static inline const TFactory::TRegistrator<TDashboardCheckEngineSignalConfiguration> Registrator{TypeName};
    };

    class TDashboardCheckAirbagSignalConfiguration : public IDashboardSignalConfiguration {
    protected:
        using TBase = IDashboardSignalConfiguration;

    private:
        TString GetSQLRequest() const override {
            return "SELECT object_id FROM (SELECT object_id, arrayReduce('median', arraySlice(groupArray(value_float),-10,10)) as `value` "
                "FROM (SELECT object_id, timestamp, value_float FROM drive_sensor_history WHERE (id = " + ToString(CAN_AIRBAG) + ") AND (subid = " + ToString(CAN_ENGINE_IS_ON) + ") AND timestamp > now() - toIntervalDay(30) "
                "ORDER BY timestamp) GROUP BY object_id) WHERE value != 0";
        }

    public:
        TString GetTypeName() const override {
            return TypeName;
        }

    private:
        static inline const TString TypeName{"dashboard_check_airbag_signal"};
        static inline const TFactory::TRegistrator<TDashboardCheckAirbagSignalConfiguration> Registrator{TypeName};
    };

    class TDashboardTyrePressureSignalConfiguration : public IDashboardSignalConfiguration {
    protected:
        using TBase = IDashboardSignalConfiguration;

    private:
        TString GetSQLRequest() const override {
            return "SELECT object_id FROM (SELECT object_id, arrayReduce('median', arraySlice(groupArray(value_float),-10,10)) as `value` "
                "FROM (SELECT object_id, timestamp, value_float FROM drive_sensor_history WHERE (id = " + ToString(CAN_INFLATION_PRESSURE) + ") AND (subid = " + ToString(CAN_ENGINE_IS_ON) + ") AND timestamp > now() - toIntervalDay(30) "
                "ORDER BY timestamp) GROUP BY object_id) WHERE value != 0";
        }

    public:
        TString GetTypeName() const override {
            return TypeName;
        }

    private:
        static inline const TString TypeName{"dashboard_tyre_pressure_signal"};
        static inline const TFactory::TRegistrator<TDashboardTyrePressureSignalConfiguration> Registrator{TypeName};
    };

    class TDashboardWasherLiquidSignalConfiguration : public IDashboardSignalConfiguration {
    protected:
        using TBase = IDashboardSignalConfiguration;

    private:
        TString GetSQLRequest() const override {
            return "SELECT object_id FROM (SELECT object_id, arrayReduce('median', arraySlice(groupArray(value_float),-10,10)) as `value` "
                "FROM (SELECT object_id, timestamp, value_float FROM drive_sensor_history WHERE (id = " + ToString(VEGA_CAN_WASHER_LIQUID) + ") AND (subid = " + ToString(CAN_ENGINE_IS_ON) + ") AND timestamp > now() - toIntervalDay(30) "
                "ORDER BY timestamp) GROUP BY object_id) WHERE value != 0";
        }

    public:
        TString GetTypeName() const override {
            return TypeName;
        }

    private:
        static inline const TString TypeName{"dashboard_washer_liquid_signal"};
        static inline const TFactory::TRegistrator<TDashboardWasherLiquidSignalConfiguration> Registrator{TypeName};
    };

    class TDashboardCheckOilSignalConfiguration : public IDashboardSignalConfiguration {
    protected:
        using TBase = IDashboardSignalConfiguration;

    private:
        TString GetSQLRequest() const override {
            return "SELECT object_id FROM (SELECT object_id, arrayReduce('median', arraySlice(groupArray(value_float),-10,10)) as `value` "
                "FROM (SELECT object_id, timestamp, value_float FROM drive_sensor_history WHERE (id = " + ToString(CAN_CHECK_OIL) + ") AND (subid = " + ToString(CAN_ENGINE_IS_ON) + ") AND timestamp > now() - toIntervalDay(30) "
                "ORDER BY timestamp) GROUP BY object_id) WHERE value != 0";
        }

    public:
        TString GetTypeName() const override {
            return TypeName;
        }

    private:
        static inline const TString TypeName{"dashboard_check_oil_signal"};
        static inline const TFactory::TRegistrator<TDashboardCheckOilSignalConfiguration> Registrator{TypeName};
    };

    class TDashboardCheckBrakeSignalConfiguration : public IDashboardSignalConfiguration {
    protected:
        using TBase = IDashboardSignalConfiguration;

    private:
        TString GetSQLRequest() const override {
            return "SELECT object_id FROM (SELECT object_id, arrayReduce('median', arraySlice(groupArray(value_float),-10,10)) as `value` "
                "FROM (SELECT object_id, timestamp, value_float FROM drive_sensor_history WHERE (id = " + ToString(CAN_CHECK_BRAKE_PADS) + ") AND (subid = " + ToString(CAN_ENGINE_IS_ON) + ") AND timestamp > now() - toIntervalDay(30) "
                "ORDER BY timestamp) GROUP BY object_id) WHERE value != 0";
        }

    public:
        TString GetTypeName() const override {
            return TypeName;
        }

    private:
        static inline const TString TypeName{"dashboard_check_brake_signal"};
        static inline const TFactory::TRegistrator<TDashboardCheckBrakeSignalConfiguration> Registrator{TypeName};
    };

    class TCheckMaintenanceMileageSignalConfiguration {
    public:
        NJson::TJsonValue GetCheckers() const {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ECarTags) + "." + ToString(NAlerts::EFetchedItems::MaintenanceMileageDelta);
                checker["min_value"] = ToString(MileageThreshold);
                checker["max_value"] = "1000000";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::MaintenanceMileageDelta);
                auto& meta = checker["meta"];
                meta["mileage_sensor"] = CAN_ODOMETER_KM;
                meta["reverse_value"] = false;
                meta["only_active"] = true;
            }
            return checkers;
        }

    private:
        R_FIELD(ui32, MileageThreshold, 0);

    public:
        DECLARE_FIELDS(
            Field(MileageThreshold, "mileage_threshold_after_maintenance", true)
        );

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
            if (!NJson::TryFieldsFromJson(json, GetFields(), errors)) {
                return false;
            }
            if (MileageThreshold < 10) {
                errors ? errors->AddMessage("DeserializeFromJson", "MileageThreshold < 10, too low") : void();
                return false;
            }
            return true;
        }

        NJson::TJsonValue SerializeParamsToJson() const {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const {
            NJson::TJsonValue result;
            return NJson::MergeJson(SerializeParamsToJson(), result);
        }

        TString GetBackgroundProcessesSuffix() const {
            return "-mileage";
        }
    };

    class TCheckMaintenanceDaysSignalConfiguration {
    public:
        NJson::TJsonValue GetCheckers() const {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ECarTags) + "." + ToString(NAlerts::EFetchedItems::MaintenanceTime);
                checker["min_value"] = ToString(DaysThreshold) + "d";
                checker["max_value"] = "inf";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::MaintenanceTime);
                auto& meta = checker["meta"];
                meta["mileage_sensor"] = CAN_ODOMETER_KM;
                meta["reverse_value"] = false;
                meta["only_active"] = true;
            }
            return checkers;
        }

    private:
        R_FIELD(ui32, DaysThreshold, 0);

    public:
        DECLARE_FIELDS(
            Field(DaysThreshold, "days_threshold_after_maintenance", true)
        );

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) {
            if (!NJson::TryFieldsFromJson(json, GetFields(), errors)) {
                return false;
            }
            if (DaysThreshold < 10) {
                errors ? errors->AddMessage("DeserializeFromJson", "DaysThreshold < 10, too low") : void();
                return false;
            }
            return true;
        }

        NJson::TJsonValue SerializeParamsToJson() const {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const {
            NJson::TJsonValue result;
            return NJson::MergeJson(SerializeParamsToJson(), result);
        }

        TString GetBackgroundProcessesSuffix() const {
            return "-days";
        }
    };

    template<class TSubProcess>
    class ICheckMaintenanceSignalConfiguration: public virtual ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    protected:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription("check_maintenance_signal");
            settings["period"] = "5m";
            settings["sharding_policy"] = NJson::TMapBuilder
                ("min_shard", 0)
                ("max_shard", 65536);
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["filter"] = GetCarTagsFilters();
                fetcherConfig["index"] = 0;
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            }
            return fetchers;
        }

        NJson::TJsonValue GetCheckers() const override {
            return SubProcess.GetCheckers();
        }

        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }

    private:
        R_FIELD(TSubProcess, SubProcess);

    public:
        ICheckMaintenanceSignalConfiguration()
            : TBase()
            , SubProcess()
        {
        }

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return SubProcess.DeserializeFromJson(json, errors);
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            return SubProcess.SerializeParamsToJson();
        }

        NJson::TJsonValue SerializeToJson() const override {
            return SubProcess.SerializeToJson();
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front() + SubProcess.GetBackgroundProcessesSuffix();
        }
    };

    class TCheckMaintenanceSignalConfiguration: public ICheckMaintenanceSignalConfiguration<TCheckMaintenanceMileageSignalConfiguration>, public ICheckMaintenanceSignalConfiguration<TCheckMaintenanceDaysSignalConfiguration> {
    protected:
        using TCheck1 = ICheckMaintenanceSignalConfiguration<TCheckMaintenanceMileageSignalConfiguration>;
        using TCheck2 = ICheckMaintenanceSignalConfiguration<TCheckMaintenanceDaysSignalConfiguration>;

        using TBase = TCheck1::TBase;

    public:
        TCheckMaintenanceSignalConfiguration()
            : TBase()
            , TCheck1()
            , TCheck2()
        {
        }

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return
                TBase::DeserializeFromJson(json, errors) &&
                TCheck1::DeserializeFromJson(json, errors) &&
                TCheck2::DeserializeFromJson(json, errors);
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            auto result = TCheck1::SerializeParamsToJson();
            return NJson::MergeJson(TCheck2::SerializeParamsToJson(), result);
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            auto mileage = TCheck1::SerializeToJson();
            for (auto& el : mileage.GetMap()) {
                result[el.first] = el.second;
            }
            auto days = TCheck2::SerializeToJson();
            for (auto& el : days.GetMap()) {
                result[el.first] = el.second;
            }
            return result;
        }

    public:
        TVector<TRTBackgroundProcessContainer> ConstructBackgroundProcesses() const override {
            TVector<TRTBackgroundProcessContainer> container;
            container.push_back(
                TCheck1::ConstructBackgroundProcess()
            );
            container.push_back(
                TCheck2::ConstructBackgroundProcess()
            );
            return container;
        }

        TVector<TString> GetBackgroundProcessesNames() const override {
            TVector<TString> container;
            container.push_back(
                TCheck1::GetBackgroundProcessesName()
            );
            container.push_back(
                TCheck2::GetBackgroundProcessesName()
            );
            return container;
        }

        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& /*settings*/) const override {
            return true;
        }

    public:
        TString GetTypeName() const {
            return TypeName;
        }

    private:
        static inline const TString TypeName{"check_maintenance_signal"};
        static inline const TFactory::TRegistrator<TCheckMaintenanceSignalConfiguration> Registrator{TypeName};
    };

    class TChangeVINSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "5m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }
        TString GetSQLRequest() const {
            return "SELECT object_id FROM (SELECT VIN_Array, object_id FROM(SELECT arrayDistinct(arrayMap(x -> substring(x, 1, 17), groupArray(value_string))) AS VIN_Array, object_id "
                "FROM( SELECT * FROM drive_sensor_history WHERE id = " + ToString(VEGA_VIN_SENSOR) + " AND value_string NOT LIKE 'negResp%' AND value_string NOT LIKE '?????????????????%'AND timestamp > now() - toIntervalDay(30) AND object_id != '') "
                "GROUP BY object_id) WHERE length(VIN_Array) > 1) WHERE ngramDistance(arrayElement(VIN_Array, 1), arrayElement(VIN_Array, 2)) > 0.29";
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["query"] = GetSQLRequest();
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds) + "." + ToString(NAlerts::EFetchedItems::Main);
                checker["min_value"] = "1";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Main);
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& settings) const override {
            return TBase::FilterProcess(settings);
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"change_vin_signal"};
        static inline const TFactory::TRegistrator<TChangeVINSignalConfiguration> Registrator{TypeName};

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return TBase::DeserializeFromJson(json, errors);
        }

        NJson::TJsonValue SerializeToJson() const override {
            return TBase::SerializeToJson();
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TEngineOverheatingSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "5m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }
        TString GetSQLRequest() const {
            return "SELECT object_id FROM (SELECT arrayFilter(x -> x >= " + ToString(MinTemp) + ", groupArray(value_float)) AS overheat, object_id "
            "FROM (SELECT object_id, value_float, timestamp FROM drive_sensor_history WHERE timestamp > now() - toIntervalHour(1) AND id = " + ToString(CAN_ENGINE_TEMP) + " ORDER BY timestamp) "
            "GROUP BY object_id) WHERE length(overheat) > 5";
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["query"] = GetSQLRequest();
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ESensorHistoryIds) + "." + ToString(NAlerts::EFetchedItems::Main);
                checker["min_value"] = "1";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Main);
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& settings) const override {
            return TBase::FilterProcess(settings);
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"engine_overheating_signal"};
        static inline const TFactory::TRegistrator<TEngineOverheatingSignalConfiguration> Registrator{TypeName};

        R_FIELD(uint8_t, MinTemp, 100);

    public:
        DECLARE_FIELDS(
            Field(MinTemp, "min_temp")
        );

            bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            if (!TBase::DeserializeFromJson(json, errors) || !NJson::TryFieldsFromJson(json, GetFields(), errors)) {
                return false;
            }
            if (MinTemp < 100) {
                errors ? errors->AddMessage("DeserializeFromJson", "min_temp < 100, too low") : void();
                return false;
            }
            return true;
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            NJson::TJsonValue result;
            NJson::FieldsToJson(result, GetFields());
            return result;
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            return NJson::MergeJson(SerializeParamsToJson(), result);
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TLabourOverdoSignal : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "5m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Session);
            return settings;
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESessions);
                fetcherConfig["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
                fetcherConfig["states_filter"] = NJson::TArrayBuilder({
                    ::ToString(TChargableTag::Riding)
                });
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESessions);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ESessions) + "." + ToString(NAlerts::EFetchedItems::SessionStateTime);
                checker["min_value"] = ::ToString(LabourLimit.Seconds());
                checker["max_value"] = ::ToString(TDuration::Max().Seconds());
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::SessionStateTime);
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        bool FilterProcess(NJson::TJsonValue& settings) const override {
            return TBase::FilterProcess(settings);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            return filter;
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"labour_overdo_signal"};
        static inline const TFactory::TRegistrator<TLabourOverdoSignal> Registrator{TypeName};

        R_FIELD(TDuration, LabourLimit);;

    public:
        DECLARE_FIELDS(
            Field(LabourLimit, "labour_limit", true)
        );

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return TBase::DeserializeFromJson(json, errors) &&
                NJson::TryFieldsFromJson(json, GetFields(), errors);
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            return NJson::MergeJson(SerializeParamsToJson(), result);
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TWirelessBlockCheckerSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "10s";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
            return settings;
        }

        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["filter"] = GetCarTagsFilters();
                fetcherConfig["index"] = 0;
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ECarTags) + "." + ToString(NAlerts::EFetchedItems::Sensor);
                checker["min_value"] = "1";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Sensor);
                checker["meta"] = NJson::TMapBuilder
                    ("sensor_id", VEGA_NRF_DESIRED_RELAY_STATE)
                    ("min_sensor_age", 1)
                ;
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto addActions = [&actionsDesc, timeZone = TimeZone](const EObjectHistoryAction& historyAction, TString&& tagName) -> void {
                auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
                NJson::TJsonValue value = NJson::TMapBuilder
                    ("tag", tagName)
                ;
                action["tag_data"] = value.GetStringRobust();
                action["max_tags"] = 500;
                action["auto_remove"] = false;
                action["save_fetched"] = false;
                action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
                action["tag_action"] = ToString(historyAction);
                action["tag_name"] = tagName;
                action["timezone"] = timeZone;
            };
            addActions(EObjectHistoryAction::Remove, "leasing_car_before_block");
            addActions(EObjectHistoryAction::Add, std::move(signalName));
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            TString blockedSignalName = GetSignalName();
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            filter["include_tags_filter"] = filter.Has("include_tags_filter")
                ? filter["include_tags_filter"].GetString() + "*(-" + blockedSignalName + ",leasing_car_before_block)*-leasing_car_is_unblock"
                : "*(-" + blockedSignalName + ",leasing_car_before_block)*-leasing_car_is_unblock";
            return filter;
        }
        bool FilterProcess(NJson::TJsonValue& /*settings*/) const override {
            return true;
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"wireless_block_checker_signal"};
        static inline const TFactory::TRegistrator<TWirelessBlockCheckerSignalConfiguration> Registrator{TypeName};

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return TBase::DeserializeFromJson(json, errors);
        }

        NJson::TJsonValue SerializeToJson() const override {
            return TBase::SerializeToJson();
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TTraceFilterSignalConfiguration : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        R_FIELD(TString, TraceTagsFilter);

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "10s";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Session);
            return settings;
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["cars_filter"] = GetCarTagsFilters();
                fetcherConfig["index"] = 0;
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ETraceTagsHistory);
                fetcherConfig["query"] = NJson::TMapBuilder
                    (
                        "tag_names",
                        NJson::ToJson(TTagsFilter::BuildFromString(GetTraceTagsFilter()).GetRelatedTagNames())
                    )
                    (
                        "actions", NJson::TArrayBuilder
                            (ToString(EObjectHistoryAction::Add))
                    )
                ;
                fetcherConfig["trace_filter"] = GetTraceTagsFilter();
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ETraceTagsHistory);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ETraceTagsHistory) + "." + ToString(NAlerts::EFetchedItems::Main);
                checker["min_value"] = "0";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::Main);
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = false;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ETraceTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        bool FilterProcess(NJson::TJsonValue& /*settings*/) const override {
            return true;
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"trace_filter_signal"};
        static inline const TFactory::TRegistrator<TTraceFilterSignalConfiguration> Registrator{TypeName};

    public:
        DECLARE_FIELDS(
            Field(TraceTagsFilter, "trace_tags_filter", true)
        );

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return TBase::DeserializeFromJson(json, errors) &&
                NJson::TryFieldsFromJson(json, GetFields(), errors);
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            NJson::MergeJson(SerializeParamsToJson(), result);
            return result;
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }

    private:
        bool Verify(const NDrive::IServer& server, TMessagesCollector* errors) const override {
            auto allowTagsAgression = MakeSet(
                StringSplitter(
                    server.GetSettings().GetValue<TString>("signals.params.trace_filter_signal").GetOrElse("")
                ).SplitBySet(",").SkipEmpty().ToList<TString>()
            );
            auto tagsContainer = TTagsFilter::BuildFromString(GetTraceTagsFilter()).GetRelatedTagNames();
            for (const auto& tagName : tagsContainer) {
                if (!allowTagsAgression.contains(tagName) ) {
                    errors ? errors->AddMessage("Verify", "unknown tag name: " + tagName) : void();
                    return false;
                }
            }
            return true;
        }
    };

    class TLittleRestSignal : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "5m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::Session);
            return settings;
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESessions);
                fetcherConfig["entity_type"] = ToString(NAlerts::EAlertEntityType::Car);
                fetcherConfig["states_filter"] = NJson::TArrayBuilder({
                    ::ToString(TChargableTag::Riding)
                });
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESessions);
                checker["fetcher"] = ToString(NAlerts::EDataFetcherType::ESessions) + "." + ToString(NAlerts::EFetchedItems::LastHistoryRide);
                checker["min_value"] = "0";
                checker["max_value"] = ::ToString(MinimalRestDuration.MicroSeconds());
                checker["skip_empty"] = false;
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::LastHistoryRide);
                auto& meta = checker["meta"];
                meta["history_depth"] = ::ToString(MinimalRestDuration);
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = ToString(NAlerts::EDataFetcherType::ECarTags);
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalCarTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        bool FilterProcess(NJson::TJsonValue& settings) const override {
            return TBase::FilterProcess(settings);
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            NJson::TJsonValue filter = TBase::GetCarTagsFilters();
            filter["use_LBS_for_area_tags"] = false;
            filter["skip_without_location"] = false;
            return filter;
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"little_rest_signal"};
        static inline const TFactory::TRegistrator<TLittleRestSignal> Registrator{TypeName};

        R_FIELD(TDuration, MinimalRestDuration);

    public:
        DECLARE_FIELDS(
            Field(MinimalRestDuration, "rest_duration", true)
        );

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return
                TBase::DeserializeFromJson(json, errors) &&
                NJson::TryFieldsFromJson(json, GetFields(), errors) &&
                MinimalRestDuration > TDuration::Zero() &&
                MinimalRestDuration < (Now() - TInstant::Zero());

        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            return NJson::MergeJson(SerializeParamsToJson(), result);
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }
    };

    class TScreeningUserSignal : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "1m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::User);
            return settings;
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::EUserData);
                fetcherConfig["without_ridings"] = true;
                fetcherConfig["check_ridings"] = true;
                fetcherConfig["user_status"] = NJson::TArrayBuilder({
                    ::ToString(NDrive::UserStatusScreening)
                });
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            R_ENSURE(GetOwnerId(), HTTP_INTERNAL_SERVER_ERROR, "empty company filter");
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::EUserData);
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::TagsFilter);
                checker["min_value"] = "1";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                auto& meta = checker["meta"];
                meta["filter"] = GetOwnerId();
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = "user_tags";
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalUserTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        bool FilterProcess(NJson::TJsonValue& /*settings*/) const override {
            return true;
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            return true;
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"user_screening_signal"};
        static inline const TFactory::TRegistrator<TScreeningUserSignal> Registrator{TypeName};

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return TBase::DeserializeFromJson(json, errors);
        }

        NJson::TJsonValue SerializeToJson() const override {
            return TBase::SerializeToJson();
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }

        TString GetSource() const override {
            return "driver";
        }
    };

    class TAcceptanceDelayedSignal : public ISignalConfiguration, public IRobotSetting {
    protected:
        using TBase = ISignalConfiguration;

    private:
        NJson::TJsonValue GetSettings() const override {
            NJson::TJsonValue settings;
            settings["bp_enabled"] = GetIsEnabled();
            settings["host_filter"] =
                NJson::TMapBuilder
                    ("ctype", "stable_maps")
                ;
            settings["bp_description"] = GetDescription(GetTypeName());
            settings["period"] = "1m";
            settings["sharding_policy"] =
                NJson::TMapBuilder
                    ("min_shard", 0)
                    ("max_shard", 65536)
                ;
            settings["entity_type"] = ToString(NAlerts::EAlertEntityType::User);
            return settings;
        }
        NJson::TJsonValue GetFetchers() const override {
            NJson::TJsonValue fetchers;
            {
                auto& fetcherConfig = fetchers.AppendValue(NJson::JSON_MAP)["config"];
                fetcherConfig["index"] = 0;
                fetcherConfig["type"] = ToString(NAlerts::EDataFetcherType::ESessions);
                fetcherConfig["entity_type"] = ToString(NAlerts::EAlertEntityType::User);
                fetcherConfig["tag_duration"] = TagDuration.MicroSeconds();
                fetcherConfig["states_filter"] =  NJson::TArrayBuilder("old_state_reservation");
            }
            return fetchers;
        }
        NJson::TJsonValue GetCheckers() const override {
            R_ENSURE(GetOwnerId(), HTTP_INTERNAL_SERVER_ERROR, "empty company filter");
            NJson::TJsonValue checkers;
            {
                auto& checker = checkers.AppendValue(NJson::JSON_MAP);
                checker["fetcher_type"] = ToString(NAlerts::EDataFetcherType::ESessions);
                checker["iterator_type"] = ToString(NAlerts::EFetchedItems::TagsFilter);
                checker["min_value"] = "1";
                checker["max_value"] = "2";
                checker["skip_empty"] = false;
                auto& meta = checker["meta"];
                meta = NJson::TMapBuilder
                    ("filter", GetOwnerId())
                ;
            }
            return checkers;
        }
        NJson::TJsonValue GetActions() const override {
            TString signalName = GetSignalName();
            NJson::TJsonValue actionsDesc;
            actionsDesc["action_type"] = "composite";
            auto& action = actionsDesc["actions"].AppendValue(NJson::JSON_MAP);
            NJson::TJsonValue value = NJson::TMapBuilder
                ("tag", signalName)
            ;
            action["tag_data"] = value.GetStringRobust();
            action["max_tags"] = 500;
            action["auto_remove"] = true;
            action["save_fetched"] = false;
            action["action_type"] = "user_tags";
            action["tag_action"] = ToString(EObjectHistoryAction::Add);
            action["tag_name"] = signalName;
            action["timezone"] = TimeZone;
            return actionsDesc;
        }
        bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const override {
            auto tagDescription = MakeAtomicShared<TTagDescription>();
            tagDescription->SetName(GetSignalName());
            tagDescription->SetDisplayName(GetOwnerName() + ":" + GetDisplayName());
            tagDescription->SetType(TGenericSignalUserTag::Type());
            const auto& tagsMeta = server.GetDriveAPI()->GetTagsManager().GetTagsMeta();
            return tagsMeta.RegisterTag(tagDescription, authorId, tx);
        }
        bool FilterProcess(NJson::TJsonValue& /*settings*/) const override {
            return true;
        }
        NJson::TJsonValue GetCarTagsFilters() const override {
            return true;
        }

    private:
        TString GetTypeName() const {
            return TypeName;
        }
        static inline const TString TypeName{"acceptance_delayed_signal"};
        static inline const TFactory::TRegistrator<TAcceptanceDelayedSignal> Registrator{TypeName};

        R_FIELD(TDuration, TagDuration);

    public:
        DECLARE_FIELDS(
            Field(TagDuration, "delay", true)
        );

        bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors) override {
            return
                TBase::DeserializeFromJson(json, errors) &&
                NJson::TryFieldsFromJson(json, GetFields(), errors) &&
                TagDuration.Seconds() > MinimalDelayInAcceptanceDelayedSignal;
        }

        NJson::TJsonValue SerializeParamsToJson() const override {
            NJson::TJsonValue result;
            return NJson::FieldsToJson(result, GetFields());
        }

        NJson::TJsonValue SerializeToJson() const override {
            NJson::TJsonValue result = TBase::SerializeToJson();
            return NJson::MergeJson(SerializeParamsToJson(), result);
        }

        TString GetBackgroundProcessesName() const override {
            return TBase::GetBackgroundProcessesNames().front();
        }

        TString GetSource() const override {
            return "booking";
        }
    };
}
