#pragma once

#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/data/scoring/scoring.h>
#include <drive/backend/database/consistency/consistency.h>
#include <drive/backend/drivematics/zone/zone.h>
#include <drive/backend/rt_background/manager/settings.h>

#include <rtline/util/types/field.h>

namespace NDrivematics {

    class TUserOrganizationAffiliationTag;

    class TSpeedingSignalConfiguration;

    struct IRobotSetting {
    public:
        virtual NJson::TJsonValue GetBaseSettings(const TString bpName) const;

        virtual NJson::TJsonValue GetSettings() const = 0;
        virtual NJson::TJsonValue GetFetchers() const = 0;
        virtual NJson::TJsonValue GetCheckers() const = 0;
        virtual NJson::TJsonValue GetActions() const = 0;

        virtual TString GetBackgroundProcessesName() const = 0;

        virtual NJson::TJsonValue ConstructBackgroundSetting() const {
            const TString bpName = GetBackgroundProcessesName();
            NJson::TJsonValue containerJson = GetBaseSettings(bpName);
            auto& settings = containerJson["bp_settings"];
            settings = GetSettings();
            auto& fetchers = settings["fetchers"];
            fetchers = GetFetchers();
            auto& checkers = settings["checkers"];
            checkers = GetCheckers();
            auto& actionsDesc = settings["action"];
            actionsDesc = GetActions();
            return containerJson;
        }
    };

    class ISignalConfiguration {
    public:
        using TFactory = NObjectFactory::TObjectFactory<ISignalConfiguration, TString>;
        using TPtr = TAtomicSharedPtr<ISignalConfiguration>;

    public:
        enum class ESignalPriority {
            Normal      /* "normal" */,
            Critical    /* "critical" */,
        };

        struct TZoneContainer {
        public:
            bool DeserializeFromJson(const NJson::TJsonValue& json);
            NJson::TJsonValue SerializeToJson() const;

            bool IsEmpty() const {
                return !HasIncluded() && !HasExcluded();
            }

        private:
            R_OPTIONAL(NDrive::TZoneIds, Included);
            R_OPTIONAL(NDrive::TZoneIds, Excluded);
        };

    public:
        virtual ~ISignalConfiguration() = default;

        virtual bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors = nullptr);
        virtual NJson::TJsonValue SerializeToJson() const;

        static ISignalConfiguration::TPtr ConstructConfiguration(const NJson::TJsonValue& json, TMessagesCollector* errors, const NDrive::IServer& server);

        virtual TVector<TString> GetBackgroundProcessesNames() const;

        virtual TRTBackgroundProcessContainer ConstructBackgroundProcess() const;
        virtual TVector<TRTBackgroundProcessContainer> ConstructBackgroundProcesses() const;
        virtual bool CreateTags(const TString& authorId, const NDrive::IServer& server, NDrive::TEntitySession& tx) const;
        virtual bool Verify(const NDrive::IServer& /*server*/, TMessagesCollector* /*errors*/) const {
            return true;
        }

        virtual NJson::TJsonValue SerializeParamsToJson() const {
            return Default<NJson::TJsonValue>();
        }

        virtual TString GetSource() const {
            return "telematics";
        }

        TString GetSignalName() const;
        TString GetDescription(TString typeName) const;

    public:
        const ui8 TimeZone = 3;

    protected:
        virtual bool FilterProcess(NJson::TJsonValue& settings) const;
        virtual NJson::TJsonValue GetCarTagsFilters() const;

    private:
        R_FIELD(TString, SignalId);
        R_FIELD(TString, DisplayName);
        R_FIELD(TString, Type);
        R_FIELD(ESignalPriority, SignalPriority, ESignalPriority::Normal);
        R_FIELD(TString, OwnerName);
        R_FIELD(bool, IsEnabled, true);
        R_OPTIONAL(TCarsFilter, CarsFilter);
        R_OPTIONAL(TDuration, ActivationThreshold);
        R_OPTIONAL(TZoneContainer, ZoneContainer);
        R_FIELD(TString, OwnerId);

        DECLARE_FIELDS(
            Field(SignalId, "signal_id"),
            Field(DisplayName, "display_name", true),
            Field(Type, "type"),
            Field(NJson::Stringify(SignalPriority), "signal_priority"),
            Field(OwnerName, "owner_name"),
            Field(IsEnabled, "is_enabled"),
            Field(CarsFilter, "cars_filter"),
            Field(NJson::Seconds(ActivationThreshold), "activation_threshold"),
            Field(ZoneContainer, "zone"),

            Field(OwnerId, "owner_id")
        );
    };

    class TSignalConfigurationDB {
    public:
        using TFactory = NObjectFactory::TObjectFactory<ISignalConfiguration, TString>;
        using TPtr = TAtomicSharedPtr<ISignalConfiguration>;
        using TId = TString;

    public:
        TSignalConfigurationDB() = default;
        TSignalConfigurationDB(TPtr signalConfiguration):
            Data(signalConfiguration)
        {
        }
        virtual ~TSignalConfigurationDB() = default;

    public:
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json, TMessagesCollector* errors);
        virtual NJson::TJsonValue SerializeToJson() const;

        bool DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*context*/);
        NStorage::TTableRecord SerializeToTableRecord() const;

        TVector<TConsistency> GetReferences() const;

        static TString GetTableName() {
            return "signals_configuration";
        }

        static TString GetHistoryTableName() {
            return GetTableName() + "_history";
        }

        TString GetInternalId() const {
            return Id;
        }

        bool HasRevision() const {
            return Revision != Max<ui64>();
        }

        TMaybe<ui64> OptionalRevision() const {
            return HasRevision() ? Revision : TMaybe<ui64>();
        }

        explicit operator bool() const {
            return !!Data;
        }

        ISignalConfiguration* operator->() const {
            return Yensured(Data).Get();
        }

    public:
        class TDecoder : public TBaseDecoder {
            R_FIELD(i32, Id, -1);
            R_FIELD(i32, Type, -1);
            R_FIELD(i32, Revision, -1);
            R_FIELD(i32, Meta, -1);

        public:
            TDecoder() = default;
            TDecoder(const TMap<TString, ui32>& decoderBase);
        };

        bool DeserializeWithDecoderVerbose(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, TMessagesCollector& errors, const IHistoryContext* hContext);
        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
            TMessagesCollector errors;
            return DeserializeWithDecoderVerbose(decoder, values, errors, hContext);
        }

    private:
        R_FIELD(TString, Id, "uuid_generate_v4()");
        R_FIELD(TString, Type);
        R_FIELD(ui64, Revision, Max<ui64>());

        R_FIELD(TPtr, Data);
    };
}

namespace {
    using namespace NDrivematics;
}

class TSignalsConfigurationsConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
        return "id IN (" + session->Quote(ids) + ")";
    }

    static NStorage::TTableRecord BuildCondition(const TString& id) {
        NStorage::TTableRecord trCondition;
        trCondition.Set("id", id);
        return trCondition;
    }

    static NStorage::TTableRecord BuildCondition(const TSignalConfigurationDB& object) {
        return BuildCondition(object.GetId());
    }
};

class TSignalsConfigurationsHistoryManager: public TDatabaseHistoryManager<TSignalConfigurationDB> {
public:
    struct TQueryOptions: public IBaseSequentialTableImpl::TQueryOptions {
    public:
        using TBase = IBaseSequentialTableImpl::TQueryOptions;

    public:
        using TBase::TBase;

    public:
        R_OPTIONAL(TSet<EObjectHistoryAction>, Actions);
        R_FIELD(NSQL::TStringContainer, SignalIds);
        R_FIELD(NSQL::TStringContainer, Types);
        R_FIELD(TString, Owner);
    };

public:
    using TBase = TDatabaseHistoryManager<TSignalConfigurationDB>;

    using TBase::TBase;

public:
    TOptionalObjectEvents<TSignalConfigurationDB> GetEvents(
        TRange<TEventId>&& idRange,
        TRange<TInstant>&& timestampRange,
        NDrive::TEntitySession& session,
        TQueryOptions&& queryOptions
    ) const;
};

class TSignalsConfigurationsStorage: private TDatabaseEntitiesManager<TSignalConfigurationDB, TSignalsConfigurationsConditionConstructor, TSignalsConfigurationsHistoryManager> {
    using TBase = TDatabaseEntitiesManager<TSignalConfigurationDB, TSignalsConfigurationsConditionConstructor, TSignalsConfigurationsHistoryManager>;

public:
    using TBase::GetObjects;
    using TBase::GetHistoryManager;

public:
    TSignalsConfigurationsStorage(const IHistoryContext& context)
        : TBase(context)
    {
    }

    [[nodiscard]] bool AddObjects(TConstArrayRef<TSignalConfigurationDB> objects, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TSignalConfigurationDB>* containerExt = nullptr) const override {
        NStorage::TObjectRecordsSet<TSignalConfigurationDB> recordsInt;
        NStorage::TObjectRecordsSet<TSignalConfigurationDB>* records = containerExt ? containerExt : &recordsInt;
        if (!TBase::AddObjects(objects, userId, session, containerExt)) {
            return false;
        }

        auto* api = NDrive::GetServerAs<NDrive::IServer>().GetDriveAPI();
        TVector<TConsistency> references;
        for (const auto& object : records->GetObjects()) {
            auto temp = object.GetReferences();
            references.insert(references.end(), temp.begin(), temp.end());
        }
        if (!api->GetConsistencyDB().AddConnectivity(references, session)) {
            return false;
        }
        return true;
    }

    [[nodiscard]] bool RemoveObjects(const TSet<typename TSignalConfigurationDB::TId>& ids, const TString& userId, NDrive::TEntitySession& session) const override {
        auto objects = GetObjects(ids, session);
        if (!objects) {
            return false;
        }
        TVector<TConsistency> references;
        for (const auto& object : *objects) {
            auto temp = object.GetReferences();
            references.insert(references.end(), temp.begin(), temp.end());
        }

        auto* api = NDrive::GetServerAs<NDrive::IServer>().GetDriveAPI();
        if (!api->GetConsistencyDB().RemoveConnectivity(references, session)) {
            return false;
        }
        if (!TBase::RemoveObjects(ids, userId, session)) {
            return false;
        }
        return true;
    }

    [[nodiscard]] TMaybe<TVector<TSignalConfigurationDB>> GetObjects(const TTagDescription::TConstPtr companyTagDescription, NDrive::TEntitySession& session) const;
    [[nodiscard]] bool UpsertObject(const TSignalConfigurationDB& object, const TString& userId, NDrive::TEntitySession& session, NStorage::TObjectRecordsSet<TSignalConfigurationDB>* /*containerExt*/ = nullptr) const override;
};
