#include "zone.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>

#include <rtline/util/types/uuid.h>

namespace NDrivematics {
    bool TZone::TStyle::DeserializeFromJson(const NJson::TJsonValue& value, TMessagesCollector* /*errors*/) {
        return
            NJson::ParseField(value["fill_color"], MutableFillColor()) &&
            NJson::ParseField(value["outline_color"], MutableOutlineColor()) &&
            !(FillColor.empty() && OutlineColor.empty());
    }
    NJson::TJsonValue TZone::TStyle::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertNonNull(result, "fill_color", GetFillColor());
        NJson::InsertNonNull(result, "outline_color", GetOutlineColor());
        return result;
    }

    bool TZone::DeserializeFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors) {
        return
            NJson::ParseField(value["id"], MutableId()) &&
            NJson::ParseField(value["owner"], OptionalOwner()) &&
            NJson::ParseField(value["name"], MutableName(), true, errors) &&
            NJson::ParseField(value["revision"], MutableRevision()) &&
            value.Has("meta") &&
            DeserializeMetaFromJson(value["meta"]);
    }
    NJson::TJsonValue TZone::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "id", GetId());
        NJson::InsertNonNull(result, "owner", OptionalOwner());
        NJson::InsertField(result, "name", GetName());
        NJson::InsertNonNull(result, "revision", OptionalRevision());
        NJson::InsertField(result, "meta", SerializeMetaToJson());
        return result;
    }

    bool TZone::DeserializeMetaFromJson(const NJson::TJsonValue& meta, TMessagesCollector*  /*errors*/) {
        return
            NJson::ParseField(meta["area_ids"], OptionalAreaIds()) &&
            NJson::ParseField(meta["style"], MutableStyle()) &&
            NJson::ParseField(meta["for_mobile"], MutableForMobile());
    }

    NJson::TJsonValue TZone::SerializeMetaToJson() const {
        NJson::TJsonValue meta;
        NJson::InsertNonNull(meta, "area_ids", OptionalAreaIds());
        NJson::InsertNonNull(meta, "style", GetStyle());
        NJson::InsertField(meta, "for_mobile", GetForMobile());
        return meta;
    }

    bool TZone::IsEmpty() const {
        return !HasAreaIds();
    }

    bool TZone::DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*context*/) {
        SetId(record.Get("id"));
        if (record.Get("owner")) {
            SetOwner(record.Get("owner"));
        }
        SetName(record.Get("name"));
        {
            ui64 revision = 0;
            TryFromString(record.Get("revision"), revision);
            if (revision) {
                SetRevision(revision);
            }
        }
        NJson::TJsonValue meta;
        if (!NJson::ReadJsonFastTree(record.Get("meta"), &meta) || !DeserializeMetaFromJson(meta)) {
            return false;
        }
        return true;
    }

    NStorage::TTableRecord TZone::SerializeToTableRecord() const {
        NStorage::TTableRecord record;
        record.Set("id", GetId());
        if (HasOwner()) {
            record.Set("owner", *OptionalOwner());
        }
        record.Set("name", GetName());
        if (HasRevision()) {
            record.Set("revision", GetRevisionRef());
        }
        record.Set("meta", SerializeMetaToJson());
        return record;
    }

    TZone::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
        Id = GetFieldDecodeIndex("id", decoderBase);
        Owner = GetFieldDecodeIndex("owner", decoderBase);
        Name = GetFieldDecodeIndex("name", decoderBase);
        Revision = GetFieldDecodeIndex("revision", decoderBase);
        Meta = GetFieldDecodeIndex("meta", decoderBase);
    }

    bool TZone::DeserializeWithDecoderVerbose(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, TMessagesCollector& errors, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, Id);
        READ_DECODER_VALUE(decoder, values, Name);
        if (decoder.GetRevision() > -1) {
            ui64 revision;
            READ_DECODER_VALUE_TEMP(decoder, values, revision, Revision);
            Revision = revision;
        }
        if (const TStringBuf& OwnerStr = decoder.GetStringValue(decoder.GetOwner(), values); !!OwnerStr) {
            SetOwner(OwnerStr);
        }
        NJson::TJsonValue meta;
        READ_DECODER_VALUE_JSON(decoder, values, meta, Meta);
        if (!DeserializeMetaFromJson(meta, &errors)) {
            return false;
        }
        return true;
    }

    TZone::TZone(TVector<TArea>& areas) {
        SetAreaIds(NDrive::TAreaIds{});
        for (const auto& area : areas) {
            MutableAreaIdsRef().insert(area.GetInternalId());
        }
    }

    TZone::TPtr TZone::Construct(const NJson::TJsonValue& value, const TString& userId, NDrive::TEntitySession& tx, const NDrive::IServer& server, bool force) {
        TVector<TArea> areas;
        TString etalonGroup;
        for (const auto& object : value["object"].GetArray()) {
            auto& area = areas.emplace_back(TArea());
            R_ENSURE(area.DeserializeFromJson(object), HTTP_INTERNAL_SERVER_ERROR, "incorrect object json", NDrive::MakeError("zone.incorrect_request"), tx);
            if (!force) {
                area.SetIdentifier(NUtil::CreateUUID());
            }
            if (etalonGroup.empty()) {
                etalonGroup = area.GetZoneName();
                continue;
            }
            R_ENSURE(etalonGroup == area.GetZoneName(), HTTP_BAD_REQUEST, "incorrect zone, all area objects must have one group", NDrive::MakeError("zone.incorrect_request"), tx);
        }

        if (!force) {
            for (auto&& area : areas) {
                R_ENSURE(server.GetDriveAPI()->GetAreasDB()->UpsertObject(area, userId, tx), HTTP_INTERNAL_SERVER_ERROR, "cannot add areas", NDrive::MakeError("zone.area.add"), tx);
            }
        }
        auto zone = MakeAtomicShared<TZone>(areas);
        zone->SetName(etalonGroup);
        if (value.Has("for_mobile") && value["for_mobile"].IsBoolean()) {
            zone->SetForMobile(value["for_mobile"].GetBoolean());
        }
        if (value.Has("style") && value["style"].IsMap()) {
            zone->MutableStyle().DeserializeFromJson(value["style"]);
        }
        zone->SetId(NUtil::CreateUUID());
        return zone;
    }
}

TString TZoneConditionConstructor::BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session) {
    return "id IN (" + session->Quote(ids) + ")";
}

TString TZoneConditionConstructor::BuildCondition(const TConstArrayRef<NDrivematics::TZone> objects, NDrive::TEntitySession& session) {
    TStringBuilder result;
        for (auto&& i : objects) {
            if (result) {
                result << ",";
            }
            if (i.GetInternalId()) {
                result << session->Quote(i.GetInternalId());
            }
        }
    return "id IN (" + result + ")";
}

NStorage::TTableRecord TZoneConditionConstructor::BuildCondition(const TString& id) {
    NStorage::TTableRecord trCondition;
    trCondition.Set("id", id);
    return trCondition;
}

NStorage::TTableRecord TZoneConditionConstructor::BuildCondition(const NDrivematics::TZone& object) {
    return BuildCondition(object.GetId());
}

TZoneStorage::TZoneStorage(const IHistoryContext& context, const THistoryConfig& hConfig)
    : TBase(context, hConfig)
{
    Y_ENSURE_BT(Start());
}

TZoneStorage::~TZoneStorage() {
    if (!Stop()) {
        ERROR_LOG << "cannot stop ZoneDB" << Endl;
    }
}

bool TZoneStorage::TOptions::Apply(const NDrivematics::TZone& entity) const {
    bool result = true;
    if (HasForMobile()) {
        result &= entity.GetForMobile() == GetForMobileRef();
    }
    if (HasCompanyObject()) {
        result &= entity.HasOwner() && entity.GetOwnerRef() == GetCompanyObjectRef()->GetName();
    } else if (HasCompanyOwner()) {
        result &= entity.HasOwner() && entity.GetOwnerRef() == GetCompanyOwnerRef();
    }
    return result;
}

[[nodiscard]] TMaybe<TVector<TZoneStorage::TObjectContainer>> TZoneStorage::GetObjects(const TZoneStorage::TOptions options, TInstant reqActuality) const {
    TVector<TObjectContainer> result;
    const auto action = [&result, &options](const TObjectContainer& entity) {
        if (options.Apply(entity)) {
            result.emplace_back(entity);
        }
    };
    if (!ForObjectsList(action, reqActuality)) {
        return {};
    }
    return {result};
}

[[nodiscard]] TMaybe<TVector<TString>> TZoneStorage::GetZoneName(const TSet<TString>& zoneNames, TInstant reqActuality) const {
    TVector<TString> result;
    std::function<void(const TObjectContainer&)> action;
    if (zoneNames.empty()) {
        action = [&result](const TObjectContainer& entity) -> void {
            result.emplace_back(entity.GetName());
        };
    } else {
        action = [&result, &zoneNames](const TObjectContainer& entity) -> void {
            if (zoneNames.contains(entity.GetName())) {
                result.emplace_back(entity.GetName());
            }
        };
    }
    if (!ForObjectsList(action, reqActuality)) {
        return {};
    }
    return {result};
}

[[nodiscard]] TMaybe<TVector<TZoneStorage::TObjectContainer>> TZoneStorage::GetZoneByName(const TSet<TString>& zoneNames, TInstant reqActuality) const {
    TVector<TObjectContainer> result;
    std::function<void(const TObjectContainer&)> action;
    if (zoneNames.empty()) {
        action = [&result](const TObjectContainer& entity) -> void {
            result.emplace_back(entity);
        };
    } else {
        action = [&result, &zoneNames](const TObjectContainer& entity) -> void {
            if (zoneNames.contains(entity.GetName())) {
                result.emplace_back(entity);
            }
        };
    }
    if (!ForObjectsList(action, reqActuality)) {
        return {};
    }
    return {result};
}

[[nodiscard]] bool TZoneStorage::GetZoneIdsInPoint(const NDrive::TZoneIds& containerZoneIds, const TGeoCoord&  coordinate, NDrive::TZoneIds& result, const TAreasDB& areaManager) const {
    auto now = TInstant::Now();
    TString zoneNameRef;
    auto actor = [&zoneNameRef](const TFullAreaInfo& area) -> bool {
        if (area.GetArea().GetType() != "zone") {
            ALERT_LOG << "TZoneStorage::GetZoneIdsInPoint: areaId: " << area.GetArea().GetInternalId() << " areaType: " << area.GetArea().GetType() << "is not 'zone'" << Endl;
            return false;
        }
        const auto& zoneName = area.GetArea().GetZoneName();
        if (!zoneName) {
            return false;
        }
        if (!zoneNameRef) {
            zoneNameRef = zoneName;
        } else if (zoneNameRef != zoneName) {
            ALERT_LOG << "TZoneStorage::GetZoneIdsInPoint: not all areas have the same group name: " << area.GetArea().GetInternalId() << " areaZoneName: " << zoneName << Endl;
            return false;
        }
        return true;
    };
    auto zonesOptional = GetObjectsByZoneIds(containerZoneIds);
    if (!zonesOptional) {
        return false;
    }
    for (const auto& zone : *zonesOptional) {
        if (zone.HasAreaIds()) {
            NDrive::TAreaIds areaIds;
            zoneNameRef.clear();
            for (auto& areaId : zone.GetAreaIdsRef()) {
                areaIds.insert(areaId);
            }
            if (!areaManager.ProcessAreaIdsImpl(areaIds, actor, areaManager.GetSnapshot(coordinate, now))) {
                return false;
            }
            if (zoneNameRef == zone.GetName()) {
                result.insert(zone.GetInternalId());
            }
        }
    }
    return true;
}

[[nodiscard]] bool TZoneStorage::GetAreasIds(const NDrive::TZoneIds& containerZoneIds, NDrive::TAreaIds& result, const TOptions* options) const {
    auto zonesOptional = GetObjectsByZoneIds(containerZoneIds);
    if (!zonesOptional) {
        return false;
    }
    for (const auto& zone : *zonesOptional) {
        if (zone.HasAreaIds()) {
            if (!options || options->Apply(zone)) {
                result.insert(zone.GetAreaIdsRef().begin(), zone.GetAreaIdsRef().end());
            }
        }
    }
    return true;
}
