#pragma once

#include <drive/backend/areas/areas.h>

#include <library/cpp/json/writer/json_value.h>

#include <rtline/library/geometry/coord.h>

namespace NDrivematics {
    class TZone;

    using TZones = TVector<TZone>;

    class TZone {
    public:
        using TPtr = TAtomicSharedPtr<TZone>;
        using TId = TString;

        struct TStyle {
        private:
            R_FIELD(TString, FillColor);
            R_FIELD(TString, OutlineColor);

        public:
            bool DeserializeFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors = nullptr);
            NJson::TJsonValue SerializeToJson() const;

            explicit operator bool() const {
                return GetFillColor() || GetOutlineColor();
            }
        };

    private:
        R_FIELD(TString, Id, "uuid_generate_v4()");
        R_OPTIONAL(TString, Owner);
        R_FIELD(TString, Name);
        R_OPTIONAL(ui64, Revision);

        R_OPTIONAL(NDrive::TAreaIds, AreaIds);
        R_FIELD(bool, ForMobile, false);
        R_FIELD(TStyle, Style);

    public:
        bool DeserializeFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors = nullptr);
        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeMetaFromJson(const NJson::TJsonValue& value, TMessagesCollector* errors = nullptr);
        NJson::TJsonValue SerializeMetaToJson() const;
        NDrive::TScheme GetScheme(const IServerBase& server) const;

        bool IsEmpty() const;

        static TPtr Construct(const NJson::TJsonValue& value, const TString& userId, NDrive::TEntitySession& tx, const NDrive::IServer& server, bool force);

    public:
        class TDecoder : public TBaseDecoder {
            R_FIELD(i32, Id, -1);
            R_FIELD(i32, Owner, -1);
            R_FIELD(i32, Name, -1);
            R_FIELD(i32, Revision, -1);
            R_FIELD(i32, Meta, -1);

        public:
            TDecoder() = default;
            TDecoder(const TMap<TString, ui32>& decoderBase);
        };

        bool DeserializeWithDecoderVerbose(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, TMessagesCollector& errors, const IHistoryContext* hContext);
        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
            TMessagesCollector errors;
            return DeserializeWithDecoderVerbose(decoder, values, errors, hContext);
        }

        bool DeserializeFromTableRecord(const NStorage::TTableRecord& record, const IHistoryContext* /*context*/);
        NStorage::TTableRecord SerializeToTableRecord() const;

    public:
        TZone() = default;
        TZone(TVector<TArea>& areas);

        static TString GetTableName() {
            return "zone";
        }

        static TString GetHistoryTableName() {
            return GetTableName() + "_history";
        }

        TString GetInternalId() const {
            return Id;
        }

        explicit operator bool() const {
            return !Id.empty();
        }
    };
}

template<>
inline bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrivematics::TZone& zone) {
    return zone.DeserializeFromJson(value);
}

template<>
inline NJson::TJsonValue NJson::ToJson(const NDrivematics::TZone& zone) {
    return zone.SerializeToJson();
}

template<>
inline bool NJson::TryFromJson(const NJson::TJsonValue& value, NDrivematics::TZone::TStyle& zoneStyle) {
    return zoneStyle.DeserializeFromJson(value);
}

template<>
inline NJson::TJsonValue NJson::ToJson(const NDrivematics::TZone::TStyle& zoneStyle) {
    return zoneStyle.SerializeToJson();
}

class TZoneConditionConstructor {
public:
    static TString BuildCondition(const TSet<TString>& ids, NDrive::TEntitySession& session);
    static TString BuildCondition(const TConstArrayRef<NDrivematics::TZone> objects, NDrive::TEntitySession& session);
    static NStorage::TTableRecord BuildCondition(const TString& id);
    static NStorage::TTableRecord BuildCondition(const NDrivematics::TZone& object);
};

class TZoneStorage : public TDBEntitiesManager<NDrivematics::TZone, TZoneConditionConstructor> {
    using TBase = TDBEntitiesManager<NDrivematics::TZone, TZoneConditionConstructor>;
    using TObjectContainer = NDrivematics::TZone;
    using TConditionConstructor = TZoneConditionConstructor;
    using THistoryManager = TDBEntitiesHistoryManager<TObjectContainer>;

public:
    using TBase::GetObjects;
    using TBase::UpsertObject;
    using TBase::ForceUpsertObject;

public:
    class TOptions {
        R_OPTIONAL(bool, ForMobile);
        R_OPTIONAL(TTagDescription::TConstPtr, CompanyObject);
        R_OPTIONAL(TString, CompanyOwner);

    public:
        TOptions(TTagDescription::TConstPtr companyObject)
            : CompanyObject(companyObject)
        {
        }
        TOptions(TString companyOwner)
            : CompanyOwner(companyOwner)
        {
        }
        TOptions() = default;

    public:
        bool Apply(const TObjectContainer& entity) const;
    };

public:
    TZoneStorage(const IHistoryContext& context, const THistoryConfig& hConfig);

    ~TZoneStorage();

    template <class TKeys, class = typename TKeys::iterator>
    [[nodiscard]] TMaybe<TVector<TObjectContainer>> GetObjectsByZoneIds(const TKeys zoneIds, TInstant reqActuality = TInstant::Zero()) const {
        return GetCachedObjectsVector(zoneIds, reqActuality);
    }
    [[nodiscard]] bool RemoveObjects(TConstArrayRef<TObjectContainer> objects, const TString& userId, NDrive::TEntitySession& session) const {
        return RemoveObjects<TObjectContainer, TConditionConstructor, THistoryManager, TConstArrayRef<TObjectContainer>>(*HistoryManager, objects, userId, session);
    }

    [[nodiscard]] TMaybe<TVector<TObjectContainer>> GetObjects(const TOptions options, TInstant reqActuality = TInstant::Zero()) const;
    [[nodiscard]] TMaybe<TVector<TString>> GetZoneName(const TSet<TString>& zoneNames, TInstant reqActuality = TInstant::Zero()) const;
    [[nodiscard]] TMaybe<TVector<TObjectContainer>> GetZoneByName(const TSet<TString>& zoneNames, TInstant reqActuality = TInstant::Zero()) const;
    [[nodiscard]] bool GetZoneIdsInPoint(const NDrive::TZoneIds& containerZoneIds, const TGeoCoord& coordinate, TSet<TString>& result, const TAreasDB& areaManager) const;
    [[nodiscard]] bool GetAreasIds(const NDrive::TZoneIds& containerZoneIds, NDrive::TAreaIds& result, const TOptions* options = nullptr) const;
};
