#include "article_matcher.h"

#include <util/generic/algorithm.h>
#include <util/generic/yexception.h>
#include <util/string/join.h>
#include <util/string/printf.h>
#include <util/string/split.h>

namespace NDrive::NFine {
    TSimpleStringRegexpMatchRule::TSimpleStringRegexpMatchRule(const TSimpleStringRegexpMatchRuleConfig& config)
        : TBase(config)
    {
    }

    bool TSimpleStringRegexpMatchRule::GetMatchResult(const TString& /* match */, TString& matchResult) const {
        matchResult = GetConfigAs<TSimpleStringRegexpMatchRuleConfig>().GetMatchResult();
        return true;
    }

    bool TGenericMatchRule::GetMatchResult(const TString& match, TString& matchResult) const {
        TVector<TString> matchParts = StringSplitter(match).SplitBySet(" .чп").SkipEmpty();
        if (matchParts.size() >= 2) {
            matchParts[1] = Sprintf("%02d", FromString<int>(matchParts[1]));
        }
        matchResult = JoinSeq("_", matchParts);
        return true;
    }

    TFineArticleMatcher::TFineArticleMatcher(const TFineArticleMatcherConfig& config)
        : Config(config)
    {
        MatchRules.push_back(MakeAtomicShared<TGenericMatchRule>(Config.GetGenericMatchRuleConfig()));

        for (auto&& [name, matchRuleConfig] : Config.GetMatchRuleConfigs()) {
            MatchRules.push_back(MakeAtomicShared<TSimpleStringRegexpMatchRule>(matchRuleConfig));
        }

        if (!AllOf(MatchRules.cbegin(), MatchRules.cend(), [](auto rulePtr) { return rulePtr->IsValid(); })) {
            ythrow yexception() << "Not all rules are defined and compiled";
        }
    }

    bool TFineArticleMatcher::DetermineArticle(const TString& rawArticle, TString& articleCode) const {
        TString recognizedArticle;
        for (auto matchRule : MatchRules) {
            if (matchRule->Match(rawArticle, recognizedArticle) && Config.GetRecognizableArticles().contains(recognizedArticle)) {
                articleCode = std::move(recognizedArticle);
                return true;
            }
        }
        return false;
    }

    bool TFineArticleMatcher::GetReadableArticle(const TString& rawArticle, TString& readableArticle, TString articleCode) const {
        if (!articleCode && !DetermineArticle(rawArticle, articleCode)) {
            return false;
        }

        auto readableArticlePtr = Config.GetReadableArticleMapping().FindPtr(articleCode);
        if (readableArticlePtr != nullptr) {
            readableArticle = readableArticlePtr->GetArticleTitle();
            return true;
        }

        return false;
    }

    TString TFineArticleMatcher::GetReadableArticleDefault(const TString& rawArticle, const TString& articleCode) const {
        TString readableArticle;
        if (GetReadableArticle(rawArticle, readableArticle, articleCode)) {
            return readableArticle;
        }
        return (rawArticle) ? rawArticle : articleCode;
    }
}
