#pragma once

#include "config.h"

#include <library/cpp/regex/pcre/regexp.h>

#include <util/generic/cast.h>
#include <util/generic/map.h>

namespace NDrive::NFine {
    class IMatchRule {
    public:
        virtual bool IsValid() const = 0;
        virtual bool Match(const TString& rawArticle, TString& article) const = 0;
        virtual ~IMatchRule() = default;
    };

    template <typename T>
    class IBaseRegexpMatchRule : public IMatchRule {
    public:
        IBaseRegexpMatchRule(const TBaseRegexpMatchRuleConfig& config);

        const TBaseRegexpMatchRuleConfig& GetConfig() const {
            return Config;
        }

        template <typename C>
        const C& GetConfigAs() const {
            return dynamic_cast<const C&>(Config);
        }

        virtual bool IsValid() const override;
        virtual bool Match(const TString& input, T& matchResult) const override;

    protected:
        virtual bool GetMatchResult(const TString& match, T& matchResult) const = 0;

        const TBaseRegexpMatchRuleConfig& Config;
        THolder<TRegExBase> Re;
    };

    template <typename T>
    IBaseRegexpMatchRule<T>::IBaseRegexpMatchRule(const TBaseRegexpMatchRuleConfig& config)
        : Config(config)
        , Re(MakeHolder<TRegExBase>(Config.GetPattern(), REG_UTF8))
    {
    }

    template <typename T>
    bool IBaseRegexpMatchRule<T>::IsValid() const {
        return !!Re && Re->IsCompiled();
    }

    template <typename T>
    bool IBaseRegexpMatchRule<T>::Match(const TString& input, T& matchResult) const {
        regmatch_t matches[NMATCHES];
        bool found = (0 == Re->Exec(input.data(), matches, 0));
        if (found) {
            TString match = input.substr(matches[0].rm_so, matches[0].rm_eo - matches[0].rm_so);

            bool foundMore = false;
            if (Config.IsRequireOneNeedle()) {
                foundMore = (0 == Re->Exec(input.data() + matches[0].rm_eo, matches, 0));
            }

            if (!foundMore && GetMatchResult(match, matchResult)) {
                return true;
            }
        }
        return false;
    }

    class TSimpleStringRegexpMatchRule : public IBaseRegexpMatchRule<TString> {
        using TBase = IBaseRegexpMatchRule<TString>;

    public:
        TSimpleStringRegexpMatchRule(const TSimpleStringRegexpMatchRuleConfig& config);

    protected:
        virtual bool GetMatchResult(const TString& match, TString& matchResult) const override;
    };

    class TGenericMatchRule : public IBaseRegexpMatchRule<TString> {
        using TBase = IBaseRegexpMatchRule<TString>;

    public:
        using TBase::TBase;

    private:
        virtual bool GetMatchResult(const TString& match, TString& matchResult) const override;
    };

    // NB. Fine article description can be found here:
    // - (official) https://avtokod.mos.ru/Pages/PenaltiesTable.aspx
    // - https://shtrafy-gibdd.ru/koap
    // - https://www.driver-helper.ru/shtrafy-gibdd/t/ljgotnyj-period-oplaty-shtrafa-gibdd-v-2018-godu
    class TFineArticleMatcher {
    public:
        using TPtr = TAtomicSharedPtr<TFineArticleMatcher>;

        TFineArticleMatcher(const TFineArticleMatcherConfig& config);

        const TFineArticleMatcherConfig& GetConfig() const noexcept {
            return Config;
        }

        bool DetermineArticle(const TString& rawArticle, TString& articleCode) const;
        bool GetReadableArticle(const TString& rawArticle, TString& readableArticle, TString articleCode = "") const;
        TString GetReadableArticleDefault(const TString& rawArticle, const TString& articleCode = "") const;

    private:
        TFineArticleMatcherConfig Config;
        TVector<TAtomicSharedPtr<IMatchRule>> MatchRules;
    };
}
