#include "entries.h"

#include <drive/library/cpp/raw_text/datetime.h>

#include <library/cpp/json/writer/json_value.h>

#include <rtline/util/instant_model.h>
#include <rtline/util/types/uuid.h>

#include <util/generic/serialized_enum.h>
#include <util/string/builder.h>

namespace {
    static TString FormatDate(TInstant ts, const TString& format = "%Y-%m-%d") {
        return NUtil::FormatDatetime(ts, format);
    }

    static bool ReadCentsValue(const NJson::TJsonValue& data, const TString& field, double& value) {
        i64 centsValue;
        JREAD_INT(data, field, centsValue);
        value = centsValue / 100.0;
        return true;
    }

    template <typename T>
    static T ConvertToCents(const double value) {
        return static_cast<T>(100 * value);  // note: unsigned int is not preferred due to possible negative sum to pay value
    }
}

namespace NDrive::NFine {
    TAutocodeFineDecoder::TAutocodeFineDecoder(const TMap<TString, ui32>& decoderBase) {
        Id = GetFieldDecodeIndex("id", decoderBase);
        SerialId = GetFieldDecodeIndex("serial_id", decoderBase);
        RulingNumber = GetFieldDecodeIndex("ruling_number", decoderBase);
        AutocodeId = GetFieldDecodeIndex("autocode_id", decoderBase);
        RulingDate = GetFieldDecodeIndex("ruling_date", decoderBase);
        ViolationTime = GetFieldDecodeIndex("violation_time", decoderBase);
        DiscountDate = GetFieldDecodeIndex("discount_date", decoderBase);
        ArticleKoap = GetFieldDecodeIndex("article_koap", decoderBase);
        ViolationPlace = GetFieldDecodeIndex("violation_place", decoderBase);
        SumToPay = GetFieldDecodeIndex("sum_to_pay", decoderBase);
        OdpsCode = GetFieldDecodeIndex("odps_code", decoderBase);
        OdpsName = GetFieldDecodeIndex("odps_name", decoderBase);
        ViolationDocumentNumber = GetFieldDecodeIndex("violation_document_number", decoderBase);
        ViolationDocumentType = GetFieldDecodeIndex("violation_document_type", decoderBase);
        HasPhoto = GetFieldDecodeIndex("has_photo", decoderBase);
        FineInformationReceivedAt = GetFieldDecodeIndex("fine_information_received_at", decoderBase);
        AutocodePaymentConfirmationId = GetFieldDecodeIndex("autocode_payment_confirmation_id", decoderBase);
        PaymentConfirmationReceivedAt = GetFieldDecodeIndex("payment_confirmation_received_at", decoderBase);
        CarId = GetFieldDecodeIndex("car_id", decoderBase);
        OrderId = GetFieldDecodeIndex("order_id", decoderBase);
        UserId = GetFieldDecodeIndex("user_id", decoderBase);
        SessionId = GetFieldDecodeIndex("session_id", decoderBase);
        ChargeEmailSentAt = GetFieldDecodeIndex("charge_email_sent_at", decoderBase);
        ChargeSmsSentAt = GetFieldDecodeIndex("charge_sms_sent_at", decoderBase);
        ChargedAt = GetFieldDecodeIndex("charged_at", decoderBase);
        ChargePassedAt = GetFieldDecodeIndex("charge_passed_at", decoderBase);
        ChargePushSentAt = GetFieldDecodeIndex("charge_push_sent_at", decoderBase);
        NeedsCharge = GetFieldDecodeIndex("needs_charge", decoderBase);
        SumToPayWithoutDiscount = GetFieldDecodeIndex("sum_to_pay_without_discount", decoderBase);
        ViolationLongitude = GetFieldDecodeIndex("violation_longitude", decoderBase);
        ViolationLatitude = GetFieldDecodeIndex("violation_latitude", decoderBase);
        IsAfterRideStartDuringOrder = GetFieldDecodeIndex("is_after_ride_start_during_order", decoderBase);
        IsCameraFixation = GetFieldDecodeIndex("is_camera_fixation", decoderBase);
        Skipped = GetFieldDecodeIndex("skipped", decoderBase);
        SourceType = GetFieldDecodeIndex("source_type", decoderBase);
        MetaInfo = GetFieldDecodeIndex("meta_info", decoderBase);
        AddedAtTimestamp = GetFieldDecodeIndex("added_at_timestamp", decoderBase);
    }

    TAutocodeFineEntry::TAutocodeFineEntry()
        : TAutocodeFineEntry(NUtil::CreateUUID())
    {
    }

    TAutocodeFineEntry::TAutocodeFineEntry(const TString& id)
        : Id(id)
    {
    }

    TAutocodeFineEntry::TEventId TAutocodeFineEntry::GetHistoryEventId() const {
        return SerialId;
    }

    const TInstant& TAutocodeFineEntry::GetHistoryInstant() const {
        return AddedAtTimestamp;
    }

    const TString& TAutocodeFineEntry::GetHistoryUserId() const {
        return Default<TString>();
    }

    bool TAutocodeFineEntry::operator < (const TAutocodeFineEntry& other) const {
        return SerialId < other.GetSerialId();
    }

    bool TAutocodeFineEntry::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /* hContext */) {
        READ_DECODER_VALUE(decoder, values, Id);
        READ_DECODER_VALUE(decoder, values, SerialId);

        READ_DECODER_VALUE(decoder, values, RulingNumber);
        if (decoder.GetAutocodeId() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, AutocodeId, 0);
        }

        READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, RulingDate);
        if (!RulingDate) {
            return false;
        }
        if (decoder.GetViolationTime() >= 0) {
            READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, ViolationTime);
        }
        if (decoder.GetDiscountDate() >= 0) {
            READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, DiscountDate);
        }

        READ_DECODER_VALUE(decoder, values, ArticleKoap);
        READ_DECODER_VALUE(decoder, values, ViolationPlace);

        READ_DECODER_VALUE(decoder, values, SumToPay);

        if (decoder.GetOdpsCode() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, OdpsCode, Default<TString>());
        }
        if (decoder.GetOdpsName() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, OdpsName, Default<TString>());
        }

        if (decoder.GetViolationDocumentNumber() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, ViolationDocumentNumber, Default<TString>());
        }
        if (decoder.GetViolationDocumentType() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, ViolationDocumentType, Default<TString>());
        }

        READ_DECODER_VALUE(decoder, values, HasPhoto);

        READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, FineInformationReceivedAt);
        if (!FineInformationReceivedAt) {
            return false;
        }
        READ_DECODER_VALUE_INSTANT(decoder, values, AddedAtTimestamp);

        if (decoder.GetAutocodePaymentConfirmationId() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, AutocodePaymentConfirmationId, 0);
        }
        if (decoder.GetPaymentConfirmationReceivedAt() >= 0) {
            READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, PaymentConfirmationReceivedAt);
        }

        READ_DECODER_VALUE(decoder, values, CarId);

        if (decoder.GetOrderId() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, OrderId, Default<TString>());
        }
        if (decoder.GetUserId() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, UserId, Default<TString>());
        }
        if (decoder.GetSessionId() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, SessionId, Default<TString>());
        }

        if (decoder.GetChargeEmailSentAt() >= 0) {
            READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, ChargeEmailSentAt);
        }
        if (decoder.GetChargeSmsSentAt() >= 0) {
            READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, ChargeSmsSentAt);
        }
        if (decoder.GetChargedAt() >= 0) {
            READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, ChargedAt);
        }
        if (decoder.GetChargePassedAt() >= 0) {
            READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, ChargePassedAt);
        }
        if (decoder.GetChargePushSentAt() >= 0) {
            READ_DECODER_VALUE_INSTANT_ISOFORMAT_OPT(decoder, values, ChargePushSentAt);
        }

        READ_DECODER_VALUE(decoder, values, NeedsCharge);
        READ_DECODER_VALUE(decoder, values, SumToPayWithoutDiscount);

        if (decoder.GetViolationLongitude() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, ViolationLongitude, 0);
        }
        if (decoder.GetViolationLatitude() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, ViolationLatitude, 0);
        }

        READ_DECODER_VALUE(decoder, values, IsAfterRideStartDuringOrder);
        READ_DECODER_VALUE(decoder, values, IsCameraFixation);

        if (decoder.GetSkipped() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, Skipped, 0);
        }

        READ_DECODER_VALUE(decoder, values, SourceType);

        if (decoder.GetMetaInfo() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, MetaInfo, Default<TString>());
        }

        return true;
    }

    TAutocodeFineMetaInfoAttachment::TAutocodeFineMetaInfoAttachment(const TString& url, const TInstant lastModifiedAt)
        : Url(url)
        , LastModifiedAt(lastModifiedAt)
    {
    }

    NJson::TJsonValue TAutocodeFineMetaInfoAttachment::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::Write(result, "url", Url);
        TJsonProcessor::WriteInstant(result, "last_modified_at", LastModifiedAt);
        return result;
    }

    bool TAutocodeFineMetaInfoAttachment::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TJsonProcessor::Read(data, "url", Url)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "last_modified_at", LastModifiedAt)) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue TAutocodeFineMetaInfoSessionBinding::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::Write(result, "session_id", SessionId);
        TJsonProcessor::Write(result, "user_id", UserId);
        TJsonProcessor::Write(result, "skipped", Skipped);
        TJsonProcessor::Write(result, "rebound_performer_id", ReboundPerformerId);
        TJsonProcessor::WriteInstant(result, "rebound_at", ReboundAt);
        return result;
    }

    bool TAutocodeFineMetaInfoSessionBinding::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TJsonProcessor::Read(data, "session_id", SessionId)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "user_id", UserId)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "skipped", Skipped)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "rebound_performer_id", ReboundPerformerId)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "rebound_at", ReboundAt)) {
            return false;
        }
        return true;
    }

    bool TAutocodeFineEntry::Parse(const NStorage::TTableRecord& record) {
        Id = record.Get("id");
        if (record.Get("serial_id") && !record.TryGet("serial_id", SerialId)) {
            return false;
        }
        RulingNumber = record.Get("ruling_number");
        if (record.Get("autocode_id") && !record.TryGet("autocode_id", AutocodeId)) {
            return false;
        }

        if (!record.Get("ruling_date") || !TInstant::TryParseIso8601(record.Get("ruling_date"), RulingDate)) {
            return false;
        }
        if (record.Get("violation_time") && !TInstant::TryParseIso8601(record.Get("violation_time"), ViolationTime)) {
            return false;
        }
        if (record.Get("discount_date") && !TInstant::TryParseIso8601(record.Get("discount_date"), DiscountDate)) {
            return false;
        }

        ArticleKoap = record.Get("article_koap");
        ViolationPlace = record.Get("violation_place");
        if (!record.Get("sum_to_pay") || !record.TryGet("sum_to_pay", SumToPay)) {
            return false;
        }

        OdpsCode = record.Get("odps_code");
        OdpsName = record.Get("odps_name");

        ViolationDocumentNumber = record.Get("violation_document_number");
        ViolationDocumentType = record.Get("violation_document_type");

        if (!record.Get("has_photo") || !record.TryGet("has_photo", HasPhoto)) {
            return false;
        }

        if (!record.Get("fine_information_received_at") || !TInstant::TryParseIso8601(record.Get("fine_information_received_at"), FineInformationReceivedAt)) {
            return false;
        }

        ui64 AddedAtTimestampSeconds;
        if (!record.Get("added_at_timestamp") || !record.TryGet("added_at_timestamp", AddedAtTimestampSeconds)) {
            return false;
        }
        AddedAtTimestamp = TInstant::Seconds(AddedAtTimestampSeconds);

        if (record.Get("autocode_payment_confirmation_id") && !record.TryGet("autocode_payment_confirmation_id", AutocodePaymentConfirmationId)) {
            return false;
        }
        if (record.Get("payment_confirmation_received_at") && !TInstant::TryParseIso8601(record.Get("payment_confirmation_received_at"), PaymentConfirmationReceivedAt)) {
            return false;
        }

        CarId = record.Get("car_id");
        OrderId = record.Get("order_id");
        UserId = record.Get("user_id");
        SessionId = record.Get("session_id");

        if (record.Get("charge_email_sent_at") && !TInstant::TryParseIso8601(record.Get("charge_email_sent_at"), ChargeEmailSentAt)) {
            return false;
        }
        if (record.Get("charge_sms_sent_at") && !TInstant::TryParseIso8601(record.Get("charge_sms_sent_at"), ChargeSmsSentAt)) {
            return false;
        }
        if (record.Get("charged_at") && !TInstant::TryParseIso8601(record.Get("charged_at"), ChargedAt)) {
            return false;
        }
        if (record.Get("charge_passed_at") && !TInstant::TryParseIso8601(record.Get("charge_passed_at"), ChargePassedAt)) {
            return false;
        }
        if (record.Get("charge_push_sent_at") && !TInstant::TryParseIso8601(record.Get("charge_push_sent_at"), ChargePushSentAt)) {
            return false;
        }

        if (!record.Get("needs_charge") || !record.TryGet("needs_charge", NeedsCharge)) {
            return false;
        }
        if (!record.Get("sum_to_pay_without_discount") || !record.TryGet("sum_to_pay_without_discount", SumToPayWithoutDiscount)) {
            return false;
        }

        if (record.Get("violation_longitude") && !record.TryGet("violation_longitude", ViolationLongitude)) {
            return false;
        }
        if (record.Get("violation_latitude") && !record.TryGet("violation_latitude", ViolationLatitude)) {
            return false;
        }

        if (!record.Get("is_after_ride_start_during_order") || !record.TryGet("is_after_ride_start_during_order", IsAfterRideStartDuringOrder)) {
            return false;
        }
        if (!record.Get("is_camera_fixation") || !record.TryGet("is_camera_fixation", IsCameraFixation)) {
            return false;
        }

        if (record.Get("skipped") && !record.TryGet("skipped", Skipped)) {
            return false;
        }

        SourceType = record.Get("source_type");
        MetaInfo = record.Get("meta_info");

        return true;
    }

    NStorage::TTableRecord TAutocodeFineEntry::SerializeToTableRecord() const {
        NStorage::TTableRecord result;

        result.Set("id", Id);
        result.Set("serial_id", (!!SerialId) ? ToString(SerialId) : "nextval('autocode_fine_serial_id_seq')");

        result.Set("ruling_number", RulingNumber);
        result.Set("autocode_id", (!!AutocodeId) ? ToString(AutocodeId) : "get_null()");

        result.Set("ruling_date", RulingDate.ToString());
        result.Set("violation_time", (!!ViolationTime) ? ViolationTime.ToString() : "get_null()");
        result.Set("discount_date", (!!DiscountDate) ? DiscountDate.ToString() : "get_null()");

        result.Set("article_koap", ArticleKoap);
        result.Set("violation_place", ViolationPlace);
        result.Set("sum_to_pay", SumToPay);

        result.Set("odps_code", (!!OdpsCode) ? OdpsCode : "get_null()");
        result.Set("odps_name", (!!OdpsName) ? OdpsName : "get_null()");

        result.Set("violation_document_number", ViolationDocumentNumber);
        result.Set("violation_document_type", ViolationDocumentType);

        result.Set("has_photo", HasPhoto);

        if (!!FineInformationReceivedAt && !!AddedAtTimestamp) {
            result.Set("fine_information_received_at", FineInformationReceivedAt.ToString());
            result.Set("added_at_timestamp", AddedAtTimestamp.Seconds());
        } else if (!!FineInformationReceivedAt) {
            result.Set("fine_information_received_at", FineInformationReceivedAt.ToString());
            result.Set("added_at_timestamp", FineInformationReceivedAt.Seconds());
        } else if (!!AddedAtTimestamp) {
            result.Set("fine_information_received_at", AddedAtTimestamp.ToString());
            result.Set("added_at_timestamp", AddedAtTimestamp.Seconds());
        } else {
            auto now = TInstant::Now();
            result.Set("fine_information_received_at", now.ToString());
            result.Set("added_at_timestamp", now.Seconds());
        }

        result.Set("autocode_payment_confirmation_id", (!!AutocodePaymentConfirmationId) ? ToString(AutocodePaymentConfirmationId) : "get_null()");
        result.Set("payment_confirmation_received_at", (!!PaymentConfirmationReceivedAt) ? PaymentConfirmationReceivedAt.ToString() : "get_null()");

        result.Set("car_id", CarId);
        result.Set("order_id", (!!OrderId) ? OrderId : "get_null()");
        result.Set("user_id", (!!UserId) ? UserId : "get_null()");
        result.Set("session_id", (!!SessionId) ? SessionId : "get_null()");

        result.Set("charge_email_sent_at", (!!ChargeEmailSentAt) ? ChargeEmailSentAt.ToString() : "get_null()");
        result.Set("charge_sms_sent_at", (!!ChargeSmsSentAt) ? ChargeSmsSentAt.ToString() : "get_null()");
        result.Set("charged_at", (!!ChargedAt) ? ChargedAt.ToString() : "get_null()");
        result.Set("charge_passed_at", (!!ChargePassedAt) ? ChargePassedAt.ToString() : "get_null()");
        result.Set("charge_push_sent_at", (!!ChargePushSentAt) ? ChargePushSentAt.ToString() : "get_null()");

        result.Set("needs_charge", NeedsCharge);
        result.Set("sum_to_pay_without_discount", SumToPayWithoutDiscount);

        if (!!ViolationLongitude && !!ViolationLatitude) {
            result.Set("violation_longitude", ViolationLongitude);
            result.Set("violation_latitude", ViolationLatitude);
        } else {
            result.Set("violation_longitude", "get_null()");
            result.Set("violation_latitude", "get_null()");
        }

        result.Set("is_after_ride_start_during_order", IsAfterRideStartDuringOrder);
        result.Set("is_camera_fixation", IsCameraFixation);

        result.Set("skipped", Skipped);

        result.Set("source_type", SourceType);
        result.Set("meta_info", (!!MetaInfo) ? MetaInfo : "get_null()");

        return result;
    }

    bool TAutocodeFineEntry::UpdateRecordHistoryInfo(NStorage::TTableRecord& record) const {
        record.ForceSet("serial_id", "nextval('autocode_fine_serial_id_seq')");
        record.ForceSet("added_at_timestamp", ToString(TInstant::Now().Seconds()));
        return true;
    }

    NJson::TJsonValue TAutocodeFineEntry::BuildReport(const NFineTraits::TReportTraits traits) const {
        NJson::TJsonValue data(NJson::JSON_MAP);

        data["id"] = Id;
        data["ruling_number"] = RulingNumber;

        data["ruling_date"] = FormatDate(RulingDate);
        data["violation_time"] = (!!ViolationTime) ? ViolationTime.SecondsFloat() : NJson::TJsonValue(NJson::JSON_NULL);

        data["article_koap"] = ArticleKoap;

        data["violation_place"] = ViolationPlace;
        data["sum_to_pay"] = (traits & NFineTraits::EReportTraits::UseCents) ? GetSumToPayCents() : GetSumToPay();

        NJson::TJsonValue& carInfo = data.InsertValue("car", NJson::JSON_MAP);
        carInfo["id"] = CarId;

        NJson::TJsonValue& serializedOrder = data.InsertValue("order", NJson::JSON_MAP);
        serializedOrder["id"] = (!!SessionId) ? SessionId : NJson::TJsonValue(NJson::JSON_NULL);
        serializedOrder["created_at"] = NJson::JSON_NULL;
        serializedOrder["completed_at"] = NJson::JSON_NULL;

        data["charge_passed_at"] = (!!ChargePassedAt) ? ChargePassedAt.SecondsFloat() : NJson::TJsonValue(NJson::JSON_NULL);

        if (traits & NFineTraits::EReportTraits::ReportChargeStatus) {
            data["needs_charge"] = NeedsCharge;
            data["charged_at"] = (!!ChargedAt) ? ChargedAt.SecondsFloat() : NJson::TJsonValue(NJson::JSON_NULL);
        }

        if (traits & NFineTraits::EReportTraits::ReportNotificationStatus) {
            data["charge_email_sent_at"] = (!!ChargeEmailSentAt) ? ChargeEmailSentAt.SecondsFloat() : NJson::TJsonValue(NJson::JSON_NULL);
            data["charge_sms_sent_at"] = (!!ChargeSmsSentAt) ? ChargeSmsSentAt.SecondsFloat() : NJson::TJsonValue(NJson::JSON_NULL);
            data["charge_push_sent_at"] = (!!ChargePushSentAt) ? ChargePushSentAt.SecondsFloat() : NJson::TJsonValue(NJson::JSON_NULL);
        }

        data["sum_to_pay_without_discount"] = (traits & NFineTraits::EReportTraits::UseCents) ? GetSumToPayWithoutDiscountCents() : GetSumToPayWithoutDiscount();

        data["violation_latitude"] = (!!ViolationLatitude) ? ViolationLatitude : NJson::TJsonValue(NJson::JSON_NULL);
        data["violation_longitude"] = (!!ViolationLongitude) ? ViolationLongitude : NJson::TJsonValue(NJson::JSON_NULL);

        if (traits & NFineTraits::EReportTraits::ReportPaymentConfirmationStatus) {
            data["payment_confirmation_received_at"] = (!!PaymentConfirmationReceivedAt) ? PaymentConfirmationReceivedAt.SecondsFloat() : NJson::TJsonValue(NJson::JSON_NULL);
        }

        if (traits & NFineTraits::EReportTraits::ReportOdpsInfo) {
            data["odps_name"] = OdpsName;
        }

        if (traits & NFineTraits::EReportTraits::ReportCameraFixationStatus) {
            data["is_camera_fixation"] = IsCameraFixation;
        }

        if (traits & NFineTraits::EReportTraits::ReportReceiveInfo) {
            data["info_received_at"] = FineInformationReceivedAt.SecondsFloat();
            data["source_type"] = SourceType;
        }

        return data;
    }

    bool TAutocodeFineEntry::DeserializeFromJson(const NJson::TJsonValue& data, const bool useCents) {
        JREAD_STRING_OPT(data, "id", Id);
        JREAD_UINT_OPT(data, "serial_id", SerialId);

        JREAD_STRING(data, "ruling_number", RulingNumber);
        JREAD_UINT_OPT(data, "autocode_id", AutocodeId);

        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "ruling_date", RulingDate);
        if (!RulingDate) {
            return false;
        }
        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "violation_time", ViolationTime);
        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "discount_date", DiscountDate);

        JREAD_STRING(data, "article_koap", ArticleKoap);
        JREAD_STRING_OPT(data, "violation_place", ViolationPlace);

        if (useCents) {
            if (!ReadCentsValue(data, "sum_to_pay", SumToPay)) {
                return false;
            }
        } else {
            JREAD_DOUBLE(data, "sum_to_pay", SumToPay);
        }

        JREAD_STRING_OPT(data, "odps_code", OdpsCode);
        JREAD_STRING_OPT(data, "odps_name", OdpsName);
        JREAD_STRING_OPT(data, "violation_document_number", ViolationDocumentNumber);
        JREAD_STRING_OPT(data, "violation_document_type", ViolationDocumentType);
        JREAD_BOOL_OPT(data, "has_photo", HasPhoto);

        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "fine_information_received_at", FineInformationReceivedAt);
        JREAD_INSTANT_OPT(data, "added_at_timestamp", AddedAtTimestamp);
        if (!FineInformationReceivedAt && !AddedAtTimestamp) {
            auto now = TInstant::Now();
            FineInformationReceivedAt = now;
            AddedAtTimestamp = now;
        } else if (!FineInformationReceivedAt) {
            FineInformationReceivedAt = AddedAtTimestamp;
        } else if (!AddedAtTimestamp) {
            AddedAtTimestamp = FineInformationReceivedAt;
        } else {
        }

        JREAD_UINT_OPT(data, "autocode_payment_confirmation_id", AutocodePaymentConfirmationId);
        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "payment_confirmation_received_at", PaymentConfirmationReceivedAt);

        JREAD_STRING(data, "car_id", CarId);
        JREAD_STRING_OPT(data, "user_id", UserId);
        // JREAD_STRING_OPT(data, "order_id", OrderId);  // completely deprecated
        JREAD_STRING_OPT(data, "session_id", SessionId);

        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "charge_email_sent_at", ChargeEmailSentAt);
        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "charge_sms_sent_at", ChargeSmsSentAt);
        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "charged_at", ChargedAt);
        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "charge_passed_at", ChargePassedAt);
        JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "charge_push_sent_at", ChargePushSentAt);

        JREAD_BOOL_OPT(data, "needs_charge", NeedsCharge);
        if (useCents) {
            if (!ReadCentsValue(data, "sum_to_pay_without_discount", SumToPayWithoutDiscount)) {
                return false;
            }
        } else {
            JREAD_DOUBLE(data, "sum_to_pay_without_discount", SumToPayWithoutDiscount);
        }

        JREAD_DOUBLE_NULLABLE_OPT(data, "violation_latitude", ViolationLatitude);
        JREAD_DOUBLE_NULLABLE_OPT(data, "violation_longitude", ViolationLongitude);

        // JREAD_BOOL_OPT(data, "is_after_ride_start_during_order", IsAfterRideStartDuringOrder);  // deprecated
        JREAD_BOOL_OPT(data, "is_camera_fixation", IsCameraFixation);
        JREAD_INT_OPT(data, "skipped", Skipped);

        JREAD_STRING_OPT(data, "source_type", SourceType);
        JREAD_STRING_OPT(data, "meta_info", MetaInfo);

        return true;
    }

    NJson::TJsonValue TAutocodeFineEntry::GetMetaInfoProperty(const TAutocodeFineEntry::EMetaInfoProperty property, const NJson::TJsonValue& defaultValue) const {
        return GetMetaInfoProperty(ToString(property), defaultValue);
    }

    TAutocodeFineEntry& TAutocodeFineEntry::SetMetaInfoProperty(const TAutocodeFineEntry::EMetaInfoProperty property, NJson::TJsonValue&& value) {
        return SetMetaInfoProperty(ToString(property), std::move(value));
    }

    bool TAutocodeFineEntry::HasSessionRebounds() const {
        NJson::TJsonValue previousSessionBindings = GetMetaInfoProperty(EMetaInfoProperty::PreviousSessionBindings);
        return !previousSessionBindings.GetArray().empty();
    }

    bool TAutocodeFineEntry::DropSessionBinding(const TString& performerId) {
        NJson::TJsonValue previousSessionBindings = GetMetaInfoProperty(EMetaInfoProperty::PreviousSessionBindings, NJson::JSON_ARRAY);

        TMetaInfoSessionBinding metaInfoSessionBinding;
        metaInfoSessionBinding.SetReboundPerformerId(performerId).SetReboundAt(ModelingNow());
        metaInfoSessionBinding.SetSessionId(SessionId).SetUserId(UserId).SetSkipped(Skipped);
        previousSessionBindings.AppendValue(metaInfoSessionBinding.SerializeToJson());

        SetMetaInfoProperty(EMetaInfoProperty::PreviousSessionBindings, std::move(previousSessionBindings));

        NeedsCharge = false;

        SessionId = Default<TString>();
        UserId = Default<TString>();
        Skipped = 0;

        // values below can be restored via related tags field
        ChargedAt = Default<TInstant>();
        ChargePassedAt = Default<TInstant>();
        ChargeEmailSentAt = Default<TInstant>();
        ChargePushSentAt = Default<TInstant>();
        ChargeSmsSentAt = Default<TInstant>();

        SetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::ChargeTagId, NJson::JSON_NULL);

        return true;
    }

    TString TAutocodeFineEntry::GetArticleCode() const {
        auto rawValue = GetMetaInfoProperty(EMetaInfoProperty::ArticleCode);
        return (rawValue.IsDefined()) ? rawValue.GetString() : "";
    }

    TAutocodeFineEntry& TAutocodeFineEntry::SetArticleCode(const TString& articleCode) {
        return SetMetaInfoProperty(EMetaInfoProperty::ArticleCode, articleCode);
    }

    bool TAutocodeFineEntry::GetHasDecree(const bool defaultValue) const {
        auto rawValue = GetMetaInfoProperty(EMetaInfoProperty::HasDecree);
        return (rawValue.IsDefined()) ? rawValue.GetBoolean() : defaultValue;
    }

    TAutocodeFineEntry& TAutocodeFineEntry::SetHasDecree(const bool hasDecree) {
        return SetMetaInfoProperty(EMetaInfoProperty::HasDecree, hasDecree);
    }

    bool TAutocodeFineEntry::GetMetaInfoAttachments(TVector<TMetaInfoAttachment>& attachments) const {
        auto existingAttachmentsJson = GetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::Attachments, NJson::JSON_ARRAY);

        for (auto&& metaInfoAttachmentJson : existingAttachmentsJson.GetArray()) {
            TMetaInfoAttachment metaInfoAttachment;
            if (!metaInfoAttachment.DeserializeFromJson(metaInfoAttachmentJson)) {
                return false;
            }

            attachments.push_back(std::move(metaInfoAttachment));
        }

        return true;
    }

    bool TAutocodeFineEntry::GenerateViolationDetailedDocumentFilePath(const TString& filePrefix, TString& filePath) const {
        if (!SessionId) {
            return false;
        }
        filePath = (TStringBuilder() << ((!!filePrefix) ? filePrefix + "/" : "") << SessionId << "/" << NUtil::CreateUUID() << ".pdf");
        return true;
    }

    bool TAutocodeFineEntry::GetCachedViolationDetailedDocumentUrl(TString& fileUrl, const TString& defaultValue) const {
        // return false in case of errors only and true if both found or not found
        // specific attachments table usage is required thereafter (along with extra session id check)
        if (!SessionId) {
            return false;
        }

        TVector<TMetaInfoAttachment> attachments;
        if (!GetMetaInfoAttachments(attachments)) {
            return false;
        }

        // NB: extra check on default value equality is required
        fileUrl = defaultValue;

        if (!attachments.empty()) {
            TString attachmentUrl = attachments.front().GetUrl();
            if (attachmentUrl.Contains(SessionId)) {
                // refer to GenerateViolationDetailedDocumentFilePath method; session id is supposed to present in file name
                // NB: all attachments are to be stored in a separate table (photos, decrees),
                //     therefore violation detailed documents are only ones that are located in meta info and have to be moved or removed completely
                fileUrl = attachmentUrl;
            }
        }

        return true;
    }

    i64 TAutocodeFineEntry::GetSumToPayCents() const {
        return ConvertToCents<i64>(SumToPay);
    }
    i64 TAutocodeFineEntry::GetSumToPayWithoutDiscountCents() const {
        return ConvertToCents<i64>(SumToPayWithoutDiscount);
    }

    NJson::TJsonValue TAutocodeFineEntry::GetMetaInfoProperty(const TString& path, const NJson::TJsonValue& defaultValue) const {
        if (!!MetaInfo) {
            NJson::TJsonValue jsonMetaInfo;
            if (NJson::ReadJsonTree(MetaInfo, &jsonMetaInfo)) {
                const NJson::TJsonValue* value = jsonMetaInfo.GetValueByPath(path);
                if (value != nullptr) {
                    return *value;
                }
            }
        }
        return defaultValue;
    }

    TAutocodeFineEntry& TAutocodeFineEntry::SetMetaInfoProperty(const TString& path, NJson::TJsonValue&& value) {
        NJson::TJsonMap jsonMetaInfo;

        if (!!MetaInfo) {
            if (!NJson::ReadJsonTree(MetaInfo, &jsonMetaInfo)) {
                jsonMetaInfo["_old_meta_info_"] = MetaInfo;
            }
        }

        jsonMetaInfo.SetValueByPath(path, std::move(value));
        MetaInfo = NJson::WriteJson(jsonMetaInfo, /* formatOutput = */ false);

        return *this;
    }

    TAutocodeFineAttachmentDecoder::TAutocodeFineAttachmentDecoder(const TMap<TString, ui32>& decoderBase) {
        Id = GetFieldDecodeIndex("id", decoderBase);
        SerialId = GetFieldDecodeIndex("serial_id", decoderBase);
        LastModifiedAt = GetFieldDecodeIndex("last_modified_at", decoderBase);
        FineId = GetFieldDecodeIndex("fine_id", decoderBase);
        Url = GetFieldDecodeIndex("url", decoderBase);
        DataType = GetFieldDecodeIndex("data_type", decoderBase);
    }

    TAutocodeFineAttachmentEntry::TEventId TAutocodeFineAttachmentEntry::GetHistoryEventId() const {
        return SerialId;
    }

    const TInstant& TAutocodeFineAttachmentEntry::GetHistoryInstant() const {
        return LastModifiedAt;
    }

    const TString& TAutocodeFineAttachmentEntry::GetHistoryUserId() const {
        return Default<TString>();
    }

    bool TAutocodeFineAttachmentEntry::operator < (const TAutocodeFineAttachmentEntry& other) const {
        return SerialId < other.GetSerialId();
    }

    bool TAutocodeFineAttachmentEntry::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /* hContext */) {
        READ_DECODER_VALUE(decoder, values, Id);
        if (decoder.GetSerialId() >= 0) {
            READ_DECODER_VALUE(decoder, values, SerialId);
        }
        if (decoder.GetLastModifiedAt() >= 0) {
            READ_DECODER_VALUE_INSTANT(decoder, values, LastModifiedAt);
        }
        READ_DECODER_VALUE(decoder, values, FineId);
        READ_DECODER_VALUE(decoder, values, Url);
        if (decoder.GetDataType() >= 0) {
            READ_DECODER_VALUE_DEF(decoder, values, DataType, EDataType::Unknown);
        }
        return true;
    }

    TAutocodeFineAttachmentEntry::TAutocodeFineAttachmentEntry(const TString& fineId)
        : TAutocodeFineAttachmentEntry(NUtil::CreateUUID(), fineId)
    {
    }

    TAutocodeFineAttachmentEntry::TAutocodeFineAttachmentEntry(const TString& id, const TString& fineId)
        : Id(id)
        , FineId(fineId)
    {
    }

    bool TAutocodeFineAttachmentEntry::Parse(const NStorage::TTableRecord& record) {
        Id = record.Get("id");
        if (record.Get("serial_id") && !record.TryGet("serial_id", SerialId)) {
            return false;
        }
        if (record.Get("last_modified_at") && !record.TryGet("last_modified_at", LastModifiedAt)) {
            return false;
        }
        FineId = record.Get("fine_id");
        Url = record.Get("url");
        if (record.Get("data_type")) {
            if (!record.TryGet("data_type", DataType)) {
                WARNING_LOG << "Unknown autocode fine attachment type: " << record.Get("data_type") << Endl;
            }
        }
        return true;
    }

    NStorage::TTableRecord TAutocodeFineAttachmentEntry::SerializeToTableRecord() const {
        NStorage::TTableRecord result;
        result.Set("id", Id);
        result.Set("serial_id", (!!SerialId) ? ToString(SerialId) : "nextval('autocode_fine_photo_serial_id_seq')");
        result.Set("last_modified_at", (!!LastModifiedAt) ? LastModifiedAt.Seconds() : ModelingNow().Seconds());
        result.Set("fine_id", FineId);
        result.Set("url", Url);
        result.Set("data_type", ToString(DataType));
        return result;
    }

    bool TAutocodeFineAttachmentEntry::UpdateRecordHistoryInfo(NStorage::TTableRecord& record) const  {
        record.ForceSet("serial_id", "nextval('autocode_fine_photo_serial_id_seq')");
        record.ForceSet("last_modified_at", ToString(ModelingNow().Seconds()));
        return true;
    }

    NJson::TJsonValue TAutocodeFineAttachmentEntry::BuildReport() const {
        NJson::TJsonValue data(NJson::JSON_MAP);
        data["id"] = Id;
        data["url"] = Url;
        return data;
    }

    TString TAutocodeFineAttachmentEntry::GetAttachmentDefaultRelativePath(const TString& fineId, const TString& attachmentId, const TString& prefix, const TString& extension) {
        return TStringBuilder() << ((!!prefix) ? prefix + "/" : "") << fineId << "/" << attachmentId << extension;
    }

    TString TAutocodeFineAttachmentEntry::GetTableName() {
        return "autocode_fine_photo";
    }
}
