#pragma once

#include <drive/backend/fines/constants.h>

#include <drive/backend/database/history/common.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>

#include <util/datetime/base.h>

namespace NDrive::NFine {
    namespace NFineTraits {
        using TReportTraits = ui64;

        static const TReportTraits NoTraits = 0;

        enum EReportTraits: ui64 {
            ReportNotificationStatus = 1 << 0,
            ReportChargeStatus = 1 << 1,
            ReportPaymentConfirmationStatus = 1 << 2,

            UseCents = 1 << 3,

            ReportReceiveInfo = 1 << 4,
            ReportOdpsInfo = 1 << 5,
            ReportCameraFixationStatus = 1 << 6,
        };

        static constexpr TReportTraits ReportAll = (
            EReportTraits::ReportNotificationStatus | EReportTraits::ReportChargeStatus | EReportTraits::ReportPaymentConfirmationStatus |
            EReportTraits::ReportReceiveInfo | EReportTraits::ReportOdpsInfo | EReportTraits::ReportCameraFixationStatus
        );
    }

    namespace NFineChargeStatusTraits {
        using TChargeStatusTraits = ui64;

        enum EChargeStatusReason : ui64 {
            Unknown = 0 /* "unknown" */,

            ExplicitChargeArticle = 1 << 0 /* "explicit_charge_article" */,
            CameraFixation = 1 << 1 /* "camera_fixation" */,
            NoCameraFixation = 1 << 2 /* "no_camera_fixation" */,

            ExplicitDismissalArticle = 1 << 3 /* "explicit_dismissal_article" */,
            ExplicitDismissalCar = 1 << 4 /* "explicit_dismissal_car" */,
            ExplicitDismissalOfferName = 1 << 5 /* "explicit_dismissal_offer_name" */,
            NoBinding = 1 << 6 /* "no_binding" */,
            SkipLimitReached = 1 << 7 /* "skip_limit_reached" */,
            InvalidAmount = 1 << 8 /* "invalid_amount" */,
            InvalidDiscount = 1 << 9 /* "invalid_discount" */,
            NotRecognizedArticle = 1 << 10 /* "not_recognized_article" */,  // not empty but not matched or does not present in recognizable articles
            UnknownArticle = 1 << 11 /* "unknown_article" */,  // empty

            InappropriateCityParking = 1 << 12 /* "inappropriate_city_parking" */,

            ManuallyPaidDuplicate = 1 << 13 /* "manually_paid_duplicate" */,

            InappropriateLongTermParking = 1 << 14 /* "inappropriate_long_term_parking" */,

            BoundChargeableTags = 1 << 15 /* "bound_chargeable_tags" */,
        };

        constexpr bool DoNeedCharge(const TChargeStatusTraits traits) noexcept {
            bool needCharge = false;
            if (!needCharge && traits & EChargeStatusReason::CameraFixation) {
                needCharge = 0 == (traits &  // allow rule itself
                                   ~EChargeStatusReason::CameraFixation);
            }
            if (!needCharge && traits & EChargeStatusReason::ExplicitChargeArticle) {
                needCharge = 0 == (traits &  // allow any combination of rules below
                                   ~EChargeStatusReason::ExplicitChargeArticle &
                                   ~EChargeStatusReason::CameraFixation &
                                   ~EChargeStatusReason::NoCameraFixation);
            }
            if (!needCharge && traits & EChargeStatusReason::InappropriateCityParking) {
                needCharge = 0 == (traits &  // allow any combination of rules below
                                   ~EChargeStatusReason::InappropriateCityParking &
                                   ~EChargeStatusReason::CameraFixation &
                                   ~EChargeStatusReason::NoCameraFixation &
                                   ~EChargeStatusReason::ExplicitChargeArticle &
                                   ~EChargeStatusReason::NotRecognizedArticle &
                                   ~EChargeStatusReason::UnknownArticle);
            }
            if (!needCharge && traits & EChargeStatusReason::InappropriateLongTermParking) {
                needCharge = 0 == (traits &  // allow any combination of rules below
                                   ~EChargeStatusReason::InappropriateLongTermParking &
                                   ~EChargeStatusReason::CameraFixation &
                                   ~EChargeStatusReason::NoCameraFixation &
                                   ~EChargeStatusReason::ExplicitChargeArticle &
                                   ~EChargeStatusReason::NotRecognizedArticle &
                                   ~EChargeStatusReason::UnknownArticle &
                                   ~EChargeStatusReason::InappropriateCityParking);
            }
            if (!needCharge && traits & EChargeStatusReason::BoundChargeableTags) {
                needCharge = 0 == (traits &  // allow any combination of rules below
                                   ~EChargeStatusReason::BoundChargeableTags &
                                   ~EChargeStatusReason::InappropriateLongTermParking &
                                   ~EChargeStatusReason::CameraFixation &
                                   ~EChargeStatusReason::NoCameraFixation &
                                   ~EChargeStatusReason::ExplicitChargeArticle &
                                   ~EChargeStatusReason::NotRecognizedArticle &
                                   ~EChargeStatusReason::UnknownArticle &
                                   ~EChargeStatusReason::InappropriateCityParking);
            }
            return needCharge;
        }
    }

    class TAutocodeFineDecoder : public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, SerialId, -1);
        R_FIELD(i32, RulingNumber, -1);
        R_FIELD(i32, AutocodeId, -1);
        R_FIELD(i32, RulingDate, -1);
        R_FIELD(i32, ViolationTime, -1);
        R_FIELD(i32, DiscountDate, -1);
        R_FIELD(i32, ArticleKoap, -1);
        R_FIELD(i32, ViolationPlace, -1);
        R_FIELD(i32, SumToPay, -1);
        R_FIELD(i32, OdpsCode, -1);
        R_FIELD(i32, OdpsName, -1);
        R_FIELD(i32, ViolationDocumentNumber, -1);
        R_FIELD(i32, ViolationDocumentType, -1);
        R_FIELD(i32, HasPhoto, -1);
        R_FIELD(i32, FineInformationReceivedAt, -1);
        R_FIELD(i32, AddedAtTimestamp, -1);
        R_FIELD(i32, AutocodePaymentConfirmationId, -1);
        R_FIELD(i32, PaymentConfirmationReceivedAt, -1);
        R_FIELD(i32, CarId, -1);
        R_FIELD(i32, OrderId, -1);
        R_FIELD(i32, SessionId, -1);
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, ChargeEmailSentAt, -1);
        R_FIELD(i32, ChargeSmsSentAt, -1);
        R_FIELD(i32, ChargedAt, -1);
        R_FIELD(i32, ChargePassedAt, -1);
        R_FIELD(i32, ChargePushSentAt, -1);
        R_FIELD(i32, NeedsCharge, -1);
        R_FIELD(i32, SumToPayWithoutDiscount, -1);
        R_FIELD(i32, ViolationLatitude, -1);
        R_FIELD(i32, ViolationLongitude, -1);
        R_FIELD(i32, IsAfterRideStartDuringOrder, -1);
        R_FIELD(i32, IsCameraFixation, -1);
        R_FIELD(i32, Skipped, -1);
        R_FIELD(i32, SourceType, -1);
        R_FIELD(i32, MetaInfo, -1);

    public:
        TAutocodeFineDecoder() = default;
        TAutocodeFineDecoder(const TMap<TString, ui32>& decoderBase);
    };

    class TAutocodeFineMetaInfoAttachment {
        R_FIELD(TString, Url);
        R_FIELD(TInstant, LastModifiedAt);

    public:
        TAutocodeFineMetaInfoAttachment() = default;
        explicit TAutocodeFineMetaInfoAttachment(const TString& url, const TInstant lastModifiedAt = TInstant::Now());

        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeFromJson(const NJson::TJsonValue& data);
    };

    class TAutocodeFineMetaInfoSessionBinding {
        R_FIELD(TString, SessionId);
        R_FIELD(TString, UserId);
        R_FIELD(i32, Skipped, 0);

        R_FIELD(TString, ReboundPerformerId);
        R_FIELD(TInstant, ReboundAt);

    public:
        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeFromJson(const NJson::TJsonValue& data);
    };

    class TAutocodeFineEntry {
        R_FIELD(TString, Id);
        R_FIELD(ui32, SerialId, 0);

        R_FIELD(TString, RulingNumber);
        R_FIELD(i64, AutocodeId, DefaultAutocodeId);

        R_FIELD(TInstant, RulingDate, Default<TInstant>());
        R_FIELD(TInstant, ViolationTime, Default<TInstant>());
        R_FIELD(TInstant, DiscountDate, Default<TInstant>());
        R_FIELD(TString, ArticleKoap);
        R_FIELD(TString, ViolationPlace);
        R_FIELD(double, SumToPay, 0);
        R_FIELD(TString, OdpsCode);
        R_FIELD(TString, OdpsName);
        R_FIELD(TString, ViolationDocumentNumber);
        R_FIELD(TString, ViolationDocumentType, "STS");
        R_FIELD(bool, HasPhoto, false);

        R_FIELD(TInstant, FineInformationReceivedAt, Default<TInstant>());
        R_FIELD(TInstant, AddedAtTimestamp, Default<TInstant>());

        R_FIELD(i64, AutocodePaymentConfirmationId, 0);
        R_FIELD(TInstant, PaymentConfirmationReceivedAt, Default<TInstant>());

        R_FIELD(TString, CarId);
        R_FIELD(TString, OrderId);       // deprecated

        R_FIELD(TString, SessionId);
        R_FIELD(TString, UserId);

        R_FIELD(TInstant, ChargeEmailSentAt, Default<TInstant>());
        R_FIELD(TInstant, ChargeSmsSentAt, Default<TInstant>());
        R_FIELD(TInstant, ChargedAt, Default<TInstant>());
        R_FIELD(TInstant, ChargePassedAt, Default<TInstant>());
        R_FIELD(TInstant, ChargePushSentAt, Default<TInstant>());

        R_FIELD(bool, NeedsCharge, false);
        R_FIELD(double, SumToPayWithoutDiscount, 0);

        R_FIELD(double, ViolationLatitude, DefaultViolationLatitude);
        R_FIELD(double, ViolationLongitude, DefaultViolationLongitude);

        R_FIELD(bool, IsAfterRideStartDuringOrder, false);       // deprecated


        R_FIELD(bool, IsCameraFixation, false);
        R_FIELD(int, Skipped, 0);

        R_FIELD(TString, SourceType, "autocode");
        R_FIELD(TString, MetaInfo);

    public:
        enum class EMetaInfoProperty {
            ArticleCode /* "article_code" */,
            ChargeTagId /* "tag_id" */,
            RelatedTagIds /* "related_tag_ids" */,
            Attachments /* "attachments" */,
            PreviousSessionBindings /* "previous_session_bindings" */,
            HasDecree /* "has_decree" */,
            // ChargeStatusReason /* "charge_status_reason" */,  // deprecated, refer to ChargeStatusTraits instead
            ChargeStatusTraits /* "charge_status_traits" */,
            RelatedOfferNames /* "related_offer_names" */,
            WillBeIncludedToBill /* "will_be_included_to_bill" */,
            RulingNumberAliases /* "ruling_number_aliases" */,
            BoundCarTags /* "bound_car_tags" */,
            PaymentData /* "payment_data" */,
        };

        using TDecoder = TAutocodeFineDecoder;
        using TMetaInfoAttachment = TAutocodeFineMetaInfoAttachment;
        using TMetaInfoSessionBinding = TAutocodeFineMetaInfoSessionBinding;

        using TEventId = NDrive::TEventId;
        static constexpr auto IncorrectEventId = NDrive::IncorrectEventId;

        static constexpr i64 DefaultAutocodeId = 0;
        static constexpr double DefaultViolationLatitude = 0.0;
        static constexpr double DefaultViolationLongitude = 0.0;

        TAutocodeFineEntry();
        explicit TAutocodeFineEntry(const TString& id);

        TEventId GetHistoryEventId() const;
        const TInstant& GetHistoryInstant() const;
        const TString& GetHistoryUserId() const;
        bool operator < (const TAutocodeFineEntry& other) const;

        constexpr bool HasBinding() const noexcept {
            return !!SessionId && !!UserId;
        }

        NJson::TJsonValue GetMetaInfoProperty(const EMetaInfoProperty property, const NJson::TJsonValue& defaultValue = NJson::JSON_UNDEFINED) const;
        TAutocodeFineEntry& SetMetaInfoProperty(const EMetaInfoProperty property, NJson::TJsonValue&& value);

        bool HasSessionRebounds() const;
        bool DropSessionBinding(const TString& performerId);

        TString GetArticleCode() const;
        TAutocodeFineEntry& SetArticleCode(const TString& articleCode);

        bool GetHasDecree(const bool defaultValue = false) const;
        TAutocodeFineEntry& SetHasDecree(const bool hasDecree);

        bool GenerateViolationDetailedDocumentFilePath(const TString& filePrefix, TString& filePath) const;
        bool GetCachedViolationDetailedDocumentUrl(TString& fileUrl, const TString& defaultValue) const;

        i64 GetSumToPayCents() const;
        i64 GetSumToPayWithoutDiscountCents() const;

        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /* hContext */);

        bool Parse(const NStorage::TTableRecord& record);
        NStorage::TTableRecord SerializeToTableRecord() const;
        bool UpdateRecordHistoryInfo(NStorage::TTableRecord& record) const;

        NJson::TJsonValue BuildReport(const NFineTraits::TReportTraits traits = NFineTraits::NoTraits) const;
        bool DeserializeFromJson(const NJson::TJsonValue& data, const bool useCents = true);

    private:
        NJson::TJsonValue GetMetaInfoProperty(const TString& path, const NJson::TJsonValue& defaultValue = NJson::JSON_UNDEFINED) const;
        TAutocodeFineEntry& SetMetaInfoProperty(const TString& path, NJson::TJsonValue&& value);

        bool GetMetaInfoAttachments(TVector<TMetaInfoAttachment>& attachments) const;
    };

    class TAutocodeFineAttachmentDecoder : public TBaseDecoder {
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, SerialId, -1);
        R_FIELD(i32, LastModifiedAt, -1);

        R_FIELD(i32, FineId, -1);
        R_FIELD(i32, Url, -1);
        R_FIELD(i32, DataType, -1);

    public:
        TAutocodeFineAttachmentDecoder() = default;
        TAutocodeFineAttachmentDecoder(const TMap<TString, ui32>& decoderBase);
    };

    class TAutocodeFineAttachmentEntry {
    public:
        enum class EDataType {
            All /* "all" */,  // for internal use only

            Photo /* "photo" */,
            Decree /* "decree" */,

            Unknown /* "unknown" */  // for internal use only
        };

        R_FIELD(TString, Id);
        R_FIELD(ui64, SerialId, 0);
        R_FIELD(TInstant, LastModifiedAt);

        R_FIELD(TString, FineId);
        R_FIELD(TString, Url);
        R_FIELD(EDataType, DataType, EDataType::Unknown);

    public:
        using TDecoder = TAutocodeFineAttachmentDecoder;
        using TEventId = NDrive::TEventId;

        static constexpr auto IncorrectEventId = NDrive::IncorrectEventId;

    public:
        TEventId GetHistoryEventId() const;
        const TInstant& GetHistoryInstant() const;
        const TString& GetHistoryUserId() const;
        bool operator < (const TAutocodeFineAttachmentEntry& other) const;

        bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /* hContext */);

        explicit TAutocodeFineAttachmentEntry(const TString& fineId = "");
        TAutocodeFineAttachmentEntry(const TString& id, const TString& fineId);

        bool Parse(const NStorage::TTableRecord& record);
        NStorage::TTableRecord SerializeToTableRecord() const;
        bool UpdateRecordHistoryInfo(NStorage::TTableRecord& record) const;

        NJson::TJsonValue BuildReport() const;

        static TString GetAttachmentDefaultRelativePath(const TString& fineId, const TString& attachmentId, const TString& prefix = "", const TString& extension = "");
        static TString GetTableName();
    };
}
