#include "config.h"

#include "scheme_util.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/cars/car.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/json/json_value.h>

#include <rtline/library/json/builder.h>
#include <rtline/library/json/merge.h>
#include <rtline/library/json/parse.h>
#include <rtline/util/json_processing.h>

#include <util/generic/algorithm.h>
#include <util/string/cast.h>
#include <util/string/join.h>
#include <util/string/split.h>

namespace {
    template <typename TContainer, typename TKeyGetter>
    bool ReadObjectsContainerAsMapping(const NJson::TJsonValue& jsonInfo, const TString& fieldName, TContainer& result, const TKeyGetter& keyGetter, const bool mustBe = false, const bool skipErrors = false) {
        if (jsonInfo.Has(fieldName) && !jsonInfo[fieldName].IsArray()) {
            return !mustBe;
        }
        for (const auto& jsonItem : jsonInfo[fieldName].GetArray()) {
            typename TContainer::mapped_type item;
            if (!item.DeserializeFromJson(jsonItem)) {
                if (!skipErrors) {
                    return false;
                }
            } else {
                auto&& key = keyGetter(item);
                result.emplace(key, std::move(item));
            }
        }
        return true;
    }

    template <typename TContainer>
    void WriteObjectsContainer(NJson::TJsonValue& target, const TString& fieldName, const TContainer& result, const bool writeEmpty = true) {
        if (!writeEmpty && result.empty()) {
            return;
        }
        CHECK_WITH_LOG(target.IsMap() || !target.IsDefined());
        Y_ASSERT(!target.Has(fieldName));
        auto& jsonArray = target.InsertValue(fieldName, NJson::JSON_ARRAY);
        for (auto&& i : result) {
            jsonArray.AppendValue(i.SerializeToJson());
        }
    }

    template <typename TContainer>
    void WriteMappedObjectsContainer(NJson::TJsonValue& target, const TString& fieldName, const TContainer& result, const bool writeEmpty = true) {
        if (!writeEmpty && result.empty()) {
            return;
        }
        CHECK_WITH_LOG(target.IsMap() || !target.IsDefined());
        Y_ASSERT(!target.Has(fieldName));
        auto& jsonArray = target.InsertValue(fieldName, NJson::JSON_ARRAY);
        for (auto&& [_, i] : result) {
            jsonArray.AppendValue(i.SerializeToJson());
        }
    }

    class TSettingDefaultsBuilder: public NJson::TBaseBuilder {
        using TBase = NJson::TBaseBuilder;

    public:
        explicit TSettingDefaultsBuilder(const TString& settingPrefix, const IServerBase* server = nullptr)
            : TBase()
            , SettingPrefix(settingPrefix)
            , Server(server)
        {
            if (!Server && NDrive::HasServer()) {
                Server = &NDrive::GetServer();
            }
        }

        template <typename T>
        TSettingDefaultsBuilder& Add(const TString& key) {
            T value;
            if (!!Server && GetSettingValue(*Server, key, value)) {
                Object[key] = std::move(value);
            }
            return *this;
        }

        template <typename T>
        TSettingDefaultsBuilder& Add(const TString& key, const T& defaultValue) {
            T value;
            if (!!Server && GetSettingValue(*Server, key, value)) {
                Object[key] = std::move(value);
            } else {
                Object[key] = defaultValue;
            }
            return *this;
        }

        template <typename T>
        TSettingDefaultsBuilder& Add(const TString& key, T&& defaultValue) {
            T value;
            if (!!Server && GetSettingValue(*Server, key, value)) {
                Object[key] = std::move(value);
            } else {
                Object[key] = std::forward<T>(defaultValue);
            }
            return *this;
        }

        TSettingDefaultsBuilder& AddJson(const TString& key) {
            return Add<NJson::TJsonValue>(key);
        }

        TSettingDefaultsBuilder& AddJson(const TString& key, const NJson::TJsonValue& defaultValue) {
            return Add<NJson::TJsonValue>(key, defaultValue);
        }

        TSettingDefaultsBuilder& AddJson(const TString& key, NJson::TJsonValue&& defaultValue) {
            return Add<NJson::TJsonValue>(key, std::forward<NJson::TJsonValue>(defaultValue));
        }

    private:
        template <typename T>
        bool GetSettingValue(const IServerBase& server, const TString& key, T& value) const {
            if (!!key && server.HasSettings()) {
                auto settingKey = JoinSeq(".", { SettingPrefix, key });
                if (server.GetSettings().GetValue<T>(settingKey, value)) {
                    return true;
                }
            }
            return false;
        }

        template <>
        inline bool GetSettingValue(const IServerBase& server, const TString& key, NJson::TJsonValue& value) const {
            TString valueStr;
            return GetSettingValue(server, key, valueStr) && NJson::ReadJsonTree(valueStr, &value);
        }

        const TString SettingPrefix;
        const IServerBase* Server;
    };
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& data, NDrive::NFine::TFineConstructorConfig::TDismissedCarInfo& result) {
    return result.DeserializeFromJson(data);
}

template <>
NJson::TJsonValue NJson::ToJson(const NDrive::NFine::TFineConstructorConfig::TDismissedCarInfo& object) {
    return object.SerializeToJson();
}

namespace NDrive::NFine {
    TBaseRegexpMatchRuleConfig::TBaseRegexpMatchRuleConfig(const TString& name)
        : Name(name)
    {
    }

    TBaseRegexpMatchRuleConfig::TBaseRegexpMatchRuleConfig(const TString& name, const TString& pattern, const bool requireOneNeedle)
        : Name(name)
        , Pattern(pattern)
        , RequireOneNeedle(requireOneNeedle)
    {
    }

    bool TBaseRegexpMatchRuleConfig::operator ==(const TBaseRegexpMatchRuleConfig& other) const {
        return std::make_tuple(Name, Pattern, RequireOneNeedle) == std::make_tuple(other.Name, other.Pattern, other.RequireOneNeedle);
    }

    NDrive::TScheme TBaseRegexpMatchRuleConfig::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("name", "Название регулярного выражения").SetRequired(true);
        scheme.Add<TFSString>("pattern", "Значение регулярного выражения").SetRequired(true);
        scheme.Add<TFSBoolean>("require_one_needle", "Требовать только одно вхождение").SetDefault(false);
        return scheme;
    }

    bool TBaseRegexpMatchRuleConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TJsonProcessor::Read(data, "name", Name) || !Name) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "pattern", Pattern, /* mustBe = */ true)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "require_one_needle", RequireOneNeedle)) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue TBaseRegexpMatchRuleConfig::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::Write(result, "name", Name);
        TJsonProcessor::Write(result, "pattern", Pattern);
        TJsonProcessor::Write(result, "require_one_needle", RequireOneNeedle);
        return result;
    }

    TSimpleStringRegexpMatchRuleConfig::TSimpleStringRegexpMatchRuleConfig(const TString& name)
        : TBase(name)
    {
    }

    TSimpleStringRegexpMatchRuleConfig::TSimpleStringRegexpMatchRuleConfig(const TString& name, const TString& pattern, const TString& matchResult, const bool requireOneNeedle)
        : TBase(name, pattern, requireOneNeedle)
        , MatchResult(matchResult)
    {
    }

    bool TSimpleStringRegexpMatchRuleConfig::operator ==(const TSimpleStringRegexpMatchRuleConfig& other) const {
        return TBase::operator==(other) && MatchResult == other.MatchResult;
    }

    NDrive::TScheme TSimpleStringRegexpMatchRuleConfig::GetScheme(const IServerBase& server) {
        NDrive::TScheme scheme = TBase::GetScheme(server);
        scheme.Add<TFSString>("match_result", "Значение в случае матчинга").SetRequired(true);
        return scheme;
    }

    bool TSimpleStringRegexpMatchRuleConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TBase::DeserializeFromJson(data)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "match_result", MatchResult, /* mustBe = */ true)) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue TSimpleStringRegexpMatchRuleConfig::SerializeToJson() const {
        NJson::TJsonValue result = TBase::SerializeToJson();
        TJsonProcessor::Write(result, "match_result", MatchResult);
        return result;
    }

    const TString TGenericMatchRuleConfig::TypeName = "generic";

    TGenericMatchRuleConfig::TGenericMatchRuleConfig()
        : TBase(TypeName)
    {
    }

    TGenericMatchRuleConfig::TGenericMatchRuleConfig(const TString& pattern)
        : TBase(TypeName, pattern, true)
    {
    }

    TFineArticleMatcherConfig::TReadableArticle::TReadableArticle(const TString& articleCode, const TString& articleTitle)
        : ArticleCode(articleCode)
        , ArticleTitle(articleTitle)
    {
    }

    bool TFineArticleMatcherConfig::TReadableArticle::operator ==(const TReadableArticle& other) const {
        return std::make_tuple(ArticleCode, ArticleTitle) == std::make_tuple(other.ArticleCode, other.ArticleTitle);
    }

    NDrive::TScheme TFineArticleMatcherConfig::TReadableArticle::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("article_code", "Статья штрафа (напр. 8_25, 12_09_3)").SetRequired(true);
        scheme.Add<TFSString>("article_title", "Описание").SetRequired(true);
        return scheme;
    }

    bool TFineArticleMatcherConfig::TReadableArticle::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TJsonProcessor::Read(data, "article_code", ArticleCode, /* mustBe = */ true) || !ArticleCode) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "article_title", ArticleTitle, /* mustBe = */ true) || !ArticleTitle) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue TFineArticleMatcherConfig::TReadableArticle::SerializeToJson() const {
        NJson::TJsonValue result;
        TJsonProcessor::Write(result, "article_code", ArticleCode);
        TJsonProcessor::Write(result, "article_title", ArticleTitle);
        return result;
    }

    const TString TFineArticleMatcherConfig::SettingPrefix = "fines.fine_article_matcher";

    bool TFineArticleMatcherConfig::operator ==(const TFineArticleMatcherConfig& other) const {
        return (std::make_tuple(GenericMatchRuleConfig, MatchRuleConfigs, ReadableArticleMapping) ==
                std::make_tuple(other.GenericMatchRuleConfig, other.MatchRuleConfigs, other.ReadableArticleMapping));
    }

    bool TFineArticleMatcherConfig::operator !=(const TFineArticleMatcherConfig& other) const {
        return !(*this == other);
    }

    TMaybe<TFineArticleMatcherConfig> TFineArticleMatcherConfig::Construct(const NJson::TJsonValue& data) {
        TFineArticleMatcherConfig instance;
        if (!instance.DeserializeFromJson(data)) {
            return {};
        }
        return instance;
    }

    NDrive::TScheme TFineArticleMatcherConfig::GetScheme(const IServerBase& server) {
        NDrive::TScheme scheme;
        scheme.Add<TFSStructure>("generic_match_rule", "Общее правило определения статьи нарушения").SetStructure(decltype(GenericMatchRuleConfig)::GetScheme(server));
        scheme.Add<TFSArray>("match_rules", "Вспомогательные правила определения статей нарушения").SetElement(decltype(MatchRuleConfigs)::mapped_type::GetScheme(server));
        scheme.Add<TFSArray>("recognizable_articles", "Распознаваемые статьи нарушения").SetElement<TFSString>();
        scheme.Add<TFSArray>("readable_articles", "Описания статей нарушения").SetElement(decltype(ReadableArticleMapping)::mapped_type::GetScheme(server));
        return scheme;
    }

    NJson::TJsonValue TFineArticleMatcherConfig::GetSettingDefaults() {
        TSettingDefaultsBuilder builder(SettingPrefix);
        builder.AddJson("generic_match_rule");
        builder.AddJson("match_rules");
        builder.AddJson("recognizable_articles");
        builder.AddJson("readable_articles");
        return builder;
    }

    bool TFineArticleMatcherConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        auto defaults = GetSettingDefaults();
        auto& fullData = NJson::MergeJson(data, defaults);

        if (!TJsonProcessor::ReadObject(fullData, "generic_match_rule", GenericMatchRuleConfig)) {
            return false;
        }

        const auto matchRuleKey = [](const decltype(MatchRuleConfigs)::mapped_type& rule) { return rule.GetName(); };
        if (!ReadObjectsContainerAsMapping(fullData, "match_rules", MatchRuleConfigs, matchRuleKey)) {
            return false;
        }

        if (!TJsonProcessor::ReadContainer(fullData, "recognizable_articles", RecognizableArticles)) {
            return false;
        }

        const auto readableArticleKey = [](const decltype(ReadableArticleMapping)::mapped_type& item) { return item.GetArticleCode(); };
        if (!ReadObjectsContainerAsMapping(fullData, "readable_articles", ReadableArticleMapping, readableArticleKey)) {
            return false;
        }

        return true;
    }

    NJson::TJsonValue TFineArticleMatcherConfig::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::WriteObject(result, "generic_match_rule", GenericMatchRuleConfig);
        WriteMappedObjectsContainer(result, "match_rules", MatchRuleConfigs);
        TJsonProcessor::WriteContainerArray(result, "recognizable_articles", RecognizableArticles);
        WriteMappedObjectsContainer(result, "readable_articles", ReadableArticleMapping);
        return result;
    }

    const TString TFineConstructorConfig::SettingPrefix = "fines.fine_constructor";
    const ui32 TFineConstructorConfig::DefaultValidAmountCentsMultiplicand = (50 * 100);

    TMaybe<TFineConstructorConfig> TFineConstructorConfig::Construct(const NJson::TJsonValue& data) {
        TFineConstructorConfig instance;
        if (!instance.DeserializeFromJson(data)) {
            return {};
        }
        return instance;
    }

    NDrive::TScheme TFineConstructorConfig::GetScheme(const IServerBase& server) {
        NDrive::TScheme scheme;

        scheme.Add<TFSNumeric>("valid_amount_cents_multiplicand", "Корректная сумма штрафа кратна (коп.)").SetMin(1).SetDefault(DefaultValidAmountCentsMultiplicand);
        scheme.Add<TFSString>("tracks_api_name", "Имя графа для поиска треков").SetDefault(DefaultTracksApiName);
        scheme.Add<TFSBoolean>("is_article_code_suggest_enabled", "Использовать подсказку статьи нарушения на основании суммы штрафа").SetDefault(true);
        scheme.Add<TFSBoolean>("do_check_manually_paid_duplicates", "Проверять на наличие ранее оплаченных вручную дубликатов").SetDefault(false);
        scheme.Add<TFSArray>("explicit_discounts", "Безусловные скидки").SetElement(decltype(ExplicitDiscounts)::mapped_type::GetScheme(server));
        scheme.Add<TFSArray>("dismissed_charge_articles", "Статьи штрафов, исключенные из списания").SetElement<TFSString>();
        scheme.Add<TFSArray>("explicit_charge_articles", "Статьи штрафов для безусловного списания").SetElement<TFSString>();
        scheme.Add<TFSArray>("dismissed_charge_cars", "Машины, исключенные из списания").SetElement(decltype(DismissedChargeCars)::mapped_type::GetScheme(server));
        scheme.Add<TFSArray>("dismissed_charge_offer_names", "Названия офферов, исключенные из списания").SetElement<TFSString>();

        // to be loaded from settings only and not be stored in config
        // scheme.Add<TFSStructure>("matching_algo", "Настройки алгоритма матчинга").SetStructure(decltype(MatchingAlgoConfig)::GetScheme(server));
        // scheme.Add<TFSStructure>("inappropriate_city_parking", "Настройки нарушений городской парковки").SetStructure(decltype(InappropriateCityParkingConfig)::GetScheme(server));
        // scheme.Add<TFSStructure>("inappropriate_long_term_parking", "Настройки нарушений городской парковки при определенных офферах").SetStructure(decltype(InappropriateLongTermParkingConfig)::GetScheme(server));
        // scheme.Add<TFSArray>("auto_dismissed_charge_cars", "Дополнительные машины, исключенные из списания").SetElement(decltype(DismissedChargeCars)::mapped_type::GetScheme(server));

        return scheme;
    }

    NJson::TJsonValue TFineConstructorConfig::GetSettingDefaults() {
        TSettingDefaultsBuilder builder(SettingPrefix);
        builder.Add<decltype(ValidAmountCentsMultiplicand)>("valid_amount_cents_multiplicand", DefaultValidAmountCentsMultiplicand);
        builder.Add<decltype(TracksApiName)>("tracks_api_name", DefaultTracksApiName);
        builder.Add<decltype(ArticleCodeSuggestEnabledFlag)>("is_article_code_suggest_enabled", true);
        builder.Add<decltype(DoCheckManuallyPaidDuplicates)>("do_check_manually_paid_duplicates", false);
        builder.AddJson("matching_algo", TMatchingAlgoConfig::GetSettingDefaults());  // either parse property itself or compose a dict from subproperties
        builder.AddJson("inappropriate_city_parking", TInappropriateCityParkingConfig::GetSettingDefaults());  // either parse property itself or compose a dict from subproperties
        builder.AddJson("inappropriate_long_term_parking", TInappropriateLongTermParkingConfig::GetSettingDefaults());  // either parse property itself or compose a dict from subproperties
        builder.AddJson("explicit_discounts");
        builder.AddJson("dismissed_charge_articles");
        builder.AddJson("explicit_charge_articles");
        builder.AddJson("dismissed_charge_cars");
        builder.AddJson("auto_dismissed_charge_cars");
        builder.AddJson("dismissed_charge_offer_names");
        return builder;
    }

    bool TFineConstructorConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        const auto explicitDiscountKey = [](const decltype(ExplicitDiscounts)::mapped_type& item) { return item.GetArticleCode(); };
        const auto explicitDismissedCarKey = [](const decltype(DismissedChargeCars)::mapped_type& item) { return item.GetKey(); };

        auto defaults = GetSettingDefaults();

        // to be read from settings only
        if (!TJsonProcessor::ReadObject(defaults, "matching_algo", MatchingAlgoConfig)) {
            return false;
        }

        if (!TJsonProcessor::ReadObject(defaults, "inappropriate_city_parking", InappropriateCityParkingConfig)) {
            return false;
        }
        if (!TJsonProcessor::ReadObject(defaults, "inappropriate_long_term_parking", InappropriateLongTermParkingConfig)) {
            return false;
        }

        if (!ReadObjectsContainerAsMapping(defaults, "auto_dismissed_charge_cars", AutoDismissedChargeCars, explicitDismissedCarKey)) {
            return false;
        }

        auto& fullData = NJson::MergeJson(data, defaults);

        if (!TJsonProcessor::Read(fullData, "valid_amount_cents_multiplicand", ValidAmountCentsMultiplicand)) {
            return false;
        }

        if (!TJsonProcessor::Read(fullData, "tracks_api_name", TracksApiName)) {
            return false;
        }

        if (!TJsonProcessor::Read(fullData, "is_article_code_suggest_enabled", ArticleCodeSuggestEnabledFlag)) {
            return false;
        }
        if (!TJsonProcessor::Read(fullData, "do_check_manually_paid_duplicates", DoCheckManuallyPaidDuplicates)) {
            return false;
        }

        if (!ReadObjectsContainerAsMapping(fullData, "explicit_discounts", ExplicitDiscounts, explicitDiscountKey)) {
            return false;
        }

        if (!TJsonProcessor::ReadContainer(fullData, "dismissed_charge_articles", DismissedChargeArticles)) {
            return false;
        }
        if (!TJsonProcessor::ReadContainer(fullData, "explicit_charge_articles", ExplicitChargeArticles)) {
            return false;
        }

        if (!ReadObjectsContainerAsMapping(fullData, "dismissed_charge_cars", DismissedChargeCars, explicitDismissedCarKey)) {
            return false;
        }

        if (!TJsonProcessor::ReadContainer(fullData, "dismissed_charge_offer_names", DismissedChargeOfferNames)) {
            return false;
        }

        if (!IsValid()) {
            return false;
        }

        return true;
    }

    NJson::TJsonValue TFineConstructorConfig::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::Write(result, "valid_amount_cents_multiplicand", ValidAmountCentsMultiplicand);
        TJsonProcessor::Write(result, "tracks_api_name", TracksApiName);
        TJsonProcessor::Write(result, "is_article_code_suggest_enabled", ArticleCodeSuggestEnabledFlag);
        TJsonProcessor::Write(result, "do_check_manually_paid_duplicates", DoCheckManuallyPaidDuplicates);
        // MatchingAlgoConfig, InappropriateCityParkingConfig, InappropriateLongTermParkingConfig should be loaded from settings only and not be stored in config
        WriteMappedObjectsContainer(result, "explicit_discounts", ExplicitDiscounts);
        TJsonProcessor::WriteContainerArray(result, "explicit_charge_articles", ExplicitChargeArticles);
        TJsonProcessor::WriteContainerArray(result, "dismissed_charge_articles", DismissedChargeArticles);
        WriteMappedObjectsContainer(result, "dismissed_charge_cars", DismissedChargeCars);
        // AutoDismissedChargeCars should be loaded from settings only and not be stored in config
        TJsonProcessor::WriteContainerArray(result, "dismissed_charge_offer_names", DismissedChargeOfferNames);
        return result;
    }

    bool TFineConstructorConfig::IsValid() const {
        TVector<TString> intersection;
        SetIntersection(ExplicitChargeArticles.begin(), ExplicitChargeArticles.end(),
                        DismissedChargeArticles.begin(), DismissedChargeArticles.end(),
                        std::back_inserter(intersection));
        if (!intersection.empty()) {
            return false;
        }
        return true;
    }

    const TDuration TFineConstructorConfig::TExplicitDiscount::DefaultExpirationTimeout = TDuration::Days(20);
    const ui32 TFineConstructorConfig::TExplicitDiscount::DefaultDiscountPercent = 50;
    const TString TFineConstructorConfig::DefaultTracksApiName = "drive_graph";

    TFineConstructorConfig::TExplicitDiscount::TExplicitDiscount(const TString& name)
        : Name(name)
    {
    }

    NDrive::TScheme TFineConstructorConfig::TExplicitDiscount::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("name", "Название безусловной скидки");
        scheme.Add<TFSString>("article_code", "Статья штрафа").SetRequired(true);
        scheme.Add<TFSDuration>("expiration_timeout", "Срок истечения скидки на оплату с даты постановления").SetDefault(DefaultExpirationTimeout);
        scheme.Add<TFSNumeric>("discount_percent", "Значение скидки (в процентах)").SetMin(0).SetMax(100).SetDefault(DefaultDiscountPercent);
        scheme.Add<TFSNumeric>("assume_rule_second_violation", "Штраф предполагает повторное нарушение статьи").SetDefault(false);
        return scheme;
    }

    bool TFineConstructorConfig::TExplicitDiscount::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TJsonProcessor::Read(data, "name", Name)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "article_code", ArticleCode, /* mustBe = */ true) || !ArticleCode) {
            return false;
        }
        if (!Name) {
            Name = ArticleCode;
        }
        if (!TJsonProcessor::Read(data, "expiration_timeout", ExpirationTimeout)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "discount_percent", DiscountPercent)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "assume_rule_second_violation", AssumeRuleSecondViolation)) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue TFineConstructorConfig::TExplicitDiscount::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::Write(result, "name", Name);
        TJsonProcessor::Write(result, "article_code", ArticleCode);
        TJsonProcessor::WriteDurationString(result, "expiration_timeout", ExpirationTimeout);
        TJsonProcessor::Write(result, "discount_percent", DiscountPercent);
        TJsonProcessor::Write(result, "assume_rule_second_violation", AssumeRuleSecondViolation);
        return result;
    }

    bool TFineConstructorConfig::TDismissedCarInfo::IsValid() const {
        return !!GetKey();
    }

    TString TFineConstructorConfig::TDismissedCarInfo::GetKey() const {
        if (!!Id) {
            return Id;
        }
        if (!!Sts) {
            return ::ToString(Sts);
        }
        if (!!Vin) {
            return Vin;
        }
        return {};
    }

    bool TFineConstructorConfig::TDismissedCarInfo::HasEqualKeyField(const TDismissedCarInfo& other) const {
        return (!!Id && Id == other.Id) || (!!Sts && Sts == other.Sts) || (!!Vin && Vin == other.Vin);
    }

    bool TFineConstructorConfig::TDismissedCarInfo::Match(const TDriveCarInfo& carInfo, const TInstant violationTime) const {
        if (!IsValid()) {
            return false;
        }

        // check car
        if (!!Id && carInfo.GetId() != Id) {
            return false;
        }
        if (!!Sts && carInfo.GetRegistrationID() != Sts) {
            return false;
        }
        if (!!Vin && carInfo.GetVin() != Vin) {
            return false;
        }

        // check violation time constraints after car match
        // NB. violation time is supposed to be provided
        if (!!violationTime) {
            if (!!Since && violationTime < Since) {
                return false;
            }
            if (!!Until && violationTime > Until) {
                return false;
            }
        } else {
            const bool hasTimeLimitations = (!!Since || !!Until);
            return !hasTimeLimitations;
        }

        return true;
    }

    NDrive::TScheme TFineConstructorConfig::TDismissedCarInfo::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("id", "Идентификатор автомобиля");
        scheme.Add<TFSString>("sts", "Номер СТС автомобиля");
        scheme.Add<TFSString>("vin", "VIN автомобиля");
        scheme.Add<TFSNumeric>("since", "Для нарушений с").SetVisual(TFSNumeric::EVisualType::DateTime);
        scheme.Add<TFSNumeric>("until", "Для нарушений до").SetVisual(TFSNumeric::EVisualType::DateTime);
        return scheme;
    }

    bool TFineConstructorConfig::TDismissedCarInfo::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TJsonProcessor::Read(data, "id", Id)) {
            return false;
        }
        if (data.Has("sts") && !TryFromJson(data["sts"], Sts)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "vin", Vin)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "since", Since)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "until", Until)) {
            return false;
        }
        return IsValid();
    }

    NJson::TJsonValue TFineConstructorConfig::TDismissedCarInfo::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::Write(result, "id", Id, { "" });
        TJsonProcessor::Write(result, "sts", Sts, { 0 });
        TJsonProcessor::Write(result, "vin", Vin, { "" });
        TJsonProcessor::WriteInstant(result, "since", Since, TInstant::Zero());
        TJsonProcessor::WriteInstant(result, "until", Until, TInstant::Zero());
        return result;
    }

    const ui32 TFineConstructorConfig::TMatchingAlgoConfig::DefaultMatchingAlgoVersion = 0;
    const ui32 TFineConstructorConfig::TMatchingAlgoConfig::DefaultRejectedSessionsSkipLimit = 4;
    const ui32 TFineConstructorConfig::TMatchingAlgoConfig::DefaultMinRideDistanceMeters = 100;

    const TString TFineConstructorConfig::TMatchingAlgoConfig::SettingPrefix = JoinSeq(".", { TFineConstructorConfig::SettingPrefix, "matching_algo" });

    NDrive::TScheme TFineConstructorConfig::TMatchingAlgoConfig::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSNumeric>("version", "Версия алгоритма").SetMin(0).SetDefault(DefaultMatchingAlgoVersion);
        scheme.Add<TFSNumeric>("rejected_sessions_skip_limit", "Максимальное количество пропущенных сессий без движения").SetMin(0).SetDefault(DefaultRejectedSessionsSkipLimit);
        scheme.Add<TFSNumeric>("min_ride_distance_meters", "Не считать движением поездки короче чем (метров)").SetMin(0).SetDefault(DefaultMinRideDistanceMeters);
        // MatchingRules are loaded from settings only and are not intended to be stored or configured separately
        return scheme;
    }

    NJson::TJsonValue TFineConstructorConfig::TMatchingAlgoConfig::GetSettingDefaults() {
        TSettingDefaultsBuilder builder(SettingPrefix);
        builder.Add<decltype(Version)>("version", DefaultMatchingAlgoVersion);
        builder.Add<decltype(RejectedSessionsSkipLimit)>("rejected_sessions_skip_limit", DefaultRejectedSessionsSkipLimit);
        builder.Add<decltype(MinRideDistanceMeters)>("min_ride_distance_meters", DefaultMinRideDistanceMeters);
        builder.AddJson("matching_rules_by_article_code");
        builder.AddJson("matching_rules_by_rule_name");
        return builder;
    }

    bool TFineConstructorConfig::TMatchingAlgoConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        auto defaults = GetSettingDefaults();
        auto& fullData = NJson::MergeJson(data, defaults);

        if (!TJsonProcessor::Read(fullData, "version", Version)) {
            return false;
        }
        if (!TJsonProcessor::Read(fullData, "rejected_sessions_skip_limit", RejectedSessionsSkipLimit)) {
            return false;
        }
        if (!TJsonProcessor::Read(fullData, "min_ride_distance_meters", MinRideDistanceMeters)) {
            return false;
        }
        {
            for (auto&& [articleCode, ruleNames] : fullData["matching_rules_by_article_code"].GetMap()) {
                if (!!articleCode) {
                    for (const auto& ruleName : ruleNames.GetArray()) {
                        if (ruleName.IsString() && !!ruleName.GetString()) {
                            MatchingRules.emplace(articleCode, ruleName.GetString());
                        }
                    }
                }
            }
            for (auto&& [ruleName, articleCodes] : fullData["matching_rules_by_rule_name"].GetMap()) {
                if (!!ruleName) {
                    for (const auto& articleCode : articleCodes.GetArray()) {
                        if (articleCode.IsString() && !!articleCode.GetString()) {
                            MatchingRules.emplace(articleCode.GetString(), ruleName);
                        }
                    }
                }
            }
        }
        return true;
    }

    NJson::TJsonValue TFineConstructorConfig::TMatchingAlgoConfig::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::Write(result, "version", Version);
        TJsonProcessor::Write(result, "rejected_sessions_skip_limit", RejectedSessionsSkipLimit);
        TJsonProcessor::Write(result, "min_ride_distance_meters", MinRideDistanceMeters);
        // MatchingRules are loaded from settings only and are not intended to be stored or configured separately
        return result;
    }

    const TString TFineConstructorConfig::TInappropriateCityParkingConfig::SettingPrefix = JoinSeq(".", { TFineConstructorConfig::SettingPrefix, "inappropriate_city_parking" });

    NDrive::TScheme TFineConstructorConfig::TInappropriateCityParkingConfig::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSBoolean>("enabled", "Активность правила").SetDefault(false);
        scheme.Add<TFSArray>("fine_article_codes", "Статьи штрафов").SetElement<TFSString>();
        scheme.Add<TFSBoolean>("is_unknown_article_code_denied", "Требовать явное совпадение статьи").SetDefault(true);
        scheme.Add<TFSNumeric>("fine_amount_to_pay_without_discount_cents", "Сумма штрафа без скидки (коп.)").SetMin(0).SetDefault(0);
        scheme.Add<TFSArray>("required_car_tags", "Обязательные теги на авто").SetElement<TFSString>();
        scheme.Add<TFSBoolean>("check_sts_change", "Проверять историю изменений СТС").SetDefault(true);
        return scheme;
    }

    NJson::TJsonValue TFineConstructorConfig::TInappropriateCityParkingConfig::GetSettingDefaults() {
        TSettingDefaultsBuilder builder(SettingPrefix);
        builder.Add<decltype(Enabled)>("enabled", false);
        builder.AddJson("fine_article_codes");
        builder.Add<decltype(IsUnknownArticleCodeDenied)>("is_unknown_article_code_denied", true);
        builder.Add<decltype(FineAmountToPayWithoutDiscountCents)>("fine_amount_to_pay_without_discount_cents", 0);
        builder.AddJson("required_car_tags");
        builder.Add<decltype(CheckSTSChange)>("check_sts_change", true);
        return builder;
    }

    bool TFineConstructorConfig::TInappropriateCityParkingConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        auto defaults = GetSettingDefaults();
        auto& fullData = NJson::MergeJson(data, defaults);
        return NJson::ParseField(fullData["enabled"], Enabled) &&
               NJson::ParseField(fullData["fine_article_codes"], FineArticleCodes) &&
               NJson::ParseField(fullData["is_unknown_article_code_denied"], IsUnknownArticleCodeDenied) &&
               NJson::ParseField(fullData["fine_amount_to_pay_without_discount_cents"], FineAmountToPayWithoutDiscountCents) &&
               NJson::ParseField(fullData["required_car_tags"], RequiredCarTags) &&
               NJson::ParseField(fullData["check_sts_change"], CheckSTSChange);
    }

    NJson::TJsonValue TFineConstructorConfig::TInappropriateCityParkingConfig::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        NJson::InsertField(result, "enabled", Enabled);
        NJson::InsertField(result, "fine_article_codes", FineArticleCodes);
        NJson::InsertField(result, "is_unknown_article_code_denied", IsUnknownArticleCodeDenied);
        NJson::InsertField(result, "fine_amount_to_pay_without_discount_cents", FineAmountToPayWithoutDiscountCents);
        NJson::InsertField(result, "required_car_tags", RequiredCarTags);
        NJson::InsertField(result, "check_sts_change", CheckSTSChange);
        return result;
    }

    const TString TFineConstructorConfig::TInappropriateLongTermParkingConfig::SettingPrefix = JoinSeq(".", { TFineConstructorConfig::SettingPrefix, "inappropriate_long_term_parking" });

    NDrive::TScheme TFineConstructorConfig::TInappropriateLongTermParkingConfig::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSBoolean>("enabled", "Активность правила").SetDefault(false);
        scheme.Add<TFSArray>("fine_article_codes", "Статьи штрафов").SetElement<TFSString>();
        scheme.Add<TFSBoolean>("is_unknown_article_code_denied", "Требовать явное совпадение статьи").SetDefault(true);
        scheme.Add<TFSNumeric>("fine_amount_to_pay_without_discount_cents", "Сумма штрафа без скидки (коп.)").SetMin(0).SetDefault(0);
        scheme.Add<TFSString>("offer_groupping_tags_filter", "Фильтр офферов по групповым тегам");
        return scheme;
    }

    NJson::TJsonValue TFineConstructorConfig::TInappropriateLongTermParkingConfig::GetSettingDefaults() {
        TSettingDefaultsBuilder builder(SettingPrefix);
        builder.Add<decltype(Enabled)>("enabled", false);
        builder.AddJson("fine_article_codes");
        builder.Add<decltype(IsUnknownArticleCodeDenied)>("is_unknown_article_code_denied", true);
        builder.Add<decltype(FineAmountToPayWithoutDiscountCents)>("fine_amount_to_pay_without_discount_cents", 0);
        builder.Add<decltype(OfferGrouppingTagsFilter)>("offer_groupping_tags_filter", "");
        return builder;
    }

    bool TFineConstructorConfig::TInappropriateLongTermParkingConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        auto defaults = GetSettingDefaults();
        auto& fullData = NJson::MergeJson(data, defaults);
        return NJson::ParseField(fullData["enabled"], Enabled) &&
               NJson::ParseField(fullData["fine_article_codes"], FineArticleCodes) &&
               NJson::ParseField(fullData["is_unknown_article_code_denied"], IsUnknownArticleCodeDenied) &&
               NJson::ParseField(fullData["fine_amount_to_pay_without_discount_cents"], FineAmountToPayWithoutDiscountCents) &&
               NJson::ParseField(fullData["offer_groupping_tags_filter"], OfferGrouppingTagsFilter);
    }

    NJson::TJsonValue TFineConstructorConfig::TInappropriateLongTermParkingConfig::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        NJson::InsertField(result, "enabled", Enabled);
        NJson::InsertField(result, "fine_article_codes", FineArticleCodes);
        NJson::InsertField(result, "is_unknown_article_code_denied", IsUnknownArticleCodeDenied);
        NJson::InsertField(result, "fine_amount_to_pay_without_discount_cents", FineAmountToPayWithoutDiscountCents);
        NJson::InsertField(result, "offer_groupping_tags_filter", OfferGrouppingTagsFilter);
        return result;
    }

    TFineAttachmentConstructorConfig::TAttachmentConfig::TAttachmentConfig(const TString& pathPrefix, const TString& extension)
        : PathPrefix(pathPrefix)
        , Extension(extension)
    {
    }

    NDrive::TScheme TFineAttachmentConstructorConfig::TAttachmentConfig::GetScheme(const IServerBase& /* server */) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("path_prefix", "Префикс пути сохраняемого файла");
        scheme.Add<TFSString>("extension", "Расширение сохраняемого файла");
        return scheme;
    }

    bool TFineAttachmentConstructorConfig::TAttachmentConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TJsonProcessor::Read(data, "path_prefix", PathPrefix)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "extension", Extension)) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue TFineAttachmentConstructorConfig::TAttachmentConfig::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::Write(result, "path_prefix", PathPrefix);
        TJsonProcessor::Write(result, "extension", Extension);
        return result;
    }

    const TString TFineAttachmentConstructorConfig::SettingPrefix = "fines.fine_attachment_constructor";
    const TString TFineAttachmentConstructorConfig::DefaultMDSBucketName = "carsharing-violations";

    TMaybe<TFineAttachmentConstructorConfig> TFineAttachmentConstructorConfig::Construct(const NJson::TJsonValue& data) {
        TFineAttachmentConstructorConfig instance;
        if (!instance.DeserializeFromJson(data)) {
            return {};
        }
        return instance;
    }

    NDrive::TScheme TFineAttachmentConstructorConfig::GetScheme(const IServerBase& server) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("mds_bucket_name", "Название бакета").SetDefault(DefaultMDSBucketName);
        scheme.Add<TFSStructure>("decree_config", "Настройки сохранения постановлений").SetStructure(decltype(DecreeConfig)::GetScheme(server));
        scheme.Add<TFSStructure>("photo_config", "Настройки сохранения фото").SetStructure(decltype(PhotoConfig)::GetScheme(server));
        return scheme;
    }

    NJson::TJsonValue TFineAttachmentConstructorConfig::GetSettingDefaults() {
        TSettingDefaultsBuilder builder(SettingPrefix);
        builder.Add<decltype(MDSBucketName)>("mds_bucket_name", DefaultMDSBucketName);
        builder.AddJson("decree_config");
        builder.AddJson("photo_config");
        return builder;
    }

    bool TFineAttachmentConstructorConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        auto defaults = GetSettingDefaults();
        auto& fullData = NJson::MergeJson(data, defaults);
        if (!TJsonProcessor::Read(fullData, "mds_bucket_name", MDSBucketName)) {
            return false;
        }
        if (!TJsonProcessor::ReadObject(fullData, "decree_config", DecreeConfig)) {
            return false;
        }
        if (!TJsonProcessor::ReadObject(fullData, "photo_config", PhotoConfig)) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue TFineAttachmentConstructorConfig::SerializeToJson() const {
        NJson::TJsonValue result(NJson::JSON_MAP);
        TJsonProcessor::Write(result, "mds_bucket_name", MDSBucketName);
        TJsonProcessor::WriteObject(result, "decree_config", DecreeConfig);
        TJsonProcessor::WriteObject(result, "photo_config", PhotoConfig);
        return result;
    }

    bool TFinesManagerConfig::Init(const TYandexConfig::Section* /* section */) {
        return true;
    }

    void TFinesManagerConfig::ToString(IOutputStream& /* os */) const {
    }
}
