#pragma once

#include <library/cpp/mediator/global_notifications/system_status.h>
#include <library/cpp/yconf/conf.h>

#include <rtline/util/types/accessor.h>

#include <library/cpp/json/json_value.h>

#include <util/datetime/base.h>
#include <util/generic/cast.h>
#include <util/generic/map.h>
#include <util/generic/set.h>

class IServerBase;

namespace NDrive {
    class TScheme;
}

class TDriveCarInfo;

namespace NDrive::NFine {
    class TBaseRegexpMatchRuleConfig {
    public:
        explicit TBaseRegexpMatchRuleConfig(const TString& name);
        TBaseRegexpMatchRuleConfig(const TString& name, const TString& pattern, const bool requireOneNeedle = false);

        virtual ~TBaseRegexpMatchRuleConfig() = default;

        bool operator ==(const TBaseRegexpMatchRuleConfig& other) const;

        static NDrive::TScheme GetScheme(const IServerBase& server);
        virtual bool DeserializeFromJson(const NJson::TJsonValue& data);
        virtual NJson::TJsonValue SerializeToJson() const;

    private:
        R_READONLY(TString, Name);
        R_READONLY(TString, Pattern);
        R_READONLY(bool, RequireOneNeedle, false);
    };

    class TSimpleStringRegexpMatchRuleConfig : public TBaseRegexpMatchRuleConfig {
        using TBase = TBaseRegexpMatchRuleConfig;

    public:
        explicit TSimpleStringRegexpMatchRuleConfig(const TString& name = "");
        TSimpleStringRegexpMatchRuleConfig(const TString& name, const TString& pattern, const TString& matchResult, const bool requireOneNeedle = false);

        bool operator ==(const TSimpleStringRegexpMatchRuleConfig& other) const;

        static NDrive::TScheme GetScheme(const IServerBase& server);
        virtual bool DeserializeFromJson(const NJson::TJsonValue& data) override;
        virtual NJson::TJsonValue SerializeToJson() const override;

    private:
        R_READONLY(TString, MatchResult);
    };

    class TGenericMatchRuleConfig : public TBaseRegexpMatchRuleConfig {
        using TBase = TBaseRegexpMatchRuleConfig;

        static const TString TypeName;

    public:
        TGenericMatchRuleConfig();
        explicit TGenericMatchRuleConfig(const TString& pattern);
    };

    class TFineArticleMatcherConfig {
    public:
        class TReadableArticle {
        public:
            TReadableArticle() = default;
            TReadableArticle(const TString& articleCode, const TString& articleTitle);

            bool operator ==(const TReadableArticle& other) const;

            static NDrive::TScheme GetScheme(const IServerBase& server);
            bool DeserializeFromJson(const NJson::TJsonValue& data);
            NJson::TJsonValue SerializeToJson() const;

        private:
            R_READONLY(TString, ArticleCode);
            R_READONLY(TString, ArticleTitle);
        };

        using TMatchRuleConfigs = TMap<TString, TSimpleStringRegexpMatchRuleConfig>;
        using TReadableArticleMapping = TMap<TString, TReadableArticle>;  // pairs { articleCode, TReadableArticle }

        static const TString SettingPrefix;

        bool operator ==(const TFineArticleMatcherConfig& other) const;
        bool operator !=(const TFineArticleMatcherConfig& other) const;

        static TMaybe<TFineArticleMatcherConfig> Construct(const NJson::TJsonValue& data = {});

        static NDrive::TScheme GetScheme(const IServerBase& server);
        static NJson::TJsonValue GetSettingDefaults();
        bool DeserializeFromJson(const NJson::TJsonValue& data);
        NJson::TJsonValue SerializeToJson() const;

    private:
        R_READONLY(TGenericMatchRuleConfig, GenericMatchRuleConfig);
        R_READONLY(TMatchRuleConfigs, MatchRuleConfigs);
        R_READONLY(TSet<TString>, RecognizableArticles);
        R_READONLY(TReadableArticleMapping, ReadableArticleMapping);
    };

    class TFineConstructorConfig {
    public:
        class TExplicitDiscount {
        public:
            static const TDuration DefaultExpirationTimeout;
            static const ui32 DefaultDiscountPercent;

            explicit TExplicitDiscount(const TString& name = "");

            static NDrive::TScheme GetScheme(const IServerBase& server);
            bool DeserializeFromJson(const NJson::TJsonValue& data);
            NJson::TJsonValue SerializeToJson() const;

        private:
            R_READONLY(TString, Name);

            R_READONLY(TString, ArticleCode);
            R_READONLY(TDuration, ExpirationTimeout, DefaultExpirationTimeout);
            R_READONLY(ui32, DiscountPercent, DefaultDiscountPercent);
            R_READONLY(bool, AssumeRuleSecondViolation, false);
        };

        class TDismissedCarInfo {
        public:
            bool IsValid() const;
            TString GetKey() const;

            bool HasEqualKeyField(const TDismissedCarInfo& other) const;

            bool Match(const TDriveCarInfo& carInfo, const TInstant violationTime) const;

            static NDrive::TScheme GetScheme(const IServerBase& server);
            bool DeserializeFromJson(const NJson::TJsonValue& data);
            NJson::TJsonValue SerializeToJson() const;

        private:
            R_FIELD(TString, Id);
            R_FIELD(ui64, Sts, 0);
            R_FIELD(TString, Vin);

            R_FIELD(TInstant, Since);
            R_FIELD(TInstant, Until);
        };

        class TMatchingAlgoConfig {
        public:
            static const ui32 DefaultMatchingAlgoVersion;
            static const ui32 DefaultRejectedSessionsSkipLimit;
            static const ui32 DefaultMinRideDistanceMeters;

            static const TString SettingPrefix;

            using TMatchingRules = std::multimap<TString, TString>;  // pairs { articleCode, IMatchingConstraint name}

            static NDrive::TScheme GetScheme(const IServerBase& server);
            static NJson::TJsonValue GetSettingDefaults();
            bool DeserializeFromJson(const NJson::TJsonValue& data);
            NJson::TJsonValue SerializeToJson() const;

        private:
            R_FIELD(ui32, Version, DefaultMatchingAlgoVersion);
            R_FIELD(ui32, RejectedSessionsSkipLimit, DefaultRejectedSessionsSkipLimit);
            R_FIELD(ui32, MinRideDistanceMeters, DefaultMinRideDistanceMeters);

            R_FIELD(TMatchingRules, MatchingRules);
        };

        class TInappropriateCityParkingConfig {
        public:
            static const TString SettingPrefix;

            static NDrive::TScheme GetScheme(const IServerBase& server);
            static NJson::TJsonValue GetSettingDefaults();
            bool DeserializeFromJson(const NJson::TJsonValue& data);
            NJson::TJsonValue SerializeToJson() const;

        private:
            R_FIELD(bool, Enabled, false);

            R_FIELD(TSet<TString>, FineArticleCodes);
            R_FIELD(bool, IsUnknownArticleCodeDenied, true);
            R_FIELD(i64, FineAmountToPayWithoutDiscountCents, 0);

            R_FIELD(TVector<TString>, RequiredCarTags);
            R_FIELD(bool, CheckSTSChange, true);
        };

        class TInappropriateLongTermParkingConfig {
        public:
            static const TString SettingPrefix;

            static NDrive::TScheme GetScheme(const IServerBase& server);
            static NJson::TJsonValue GetSettingDefaults();
            bool DeserializeFromJson(const NJson::TJsonValue& data);
            NJson::TJsonValue SerializeToJson() const;

        private:
            R_FIELD(bool, Enabled, false);

            R_FIELD(TSet<TString>, FineArticleCodes);
            R_FIELD(bool, IsUnknownArticleCodeDenied, true);
            R_FIELD(i64, FineAmountToPayWithoutDiscountCents, 0);

            R_FIELD(TString, OfferGrouppingTagsFilter);
        };

        using TExplicitDiscounts = TMap<TString, TExplicitDiscount>;  // pairs { articleCode, TExplicitDiscount }
        using TDismissedCarInfoMapping = TMap<TString, TDismissedCarInfo>;  // pairs { unique key, TDismissedCarInfo }

        static const TString SettingPrefix;
        static const ui32 DefaultValidAmountCentsMultiplicand;
        static const TString DefaultTracksApiName;

    public:
        static TMaybe<TFineConstructorConfig> Construct(const NJson::TJsonValue& data = {});

        static NDrive::TScheme GetScheme(const IServerBase& server);
        static NJson::TJsonValue GetSettingDefaults();
        bool DeserializeFromJson(const NJson::TJsonValue& data);
        NJson::TJsonValue SerializeToJson() const;

    private:
        bool IsValid() const;

    private:
        R_READONLY(ui32, ValidAmountCentsMultiplicand, DefaultValidAmountCentsMultiplicand);       // amount to charge must be a multiple of the value
        R_READONLY(TString, TracksApiName, DefaultTracksApiName);
        R_READONLY(bool, ArticleCodeSuggestEnabledFlag, true);
        R_READONLY(bool, DoCheckManuallyPaidDuplicates, false);

        R_READONLY(TMatchingAlgoConfig, MatchingAlgoConfig);
        R_READONLY(TInappropriateCityParkingConfig, InappropriateCityParkingConfig);
        R_READONLY(TInappropriateLongTermParkingConfig, InappropriateLongTermParkingConfig);

        R_READONLY(TSet<TString>, ExplicitChargeArticles);
        R_READONLY(TSet<TString>, DismissedChargeArticles);

        R_READONLY(TExplicitDiscounts, ExplicitDiscounts);

        R_READONLY(TDismissedCarInfoMapping, DismissedChargeCars);       // to be set manually

        R_READONLY(TDismissedCarInfoMapping, AutoDismissedChargeCars);       // hidden, to be set automatically, can be huge and complex

        R_READONLY(TSet<TString>, DismissedChargeOfferNames);
    };

    class TFineAttachmentConstructorConfig {
    public:
        class TAttachmentConfig {
        public:
            TAttachmentConfig() = default;
            TAttachmentConfig(const TString& pathPrefix, const TString& extension);

            static NDrive::TScheme GetScheme(const IServerBase& server);
            bool DeserializeFromJson(const NJson::TJsonValue& data);
            NJson::TJsonValue SerializeToJson() const;

        private:
            R_FIELD(TString, PathPrefix);
            R_FIELD(TString, Extension);
        };

        static const TString SettingPrefix;
        static const TString DefaultMDSBucketName;

    public:
        static TMaybe<TFineAttachmentConstructorConfig> Construct(const NJson::TJsonValue& data = {});

        static NDrive::TScheme GetScheme(const IServerBase& server);
        static NJson::TJsonValue GetSettingDefaults();
        bool DeserializeFromJson(const NJson::TJsonValue& data);
        NJson::TJsonValue SerializeToJson() const;

    private:
        R_READONLY(TString, MDSBucketName, DefaultMDSBucketName);

        R_READONLY(TAttachmentConfig, DecreeConfig, TAttachmentConfig("decrees", ".pdf"));
        R_READONLY(TAttachmentConfig, PhotoConfig, TAttachmentConfig("", ""));
    };

    class TFinesManagerConfig {
    public:
        bool Init(const TYandexConfig::Section* section);
        void ToString(IOutputStream& os) const;
    };
}
