#include "fine_attachment_constructor.h"

#include <drive/backend/database/drive_api.h>

#include <drive/library/cpp/mds/client.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <util/string/builder.h>

namespace NDrive::NFine {
    TFineAttachmentConstructor::TFineAttachmentConstructor(const TFineAttachmentConstructorConfig& config, const NDrive::IServer& server)
        : Config(config)
        , Server(server)
    {
    }

    bool TFineAttachmentConstructor::Init(const NDrive::NFine::TFinesManager& manager, const TSet<TString>& fineIds, NDrive::TEntitySession& tx) {
        auto existingFinePhotos = manager.GetFineAttachments(fineIds, NDrive::NFine::TAutocodeFineAttachmentEntry::EDataType::Photo, tx);
        if (!existingFinePhotos) {
            return false;
        }
        for (auto&& photo : *existingFinePhotos) {
            KnownPhotos[photo.GetFineId()].emplace_back(std::move(photo));
        }
        auto existingFineDecrees = manager.GetFineAttachments(fineIds, NDrive::NFine::TAutocodeFineAttachmentEntry::EDataType::Decree, tx);
        if (!existingFineDecrees) {
            return false;
        }
        for (auto&& degree : *existingFineDecrees) {
            KnownDegrees[degree.GetFineId()].emplace_back(std::move(degree));
        }
        return true;
    }

    const TFineAttachmentConstructorConfig& TFineAttachmentConstructor::GetConfig() const {
        return Config;
    }

    bool TFineAttachmentConstructor::ConstructAutocodeFineAttachmentEntry(const TString& fineId, const TAutocodeFineAttachmentEntry::EDataType dateType, const TString& data, TAutocodeFineAttachmentEntry& fineAttachmentEntry, TMessagesCollector& errors) const {
        fineAttachmentEntry.SetFineId(fineId);
        fineAttachmentEntry.SetDataType(dateType);
        TString fileName = TAutocodeFineAttachmentEntry::GetAttachmentDefaultRelativePath(fineAttachmentEntry.GetFineId(), fineAttachmentEntry.GetId(), Config.GetPhotoConfig().GetPathPrefix(), Config.GetPhotoConfig().GetExtension());
        return UploadAttachment(fineId, fileName, data, fineAttachmentEntry.MutableUrl(), errors);
    }

    bool TFineAttachmentConstructor::ConstructAutocodeFineAttachmentEntry(const TString& fineId, const TAutocodeFinePhoto& finePhoto, TAutocodeFineAttachmentEntry& fineAttachmentEntry, TMessagesCollector& errors) const {
        return ConstructAutocodeFineAttachmentEntry(fineId, TAutocodeFineAttachmentEntry::EDataType::Photo, finePhoto.GetData(), fineAttachmentEntry, errors);
    }

    bool TFineAttachmentConstructor::ConstructAutocodeFineAttachmentEntry(const TString& fineId, const TElementPenaltyPhoto& finePhoto, TAutocodeFineAttachmentEntry& fineAttachmentEntry, TMessagesCollector& errors) const {
        return ConstructAutocodeFineAttachmentEntry(fineId, TAutocodeFineAttachmentEntry::EDataType::Photo, finePhoto.GetData(), fineAttachmentEntry, errors);
    }

    bool TFineAttachmentConstructor::ConstructAutocodeFineAttachmentEntry(const TString& fineId, const TCarPenaltyDecree& penaltyDecree, TAutocodeFineAttachmentEntry& fineAttachmentEntry, TMessagesCollector& errors) const {
        return ConstructAutocodeFineAttachmentEntry(fineId, TAutocodeFineAttachmentEntry::EDataType::Decree, penaltyDecree.GetContent(), fineAttachmentEntry, errors);
    }

    bool TFineAttachmentConstructor::UploadAttachment(const TString& fineId, const TString& fileName, const TString& data, TString& filePath, TMessagesCollector& errors) const {
        CHECK_WITH_LOG(!!Server.GetDriveAPI());
        if (!Server.GetDriveAPI()->HasMDSClient()) {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "No MDS client configured");
            return false;
        }

        const auto& mdsClient = Server.GetDriveAPI()->GetMDSClient();
        TString mdsBucketName = Config.GetMDSBucketName();

        const TS3Client::TBucket* mdsBucketPtr = mdsClient.GetBucket(mdsBucketName);
        if (mdsBucketPtr == nullptr) {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "MDS bucket " << mdsBucketName << " does not exists");
            return false;
        }

        filePath = mdsClient.GetTmpFilePath(mdsBucketName, fileName);

        if (mdsBucketPtr->PutKey(fileName, data, errors) != HTTP_OK) {
            errors.AddMessage(__LOCATION__, TStringBuilder()
                                            << "Error uploading attachment for fine " << fineId
                                            << " into bucket " << mdsBucketName << " by path " << fileName
                                            << " : " << errors.GetStringReport());
            return false;
        }

        return true;
    }
}
