#include "fine_constructor.h"

#include <drive/backend/fines/article_matcher.h>
#include <drive/backend/fines/filters.h>
#include <drive/backend/fines/manager.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/actions/abstract/action.h>
#include <drive/backend/car_attachments/registry/registry.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/history/event.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/data/alerts/traces.h>
#include <drive/backend/roles/action.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_filter.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>

#include <util/generic/algorithm.h>
#include <util/string/cast.h>
#include <util/string/strip.h>
#include <util/string/subst.h>

namespace NJson {
    template <>
    TJsonValue ToJson(const NDrive::NFine::TFineConstructor::TChargeStatusReasons& object) {
        NJson::TJsonValue chargeStatusTraits = NJson::JSON_ARRAY;
        if (object) {
            for (auto&& reason: object) {
                chargeStatusTraits.AppendValue(::ToString(reason));
            }
        }
        return chargeStatusTraits;
    }
}

namespace {
    template <typename T, typename V>
    inline T ApplyDiscount(const V value, ui32 discountPercent) {
        return static_cast<T>(value * (100 - discountPercent) / 100.0);
    }

    template <typename T>
    inline bool IsMultipleOf(const T value, const ui32 divisor) {
        return divisor * static_cast<T>(value / divisor) == value;
    }

    template <typename T>
    static T ConvertToCents(const double value) {
        return static_cast<T>(100 * value);  // note: unsigned int is not preferred due to possible negative sum to pay value
    }

    TMaybe<TTagsFilter> GetTagsFilter(const NDrive::IServer& server, const TString& article) {
        auto filters = server.GetSettings().GetJsonValue("fines.tag_filters_to_charge");
        if (!filters.IsDefined()) {
            return {};
        }
        if (auto ptr = filters.GetMap().FindPtr(article)) {
            return TTagsFilter::BuildFromString(ptr->GetString());
        }
        return TTagsFilter();
    }
}

namespace NDrive::NFine {
    const ui32 TFineConstructor::DefaultDiscountPercent = TFineConstructorConfig::TExplicitDiscount::DefaultDiscountPercent;
    const TDuration TFineConstructor::DefaultDiscountTimeout = TFineConstructorConfig::TExplicitDiscount::DefaultExpirationTimeout;

    TVector<TString> TFineConstructor::TContext::GetOfferNames() const {
        TVector<TString> offerNames;
        for (auto&& offer: Offers) {
            if (offer && offer->GetName()) {
                offerNames.push_back(offer->GetName());
            }
        }
        return offerNames;
    }

    TFineConstructor::TFineConstructor(const TFineConstructorConfig& config, const NDrive::IServer& server)
        : Config(config)
        , Server(server)
    {
        CHECK_WITH_LOG(!!Server.GetDriveAPI());
        ArticleMatcherPtr = Server.GetDriveAPI()->GetFinesManager().GetFineArticleMatcherPtr(/* ensureActual = */ true);
    }

    TFineConstructor::TPtr TFineConstructor::ConstructDefault(const NDrive::IServer& server) {
        auto config = TFineConstructorConfig::Construct();
        if (!config) {
            return nullptr;
        }
        return MakeAtomicShared<TFineConstructor>(*config, server);
    }

    const TFineConstructorConfig& TFineConstructor::GetConfig() const {
        return Config;
    }

    bool TFineConstructor::ConstructAutocodeFineEntry(const TAutocodeFine& fine, TAutocodeFineEntry& entry) const {
        if (!!entry.GetChargedAt()) {
            return false;  // already charged fine must not be updated
        }

        bool requireSessionRebound = false;

        entry.SetSourceType(ToString(ESourceType::Autocode));

        TString rulingNumber = fine.GetRulingNumber();
        if (!ProcessRulingNumber(rulingNumber) || !rulingNumber) {
            return false;
        }
        entry.SetRulingNumber(rulingNumber);

        if (!!fine.GetId()) {
            entry.SetAutocodeId(fine.GetId());
        } else {
            entry.SetAutocodeId(TAutocodeFineEntry::DefaultAutocodeId);
        }

        const TInstant rulingDate = fine.GetRulingDate();
        entry.SetRulingDate(rulingDate);

        const TInstant violationTime = fine.GetViolationDateWithTime();
        if (violationTime != entry.GetViolationTime()) {
            requireSessionRebound = true;
        }
        entry.SetViolationTime(violationTime);

        const double sumToPayWithoutDiscount = fine.GetSumToPay();
        entry.SetSumToPayWithoutDiscount(sumToPayWithoutDiscount);

        // Refer to TFineArticleMatcher to found out almost all possible articles
        TString rawArticle = fine.GetArticleKoap();
        if (!ProcessArticle(rawArticle)) {
            return false;
        }
        entry.SetArticleKoap(rawArticle);

        TString articleCode;
        GetArticleCode(rawArticle, sumToPayWithoutDiscount, articleCode);  // do not skip fine but it won't be processed
        if (articleCode != entry.GetArticleCode()) {
            requireSessionRebound = true;
        }
        entry.SetArticleCode(articleCode);  // overwrite article code

        TString violationPlace = fine.GetViolationPlace();
        if (!ProcessViolationPlace(violationPlace)) {
            return false;
        }
        entry.SetViolationPlace(violationPlace);

        entry.SetDiscountDate(fine.GetDiscountDate());

        CalculatePaymentDiscount(articleCode, rulingDate, sumToPayWithoutDiscount, entry.MutableDiscountDate(), entry.MutableSumToPay());

        TString odpsName = fine.GetOdpsName();
        if (!ProcessOdpsName(odpsName)) {
            return false;
        }
        entry.SetOdpsName(odpsName);
        entry.SetOdpsCode(fine.GetOdpsCode());

        const auto carSts = fine.GetViolationDocumentNumber();
        entry.SetViolationDocumentType(ToString(EDocumentType::STS));
        entry.SetViolationDocumentNumber(ToString(carSts));

        entry.SetHasPhoto(fine.GetHasPhoto());

        TString carId;
        if (!GetCarBySts(carSts, carId)) {
            return false;
        }
        entry.SetCarId(carId);

        const auto now = ModelingNow();
        entry.SetFineInformationReceivedAt(now).SetAddedAtTimestamp(now);

        if (!UpdateMetaInfoFields(fine, entry)) {
            return false;
        }

        if (!UpdateDependentFields(entry, requireSessionRebound)) {
            return false;
        }

        return true;
    }

    bool TFineConstructor::UpdateMetaInfoFields(const TAutocodeFine& /* fine */, TAutocodeFineEntry& /* entry */) const {
        return true;
    }

    bool TFineConstructor::ConstructAutocodeFineEntry(const TCarPenaltyInfo& penalty, TAutocodeFineEntry& entry) const {
        if (!!entry.GetChargedAt()) {
            return false;  // already charged fine must not be updated
        }

        bool requireSessionRebound = false;

        entry.SetSourceType(ToString(ESourceType::Major));

        TString rulingNumber = penalty.GetRulingNumber();
        if (!ProcessRulingNumber(rulingNumber) || !rulingNumber) {
            return false;
        }
        if (entry.GetRulingNumber() && entry.GetRulingNumber() != rulingNumber) {
            NJson::TJsonValue json;
            {
                TString setting = Server.GetSettings().GetValueDef<TString>("fines.major.ruling_number_aliases", "[]");
                if (!NJson::ReadJsonFastTree(setting, &json)) {
                    return false;
                }
            }
            bool needReplace = false;
            {
                auto getPriority = [](const NJson::TJsonValue& aliases, const TString& val) {
                    ui32 pos = 0;
                    for (auto&& alias : aliases.GetArray()) {
                        if (val.StartsWith(alias.GetString())) {
                            return pos;
                        }
                        ++pos;
                    }
                    return Max<ui32>();
                };
                bool found = false;
                for (auto&& aliases : json.GetArray()) {
                    ui32 currentPos = getPriority(aliases, rulingNumber);
                    ui32 dbPos = getPriority(aliases, entry.GetRulingNumber());
                    if (currentPos != Max<ui32>() && dbPos != Max<ui32>()) {
                        if (currentPos != dbPos) {
                            needReplace =  dbPos > currentPos;
                            found = true;
                        }
                        break;
                    }
                }
                if (!found) {
                    return false;
                }
            }
            auto addAlias = [&entry](const TString& val) {
                auto aliases = entry.GetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::RulingNumberAliases, NJson::JSON_ARRAY);
                aliases.AppendValue(val);
                entry.SetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::RulingNumberAliases, std::move(aliases));
            };
            if (needReplace) {
                addAlias(entry.GetRulingNumber());
                entry.SetRulingNumber(rulingNumber);
            } else {
                addAlias(rulingNumber);
            }
        } else {
            entry.SetRulingNumber(rulingNumber);
        }

        if (!!penalty.GetId()) {
            entry.SetAutocodeId(penalty.GetId());
        } else {
            entry.SetAutocodeId(TAutocodeFineEntry::DefaultAutocodeId);
        }

        const TInstant rulingDate = penalty.GetRulingDate();
        entry.SetRulingDate(rulingDate);

        const TInstant violationTime = penalty.GetViolationTime();
        if (violationTime != entry.GetViolationTime()) {
            requireSessionRebound = true;
        }
        entry.SetViolationTime(violationTime);

        const double sumToPayWithoutDiscount = penalty.GetSumToPayWithoutDiscount();
        entry.SetSumToPayWithoutDiscount(sumToPayWithoutDiscount);

        // Refer to TFineArticleMatcher to found out almost all possible articles
        TString rawArticle = penalty.GetRequisites().GetField(ERequisiteFieldName::ArticleKoap);
        if (!ProcessArticle(rawArticle)) {
            return false;
        }
        entry.SetArticleKoap(rawArticle);

        TString articleCode;
        GetArticleCode(rawArticle, sumToPayWithoutDiscount, articleCode);  // do not skip fine but it won't be processed
        if (articleCode != entry.GetArticleCode()) {
            requireSessionRebound = true;
        }
        entry.SetArticleCode(articleCode);  // overwrite article code

        TString violationPlace = penalty.GetRequisites().GetField(ERequisiteFieldName::ViolationPlace);
        if (!ProcessViolationPlace(violationPlace)) {
            return false;
        }
        entry.SetViolationPlace(violationPlace);

        entry.SetDiscountDate(penalty.GetDiscountUntil());

        entry.SetSumToPay(penalty.GetSumToPay());

        // discounted amount to pay is provided, so the function below does not perform any extra logic
        // it's remained just for unification and completeness
        CalculatePaymentDiscount(articleCode, rulingDate, sumToPayWithoutDiscount, entry.MutableDiscountDate(), entry.MutableSumToPay());

        TString odpsName = penalty.GetPenaltyIssuanceName();
        if (!ProcessOdpsName(odpsName)) {
            return false;
        }
        entry.SetOdpsName(odpsName);  // no odps code

        entry.SetHasPhoto(false);

        const TString carVin = penalty.GetVin();
        entry.SetViolationDocumentType(ToString(EDocumentType::VIN));
        entry.SetViolationDocumentNumber(carVin);

        TString carId;
        if (!GetCarByVin(carVin, carId)) {
            return false;
        }
        entry.SetCarId(carId);

        const auto now = ModelingNow();
        entry.SetFineInformationReceivedAt(now).SetAddedAtTimestamp(now);

        if (!UpdateMetaInfoFields(penalty, entry)) {
            return false;
        }

        if (!UpdateDependentFields(entry, requireSessionRebound)) {
            return false;
        }

        return true;
    }

    bool TFineConstructor::UpdateMetaInfoFields(const TCarPenaltyInfo& penalty, TAutocodeFineEntry& entry) const {
        const bool hasDecree = penalty.GetIsDecreeExists();
        entry.SetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::HasDecree, hasDecree);

        auto willBeIncludedToBillJson = (penalty.HasWillBeIncludedToBill()) ? NJson::TJsonValue(penalty.GetWillBeIncludedToBillRef()) : NJson::TJsonValue(NJson::JSON_NULL);
        entry.SetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::WillBeIncludedToBill, std::move(willBeIncludedToBillJson));

        return true;
    }

    bool TFineConstructor::ConstructAutocodeFineEntry(const TElementPenaltyInfo& penalty, TAutocodeFineEntry& entry) const {
        if (!!entry.GetChargedAt()) {
            return false;  // already charged fine must not be updated
        }

        bool requireSessionRebound = false;

        entry.SetSourceType(ToString(ESourceType::Element));

        TString rulingNumber = penalty.GetProtocolNumber();
        if (!ProcessRulingNumber(rulingNumber) || !rulingNumber) {
            return false;
        }
        entry.SetRulingNumber(rulingNumber);

        entry.SetAutocodeId(TAutocodeFineEntry::DefaultAutocodeId);

        const TInstant rulingDate = penalty.GetProtocolDate();
        entry.SetRulingDate(rulingDate);

        const TInstant violationTime = penalty.GetViolationTimestamp();
        if (violationTime != entry.GetViolationTime()) {
            requireSessionRebound = true;
        }
        entry.SetViolationTime(violationTime);

        const double sumToPayWithoutDiscount = penalty.GetSum();
        entry.SetSumToPayWithoutDiscount(sumToPayWithoutDiscount);

        TString rawArticle = penalty.GetArticle();
        if (!ProcessArticle(rawArticle)) {
            return false;
        }
        entry.SetArticleKoap(rawArticle);

        TString articleCode;
        GetArticleCode(rawArticle, sumToPayWithoutDiscount, articleCode);
        if (!articleCode && penalty.GetArticleCode()) {
            GetArticleCode(penalty.GetArticleCode(), sumToPayWithoutDiscount, articleCode);
        }
        if (articleCode != entry.GetArticleCode()) {
            requireSessionRebound = true;
        }
        entry.SetArticleCode(articleCode);

        TString violationPlace = penalty.GetViolationPlace();
        if (!ProcessViolationPlace(violationPlace)) {
            return false;
        }
        entry.SetViolationPlace(violationPlace);

        entry.SetDiscountDate(penalty.GetDiscountDate());

        if (penalty.IsPossibleDiscount()) {
            CalculatePaymentDiscount(articleCode, rulingDate, sumToPayWithoutDiscount, entry.MutableDiscountDate(), entry.MutableSumToPay());
        } else {
            entry.SetSumToPay(sumToPayWithoutDiscount);
        }

        TString odpsName = penalty.GetDepartment();
        if (!ProcessOdpsName(odpsName)) {
            return false;
        }
        entry.SetOdpsName(odpsName);

        TString carId;
        const auto carSts = penalty.GetCarSTS();
        if (carSts) {
            entry.SetViolationDocumentType(ToString(EDocumentType::STS));
            entry.SetViolationDocumentNumber(carSts);
            ui64 stsNumber = 0;
            if (!TryFromString(carSts, stsNumber)) {
                NDrive::TEventLog::Log("FineConstructWarning", NJson::TMapBuilder
                    ("warning", "fail to sts from string")
                    ("sts", carSts)
                    ("ruling_number", rulingNumber)
                );
            } else if (!GetCarBySts(stsNumber, carId)) {
                NDrive::TEventLog::Log("FineConstructWarning", NJson::TMapBuilder
                    ("warning", "fail to find car by sts")
                    ("sts", carSts)
                    ("ruling_number", rulingNumber)
                );
            }
        }
        if (!carId && penalty.GetCarNumber()) {
            const TString& carNumber = ToLowerUTF8(penalty.GetCarNumber());
            entry.SetViolationDocumentType(ToString(EDocumentType::CarNumber));
            entry.SetViolationDocumentNumber(carNumber);
            auto gCar = Server.GetDriveAPI()->GetCarNumbers()->GetCachedOrFetch(NContainer::Scalar(carNumber));
            if (auto carPtr = gCar.GetResultPtr(carNumber)) {
                carId = carPtr->GetId();
            } else {
                NDrive::TEventLog::Log("FineConstructWarning", NJson::TMapBuilder
                    ("warning", "fail to find car by number")
                    ("car_number", carNumber)
                    ("ruling_number", rulingNumber)
                );
                return false;
            }
        }
        if (!carId) {
            return false;
        }
        entry.SetCarId(carId);

        entry.SetHasPhoto(penalty.IsPhotoFixation());

        const auto now = ModelingNow();
        entry.SetFineInformationReceivedAt(now).SetAddedAtTimestamp(now);

        if (!UpdateMetaInfoFields(penalty, entry)) {
            return false;
        }

        if (!UpdateDependentFields(entry, requireSessionRebound)) {
            return false;
        }

        return true;
    }

    bool TFineConstructor::UpdateMetaInfoFields(const TElementPenaltyInfo& penalty, TAutocodeFineEntry& entry) const {
        if (!penalty.GetPaymentData().IsMap()) {
            return true;
        }
        entry.SetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::PaymentData, NJson::TJsonValue(penalty.GetPaymentData()));
        return true;
    }

    bool TFineConstructor::UpdateDependentFields(TAutocodeFineEntry& entry, const bool requireSessionRebound) const {
        if (!!entry.GetChargedAt()) {
            return false;  // already charged fine must not be updated
        }

        // an external update for an old fine can be triggered so article code re-check can be required
        TString articleCode;
        GetArticleCode(entry.GetArticleKoap(), entry.GetSumToPayWithoutDiscount(), articleCode);
        if (articleCode != entry.GetArticleCode()) {
            entry.SetArticleCode(articleCode);  // overwrite article code
        }

        entry.SetIsCameraFixation(IsCameraFixation(entry.GetOdpsName(), entry.GetRulingNumber(), entry.GetSourceType()));

        if (IsBindingUpdateRequired(entry, requireSessionRebound)) {
            UpdateBinding(entry);
        }

        TContext context;
        if (entry.GetSessionId()) {
            GetSessionOffers(entry.GetSessionId(), context.Offers);
        }

        if (IsBindingDependentFieldsUpdateRequired(entry, requireSessionRebound)) {
            UpdateBindingDependentFields(context, entry);
        }

        auto chargeStatusReasons = GetChargeStatusReasons(context, entry);
        entry.SetNeedsCharge(DoNeedCharge(chargeStatusReasons));
        entry.SetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::ChargeStatusTraits, NJson::ToJson(chargeStatusReasons));

        return true;
    }

    bool TFineConstructor::IsBindingUpdateRequired(const TAutocodeFineEntry& entry, const bool requireSessionRebound) const {
        return (!entry.HasBinding() || requireSessionRebound) &&  // no binding or existing binding is incorrect
               !(entry.HasSessionRebounds() && !requireSessionRebound);  // do not update binding with manual binding drop without explicit request
    }

    bool TFineConstructor::UpdateBinding(TAutocodeFineEntry& entry) const {
        bool matched = false;

        const TString& articleCode = entry.GetArticleCode();

        NDrive::NSession::TSessionBindingInfo nearestSessionInfo;
        if (!!entry.GetViolationTime() && !!articleCode) {
            matched = MatchSession(articleCode, entry.GetCarId(), entry.GetViolationTime(), nearestSessionInfo);
        }

        // overwrite fields
        if (matched) {
            entry.SetSessionId(nearestSessionInfo.GetSessionId());
            entry.SetUserId(nearestSessionInfo.GetUserId());
            TSet<TString> tags;
            if (!MatchTags(entry, tags)) {
                return false;
            }
            if (!tags.empty()) {
                entry.SetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::BoundCarTags, NJson::ToJson(tags));
            }
        } else {
            entry.SetSessionId("");
            entry.SetUserId("");
        }

        entry.SetSkipped(nearestSessionInfo.GetSkipped());

        NDrive::TEventLog::Log("FineMatching", NJson::TMapBuilder
            ("fine_id", entry.GetId())
            ("added_at_timestamp", entry.GetAddedAtTimestamp().Seconds())
            ("matched", matched)
            ("algo_version", Config.GetMatchingAlgoConfig().GetVersion())
            ("has_algo_required_fields", !!entry.GetViolationTime() && !!articleCode)
            ("compiled_rides_matching_meta", NJson::ToJson(nearestSessionInfo.GetCompiledRidesMatchingMetaInfo()))
            ("billing_sessions_matching_meta", NJson::ToJson(nearestSessionInfo.GetBillingSessionsMatchingMetaInfo()))
        );

        return true;
    }

    bool TFineConstructor::IsBindingDependentFieldsUpdateRequired(const TAutocodeFineEntry& entry, const bool requireSessionRebound) const {
        return IsBindingUpdateRequired(entry, requireSessionRebound) ||  // binding info has been overwritten
               entry.HasSessionRebounds();  // or binding has been updated externally
    }

    bool TFineConstructor::UpdateBindingDependentFields(const TContext& context, TAutocodeFineEntry& entry) const {
        // overwrite fields
        entry.SetViolationLongitude(TAutocodeFineEntry::DefaultViolationLongitude);
        entry.SetViolationLatitude(TAutocodeFineEntry::DefaultViolationLatitude);

        if (!!entry.GetSessionId() && !!entry.GetViolationTime()) {
            double violationLongitude, violationLatitude;

            if (GetViolationCoords(entry.GetSessionId(), entry.GetViolationTime(), violationLongitude, violationLatitude)) {
                entry.SetViolationLongitude(violationLongitude);
                entry.SetViolationLatitude(violationLatitude);
            }
        }

        entry.SetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::RelatedOfferNames, NJson::ToJson(context.GetOfferNames()));  // overwrite if exists

        return true;
    }

    bool TFineConstructor::ProcessRulingNumber(TString& rulingNumber) const {
        StripInPlace(rulingNumber);
        return true;
    }

    bool TFineConstructor::ProcessArticle(TString& article) const {
        StripInPlace(article);
        return true;
    }

    bool TFineConstructor::ProcessViolationPlace(TString& violationPlace) const {
        SubstGlobal(violationPlace, "&nbsp;", " ");
        SubstGlobal(violationPlace, "На карте", "");
        SubstGlobal(violationPlace, "\"", "\\\"");
        StripInPlace(violationPlace);
        return true;
    }

    bool TFineConstructor::ProcessOdpsName(TString& odpsName) const {
        StripInPlace(odpsName);
        return true;
    }

    bool TFineConstructor::TrySuggestArticleCode(const double sumToPayWithoutDiscount, TString& articleCode) const {
        if (!Config.GetArticleCodeSuggestEnabledFlag()) {
            return false;
        }

        auto amountCents = ConvertToCents<i64>(sumToPayWithoutDiscount);
        if (amountCents == 15000000 || amountCents == 30000000) {
            articleCode = "8_25";
            return true;
        }

        return false;
    }

    bool TFineConstructor::GetArticleCode(const TString& rawArticle, const double sumToPayWithoutDiscount, TString& articleCode) const {
        if (!!ArticleMatcherPtr && ArticleMatcherPtr->DetermineArticle(rawArticle, articleCode)) {
            return true;
        }
        return TrySuggestArticleCode(sumToPayWithoutDiscount, articleCode);
    }

    bool TFineConstructor::IsFineDiscountValid(const TAutocodeFineEntry& entry) const {
        ui32 expectedDiscountPercent = (entry.GetDiscountDate()) ? DefaultDiscountPercent : 0;

        const TString& articleCode = entry.GetArticleCode();
        if (articleCode) {
            const auto& explicitDiscounts = Config.GetExplicitDiscounts();
            const auto* discountPtr = explicitDiscounts.FindPtr(articleCode);
            if (discountPtr) {
                expectedDiscountPercent = discountPtr->GetDiscountPercent();
            }
        }

        return (entry.GetSumToPayCents() * 100) == (entry.GetSumToPayWithoutDiscountCents() * (100 - expectedDiscountPercent));
    }

    bool TFineConstructor::AreFineAmountsToPayValid(const TAutocodeFineEntry& entry) const {
        return IsFineAmountToPayValid(entry.GetSumToPayCents()) && IsFineAmountToPayValid(entry.GetSumToPayWithoutDiscountCents());
    }

    bool TFineConstructor::IsFineAmountToPayValid(const i64 amountCents) const {
        return amountCents > 0 && (!Config.GetValidAmountCentsMultiplicand() || IsMultipleOf(amountCents, Config.GetValidAmountCentsMultiplicand()));
    }

    bool TFineConstructor::CalculatePaymentDiscount(const TString& articleCode, const TInstant rulingDate, const double sumToPayWithoutDiscount, TInstant& discountDate, double& sumToPay) const {
        // method is intended to fill sum to pay depending on other fine field
        // correctness is checked using IsFineDiscountValid

        if (!!sumToPay) {
            return false;
        }

        if (!!discountDate) {
            sumToPay = ApplyDiscount<double>(sumToPayWithoutDiscount, DefaultDiscountPercent);
            return true;
        }

        if (!!articleCode) {
            const auto& explicitDiscounts = Config.GetExplicitDiscounts();
            const auto* discountPtr = explicitDiscounts.FindPtr(articleCode);
            if (discountPtr != nullptr) {
                sumToPay = ApplyDiscount<double>(sumToPayWithoutDiscount, discountPtr->GetDiscountPercent());
                if (discountPtr->GetDiscountPercent() && discountPtr->GetExpirationTimeout()) {
                    discountDate = rulingDate + discountPtr->GetExpirationTimeout();
                }
                return true;
            }
        }

        sumToPay = sumToPayWithoutDiscount;
        return true;
    }

    bool TFineConstructor::GetParsedOdpsName(const TString& rawOdpsName, EOdpsName& result) const {
        if (!TryFromString<EOdpsName>(rawOdpsName, result)) {
            if (rawOdpsName.Contains(ToString(EOdpsName::GIBDD))) {
                result = EOdpsName::GIBDD;
                return true;
            }
            if (rawOdpsName.Contains(ToString(EFullOdpsName::GIBDD))) {
                result = EOdpsName::GIBDD;
                return true;
            }
            if (rawOdpsName.Contains(ToString(EFullOdpsName::MADI))) {
                result = EOdpsName::MADI;
                return true;
            }
            return false;
        }
        return true;
    }

    bool TFineConstructor::IsCameraFixation(const TString& rawOdpsName, const TString& rulingNumber, const TString& source) const {
        // Ruling number explanation (odps name, fixation type) could be found here:
        //   https://www.kommersant.ru/doc/3311990 (not official)
        EOdpsName parsedOdpsName;
        if (!GetParsedOdpsName(rawOdpsName, parsedOdpsName)) {
            return false;
        }
        if (parsedOdpsName == EOdpsName::GIBDD) {
            return rulingNumber.substr(3, 3) == "101" || rulingNumber.substr(3, 3) == "105";
        }
        if (parsedOdpsName == EOdpsName::MADI) {
            return rulingNumber.substr(8, 3) == "101";
        }
        // unpaid city parking - disabled?
        // if (parsedOdpsName == EOdpsName::AMPP) {
        //     return GetRulingNumber().substr(6, 3) == "101";
        // }
        if (source == ToString(ESourceType::Element)) {
            if (TString elementGIBDD = Server.GetSettings().GetValueDef<TString>("fines.element.odbs.gibdd", "")) {
                TSet<TString> variants;
                StringSplitter(elementGIBDD).Split(',').SkipEmpty().Collect(&variants);
                if (variants.contains(rawOdpsName)) {
                    return rulingNumber.substr(3, 3) == "101" || rulingNumber.substr(3, 3) == "105";
                }
            }
            if (TString elementMADI = Server.GetSettings().GetValueDef<TString>("fines.element.odbs.madi", "")) {
                TSet<TString> variants;
                StringSplitter(elementMADI).Split(',').SkipEmpty().Collect(&variants);
                if (variants.contains(rawOdpsName)) {
                    return rulingNumber.substr(8, 3) == "101";
                }
            }
        }
        return false;
    }

    bool TFineConstructor::IsInappropriateCityParking(const TAutocodeFineEntry& entry) const {
        const auto& specificConfig = Config.GetInappropriateCityParkingConfig();
        if (!specificConfig.GetEnabled()) {
            return false;
        }

        if (specificConfig.GetFineArticleCodes()) {
            TString articleCode = entry.GetArticleCode();
            if (!articleCode) {
                if (specificConfig.GetIsUnknownArticleCodeDenied()) {
                    return false;
                }
            } else {
                if (!specificConfig.GetFineArticleCodes().contains(articleCode)) {
                    return false;
                }
            }
        }

        if (specificConfig.GetFineAmountToPayWithoutDiscountCents()) {
            if (specificConfig.GetFineAmountToPayWithoutDiscountCents() != entry.GetSumToPayWithoutDiscountCents()) {
                return false;
            }
        }

        if (specificConfig.GetRequiredCarTags()) {
            const auto& tagsManagerImpl = Server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
            auto session = tagsManagerImpl.BuildSession(/* readonly = */ true);
            auto carTags = tagsManagerImpl.RestoreEntityTags(entry.GetCarId(), specificConfig.GetRequiredCarTags(), session);
            if (!carTags || carTags->empty()) {
                return false;
            }
        }

        if (specificConfig.GetCheckSTSChange()) {
            auto tx = Server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto optionalEvents = Server.GetDriveDatabase().GetCarManager().GetHistoryManager().GetEvents(entry.GetCarId(), {}, tx);
            if (!optionalEvents) {
                ERROR_LOG << "cannot GetEvents for " << entry.GetCarId() << ": " << tx.GetStringReport() << Endl;
                return false;
            }

            TSet<ui64> carStsValues;
            for (auto&& event: *optionalEvents) {
                auto stsValue = event.GetRegistrationID();
                if (stsValue) {
                    carStsValues.insert(stsValue);
                }
            }

            if (carStsValues.size() > 1) {
                return false;
            }
        }

        TInstant parkingPermitStartTimestamp;
        {
            TCarGenericAttachment currentRegistryDocument;
            if (!Server.GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(entry.GetCarId(), EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument)) {
                return false;
            }
            auto baseDocuments = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());
            if (baseDocuments) {
                parkingPermitStartTimestamp = baseDocuments->GetParkingPermitStartDate();
            }
        }
        if (!entry.GetViolationTime() || !parkingPermitStartTimestamp || entry.GetViolationTime() < parkingPermitStartTimestamp) {
            return false;
        }

        return true;
    }

    bool TFineConstructor::IsInappropriateLongTermParking(const TAutocodeFineEntry& entry) const {
        const auto& specificConfig = Config.GetInappropriateLongTermParkingConfig();
        if (!specificConfig.GetEnabled()) {
            return false;
        }

        if (specificConfig.GetFineArticleCodes()) {
            TString articleCode = entry.GetArticleCode();
            if (!articleCode) {
                if (specificConfig.GetIsUnknownArticleCodeDenied()) {
                    return false;
                }
            } else {
                if (!specificConfig.GetFineArticleCodes().contains(articleCode)) {
                    return false;
                }
            }
        }

        if (specificConfig.GetFineAmountToPayWithoutDiscountCents()) {
            if (specificConfig.GetFineAmountToPayWithoutDiscountCents() != entry.GetSumToPayWithoutDiscountCents()) {
                return false;
            }
        }

        {
            TTagsFilter offerAttributesFilter;
            if (!offerAttributesFilter.DeserializeFromString(specificConfig.GetOfferGrouppingTagsFilter()) || offerAttributesFilter.IsEmpty()) {
                return false;
            }

            TOfferPtrs offers;
            if (!entry.GetSessionId() || !GetSessionOffers(entry.GetSessionId(), offers) || !offers) {
                return false;
            }

            auto action = Server.GetDriveAPI()->GetRolesManager()->GetAction(offers.back()->GetBehaviourConstructorId());
            if (!action || !offerAttributesFilter.IsMatching((*action)->GetGrouppingTags())) {
                return false;
            }
        }

        return true;
    }

    bool TFineConstructor::IsBoundTagsReason(const TAutocodeFineEntry& entry) const {
        auto tagsFilter = GetTagsFilter(Server, entry.GetArticleCode());
        if (!tagsFilter) {
            NDrive::TEventLog::Log("FineMatchingError", NJson::TMapBuilder
                ("fine_id", entry.GetId())
                ("ruling_number", entry.GetRulingNumber())
                ("car_id", entry.GetCarId())
                ("source", entry.GetSourceType())
                ("error", "fail to parse tags filter settings")
            );
            return false;
        }
        auto entryTags = entry.GetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::BoundCarTags, NJson::JSON_ARRAY);
        TSet<TString> bindedTags;
        Transform(entryTags.GetArray().begin(), entryTags.GetArray().end(), std::inserter(bindedTags, bindedTags.begin()), [](auto&& item) { return item.GetString(); });
        return tagsFilter->IsMatching(bindedTags);
    }

    bool TFineConstructor::IsOfferDismissed(const TContext& context) const {
        const TSet<TString>& dismissedOfferNames = Config.GetDismissedChargeOfferNames();
        TSet<TString> relatedOfferNames = MakeSet(context.GetOfferNames());

        TVector<TString> intersection;
        SetIntersection(dismissedOfferNames.begin(), dismissedOfferNames.end(),
                        relatedOfferNames.begin(), relatedOfferNames.end(),
                        std::back_inserter(intersection));
        return !intersection.empty();
    }

    bool TFineConstructor::IsManuallyPaidDuplicate(const TAutocodeFineEntry& entry, NDrive::TEntitySession& tx) const {
        if (!Config.GetDoCheckManuallyPaidDuplicates()) {
            return false;
        }

        // NB. Fine is traited as manually paid duplicate if
        //   it's assigned to the same car and user and
        //   it has the same article code and violation time as one of _received_ earlier

        const TString& carId = entry.GetCarId();
        const TString& userId = entry.GetUserId();
        const TInstant violationTimestamp = entry.GetViolationTime();
        const TString& articleCode = entry.GetArticleCode();
        if (!userId || !violationTimestamp || !articleCode) {
            return false;
        }

        TFineFilterGroup filters = {
            MakeAtomicShared<TFineCarFilter>(carId),
            MakeAtomicShared<TFineUserFilter>(userId),
            MakeAtomicShared<TFineViolationTimeFilter>(violationTimestamp, violationTimestamp + TDuration::Seconds(1)),
            MakeAtomicShared<TFineArticleFilter>(ArticleMatcherPtr, TSet<TString>{articleCode})
        };

        auto fines = Server.GetDriveAPI()->GetFinesManager().GetFinesByCarId(entry.GetCarId(), filters, tx);
        R_ENSURE(fines, {}, "Fail to fetch fines for IsManuallyPaidDuplicate", tx);

        return !fines->empty();
    }

    bool TFineConstructor::DoNeedCharge(const TChargeStatusReasons& reasons) const {
        ui64 traits = Accumulate(reasons, 0ull, [](auto lhs, auto rhs) { return lhs | rhs; });
        return NFineChargeStatusTraits::DoNeedCharge(traits);
    }

    TFineConstructor::TChargeStatusReasons TFineConstructor::GetChargeStatusReasons(const TContext& context, const TAutocodeFineEntry& entry) const {
        TChargeStatusReasons reasons;

        if (!entry.HasBinding()) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::NoBinding);
        }

        if (static_cast<ui32>(entry.GetSkipped()) > Config.GetMatchingAlgoConfig().GetRejectedSessionsSkipLimit()) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::SkipLimitReached);
        }

        if (!AreFineAmountsToPayValid(entry)) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::InvalidAmount);
        }

        if (!IsFineDiscountValid(entry)) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::InvalidDiscount);
        }

        if (!Config.GetDismissedChargeCars().empty() || !Config.GetAutoDismissedChargeCars().empty()) {
            const auto& carId = entry.GetCarId();
            auto gCars = Server.GetDriveAPI()->GetCarsData()->FetchInfo(carId, TInstant::Zero());

            auto carInfoPtr = gCars.GetResultPtr(carId);
            if (!!carInfoPtr) {
                const auto matchCarActor = [carInfoPtr, &entry](auto&& dismissalInfoPair) {
                    return dismissalInfoPair.second.Match(*carInfoPtr, entry.GetViolationTime());
                };

                if (AnyOf(Config.GetDismissedChargeCars(), matchCarActor) || AnyOf(Config.GetAutoDismissedChargeCars(), matchCarActor)) {
                    reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::ExplicitDismissalCar);
                }
            }
        }

        if (!entry.GetArticleKoap()) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::UnknownArticle);
        }

        TString articleCode = entry.GetArticleCode();
        if (!articleCode) {
            if (entry.GetArticleKoap()) {
                reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::NotRecognizedArticle);
            }  // unknown otherwise, already added
        }

        if (Config.GetDismissedChargeArticles().contains(articleCode)) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::ExplicitDismissalArticle);
        }

        if (IsOfferDismissed(context)) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::ExplicitDismissalOfferName);
        }

        if (Config.GetDoCheckManuallyPaidDuplicates()) {
            auto tx = Yensured(Server.GetDriveAPI())->GetFinesManager().BuildTx<NSQL::ReadOnly>();
            if (IsManuallyPaidDuplicate(entry, tx)) {
                reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::ManuallyPaidDuplicate);
            }
        }

        if (IsInappropriateCityParking(entry)) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::InappropriateCityParking);
        }

        if (IsInappropriateLongTermParking(entry)) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::InappropriateLongTermParking);
        }

        if (IsBoundTagsReason(entry)) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::BoundChargeableTags);
        }

        if (Config.GetExplicitChargeArticles().contains(articleCode)) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::ExplicitChargeArticle);
        }

        if (!entry.GetIsCameraFixation()) {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::NoCameraFixation);
        } else {
            reasons.push_back(NFineChargeStatusTraits::EChargeStatusReason::CameraFixation);
        }

        return reasons;
    }

    bool TFineConstructor::GetCarByVin(const TString& vin, TString& carId) const {
        CHECK_WITH_LOG(!!Server.GetDriveAPI());
        auto gVins = Server.GetDriveAPI()->GetCarVins()->FetchInfo(vin, TInstant::Zero());
        auto carInfoPtr = gVins.GetResultPtr(vin);
        if (carInfoPtr != nullptr) {
            carId = carInfoPtr->GetId();
            return true;
        }
        return false;
    }

    bool TFineConstructor::GetCarBySts(const ui64& sts, TString& carId) const {
        CHECK_WITH_LOG(!!Server.GetDriveAPI());
        auto gCars = Server.GetDriveAPI()->GetCarsData()->FetchInfo(TInstant::Zero());
        for (auto&& [_, carInfo] : gCars.GetResult()) {
            if (carInfo.GetRegistrationID() == sts) {
                carId = carInfo.GetId();
                return true;
            }
        }
        return false;
    }

    bool TFineConstructor::MatchSession(const TString& articleCode, const TString& carId, const TInstant timestamp, NDrive::NSession::TSessionBindingInfo& sessionInfo) const {
        const auto& matchingAlgoConfig = Config.GetMatchingAlgoConfig();

        NDrive::NSession::TMatchingOptions options;
        options.RejectedSessionsSkipLimit = matchingAlgoConfig.GetRejectedSessionsSkipLimit();
        options.MinRideDistanceMeters = matchingAlgoConfig.GetMinRideDistanceMeters();

        TSet<TString> constraintNames;
        auto it = matchingAlgoConfig.GetMatchingRules().equal_range(articleCode);
        for (auto pairPtr = it.first; pairPtr != it.second; ++pairPtr) {  // iretators to pair { articleCode, constraintName }
            constraintNames.emplace(pairPtr->second);
        }

        auto constraints = NDrive::NSession::TMatchingConstraintsGroup::Construct(&Server, options, constraintNames);
        return !!constraints && constraints->MatchSession(carId, timestamp, sessionInfo);
    }

    bool TFineConstructor::MatchTags(const TAutocodeFineEntry& entry, TSet<TString>& tags) const {
        auto tagsFilter = GetTagsFilter(Server, entry.GetArticleCode());
        if (!tagsFilter) {
            NDrive::TEventLog::Log("FineMatchingError", NJson::TMapBuilder
                ("fine_id", entry.GetId())
                ("ruling_number", entry.GetRulingNumber())
                ("car_id", entry.GetCarId())
                ("source", entry.GetSourceType())
                ("error", "fail to parse tags filter settings")
            );
            return false;
        }
        if (!*tagsFilter) {
            return true;
        }
        auto tx = Server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (auto tagsState = Server.GetDriveAPI()->GetTagsManager().GetDeviceTags().SimpleRollbackTagsState(entry.GetCarId(), entry.GetViolationTime(), tagsFilter->GetRelatedTagNames(), tx)) {
            tags.insert(tagsState->begin(), tagsState->end());
        } else {
            NDrive::TEventLog::Log("FineMatchingError", NJson::TMapBuilder
                ("fine_id", entry.GetId())
                ("ruling_number", entry.GetRulingNumber())
                ("car_id", entry.GetCarId())
                ("source", entry.GetSourceType())
                ("error", tx.GetReport())
            );
            return false;
        }
        return true;
    }

    bool TFineConstructor::GetViolationCoords(const TString& sessionId, const TInstant violationTimestamp, double& violationLongitude, double& violationLatitude) const {
        TTracesAccessor ta(&Server, Config.GetTracksApiName());
        ta.SetOverrideText(Sprintf("s_session_id:\"%s\" ", sessionId.data())).SetNeedTrace(true);

        TMessagesCollector errors;
        if (!ta.Execute(errors, TDuration::Zero())) {
            return false;
        }

        const ui64 violationTimestampSeconds = violationTimestamp.Seconds();
        ui64 foundPointTimestampSeconds = TInstant::Zero().Seconds();

        for (auto&& i : ta.GetTraces()) {
            const auto& timestamps = i.GetTimestamps();
            const auto& coords = i.GetCoords();

            if (timestamps.size() > coords.size()) {
                return false;
            }

            for (size_t idx = 0; idx < timestamps.size(); ++idx) {
                // tracks are supposed to be mixed a bit
                if (
                    timestamps[idx] <= violationTimestampSeconds &&
                    (!!foundPointTimestampSeconds || timestamps[idx] >= foundPointTimestampSeconds)
                ) {
                    foundPointTimestampSeconds = timestamps[idx];
                    violationLongitude = coords[idx].X;
                    violationLatitude = coords[idx].Y;
                } else {
                    break;
                }
            }
        }

        return (!!foundPointTimestampSeconds);
    }

    bool TFineConstructor::GetSessionOffers(const TString& sessionId, TOfferPtrs& offers) const {
        auto tx = Server.GetDriveAPI()->BuildTx<NSQL::ReadOnly>();

        THistoryRidesContext sessionsContext(Server);
        auto ydbTx = Server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("fine_constructor", &Server);
        if (!sessionsContext.InitializeSession(sessionId, tx, ydbTx)) {
            return false;
        }

        auto sessionsIterator = sessionsContext.GetIterator();

        THistoryRideObject sessionInfo;
        while (sessionsIterator.GetAndNext(sessionInfo)) {
            THistoryRideObject::FetchFullRiding(&Server, NContainer::Scalar(sessionInfo));
            auto offer = sessionInfo.GetOffer();
            if (offer) {
                offers.push_back(offer);
            }
        }

        return true;
    }
}
