#pragma once

#include <drive/backend/fines/config.h>
#include <drive/backend/fines/autocode/entries.h>

#include <drive/backend/major/entities.h>
#include <drive/backend/sessions/matcher/session_matcher.h>

#include <drive/library/cpp/autocode/entities.h>
#include <drive/library/cpp/element/entities.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/range.h>


class IOffer;

namespace NDrive {
    class IServer;
}

namespace NDrive::NFine {
    class TFineArticleMatcher;

    class TFineConstructor {
        static const ui32 DefaultDiscountPercent;
        static const TDuration DefaultDiscountTimeout;

        using TAutocodeFine = NDrive::NAutocode::TAutocodeFine;

        using TCarPenaltyInfo = NMajorClient::TCarPenaltiesRequest::TCarPenaltyInfo;
        using TRequisites = TCarPenaltyInfo::TRequisites;
        using ERequisiteFieldName = TRequisites::ERequisiteFieldName;

        using TElementPenaltyInfo = NDrive::TElementFine;

        using TOfferPtrs = TVector<TAtomicSharedPtr<IOffer>>;

        struct TContext {
            TVector<TString> GetOfferNames() const;

            TOfferPtrs Offers;
        };

    public:
        using TPtr = TAtomicSharedPtr<TFineConstructor>;
        using TChargeStatusReasons = TVector<NFineChargeStatusTraits::EChargeStatusReason>;

        TFineConstructor(const TFineConstructorConfig& config, const NDrive::IServer& server);

        static TFineConstructor::TPtr ConstructDefault(const NDrive::IServer& server);

        const TFineConstructorConfig& GetConfig() const;

        bool ConstructAutocodeFineEntry(const TAutocodeFine& fine, TAutocodeFineEntry& entry) const;
        bool ConstructAutocodeFineEntry(const TCarPenaltyInfo& penalty, TAutocodeFineEntry& entry) const;
        bool ConstructAutocodeFineEntry(const TElementPenaltyInfo& penalty, TAutocodeFineEntry& entry) const;

        bool UpdateMetaInfoFields(const TAutocodeFine& fine, TAutocodeFineEntry& entry) const;
        bool UpdateMetaInfoFields(const TCarPenaltyInfo& penalty, TAutocodeFineEntry& entry) const;
        bool UpdateMetaInfoFields(const TElementPenaltyInfo& penalty, TAutocodeFineEntry& entry) const;

        bool UpdateDependentFields(TAutocodeFineEntry& entry, const bool requireSessionRebound) const;

    private:
        bool IsBindingUpdateRequired(const TAutocodeFineEntry& entry, const bool requireSessionRebound) const;
        bool UpdateBinding(TAutocodeFineEntry& entry) const;

        bool IsBindingDependentFieldsUpdateRequired(const TAutocodeFineEntry& entry, const bool requireSessionRebound) const;
        bool UpdateBindingDependentFields(const TContext& context, TAutocodeFineEntry& entry) const;

        bool ProcessRulingNumber(TString& rulingNumber) const;
        bool ProcessArticle(TString& article) const;
        bool ProcessViolationPlace(TString& violationPlace) const;
        bool ProcessOdpsName(TString& odpsName) const;

        bool TrySuggestArticleCode(const double sumToPayWithoutDiscount, TString& articleCode) const;
        bool GetArticleCode(const TString& rawArticle, const double sumToPayWithoutDiscount, TString& articleCode) const;

        bool IsFineDiscountValid(const TAutocodeFineEntry& entry) const;
        bool AreFineAmountsToPayValid(const TAutocodeFineEntry& entry) const;
        bool IsFineAmountToPayValid(const i64 amountCents) const;
        bool CalculatePaymentDiscount(const TString& articleCode, const TInstant rulingDate, const double sumToPayWithoutDiscount, TInstant& discountDate, double& sumToPay) const;

        bool GetParsedOdpsName(const TString& rawOdpsName, EOdpsName& result) const;
        bool IsCameraFixation(const TString& rawOdpsName, const TString& rulingNumber, const TString& source) const;
        bool IsInappropriateCityParking(const TAutocodeFineEntry& entry) const;
        bool IsInappropriateLongTermParking(const TAutocodeFineEntry& entry) const;
        bool IsBoundTagsReason(const TAutocodeFineEntry& entry) const;

        bool IsOfferDismissed(const TContext& context) const;
        bool IsManuallyPaidDuplicate(const TAutocodeFineEntry& entry, NDrive::TEntitySession& tx) const;

        bool DoNeedCharge(const TChargeStatusReasons& reasons) const;
        TChargeStatusReasons GetChargeStatusReasons(const TContext& context, const TAutocodeFineEntry& entry) const;

        bool GetCarByVin(const TString& vin, TString& carId) const;
        bool GetCarBySts(const ui64& sts, TString& carId) const;

        bool MatchSession(const TString& articleCode, const TString& carId, const TInstant timestamp, NDrive::NSession::TSessionBindingInfo& sessionInfo) const;
        bool MatchTags(const TAutocodeFineEntry& entry, TSet<TString>& tags) const;

        bool GetViolationCoords(const TString& sessionId, const TInstant violationTimestamp, double& violationLongitude, double& violationLatitude) const;
        bool GetSessionOffers(const TString& sessionId, TOfferPtrs& offers) const;

    private:
        const TFineConstructorConfig Config;
        const NDrive::IServer& Server;

        TAtomicSharedPtr<TFineArticleMatcher> ArticleMatcherPtr;
    };
}
