#pragma once

#include <drive/backend/fines/scheme_util.h>

#include <drive/backend/tags/tags.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/messages_collector.h>

namespace NDrive::NFine {
    class TFineFetchContext;

    class IFineTagConfig {
        R_FIELD(TString, TagNameTemplate);
        R_FIELD(TString, TagDataTemplate);

        R_FIELD(bool, Enabled, true);

    public:
        using TPtr = TAtomicSharedPtr<IFineTagConfig>;

        virtual ~IFineTagConfig() = default;

        bool IsTagNullable() const;

        ITag::TPtr CreateTag(const TFineFetchContext& context, TMessagesCollector& errors) const;

        virtual bool DeserializeFromJson(const NJson::TJsonValue& data) = 0;
        virtual NJson::TJsonValue SerializeToJson() const = 0;

    private:
        TString GetTagName(const TFineFetchContext& context, TMessagesCollector& errors) const;
        TString GetTagData(const TFineFetchContext& context, TMessagesCollector& errors) const;
    };

    template <typename TImpl>
    class TFineTagConfig : public IFineTagConfig {
    public:
        static NDrive::TScheme GetScheme(const IServerBase& server);

        virtual bool DeserializeFromJson(const NJson::TJsonValue& data) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
    };

    template <typename TImpl>
    NDrive::TScheme TFineTagConfig<TImpl>::GetScheme(const IServerBase& server) {
        NDrive::TScheme scheme;
        NDrive::TryInitDefaultFromSettings(scheme.Add<TFSString>("tag_name_template", "Шаблон названия тега"), server, TImpl::TSchemeSettings::GetTagNameKey());
        NDrive::TryInitDefaultFromSettings(scheme.Add<TFSJson>("tag_data_template", "Шаблон содержимого тега"), server, TImpl::TSchemeSettings::GetTagDataKey());
        scheme.Add<TFSBoolean>("enabled", "Выставлять ли тег").SetDefault(true);
        return scheme;
    }

    template <typename TImpl>
    bool TFineTagConfig<TImpl>::DeserializeFromJson(const NJson::TJsonValue& data) {
        auto scheme = TImpl::GetScheme(NDrive::GetServer());
        if (!NDrive::ReadFieldSchemeDefault(scheme, data, "tag_name_template", MutableTagNameTemplate(), /* required = */ false, /* schemeRequired = */ false, /* skipEmpty = */ true)) {
            return false;
        }
        if (!NDrive::ReadFieldSchemeDefault(scheme, data, "tag_data_template", MutableTagDataTemplate(), /* required = */ false, /* schemeRequired = */ false, /* skipEmpty = */ true)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "enabled", MutableEnabled())) {
            return false;
        }
        return true;
    }

    template <typename TImpl>
    NJson::TJsonValue TFineTagConfig<TImpl>::SerializeToJson() const {
        NJson::TJsonValue result;
        TJsonProcessor::Write(result, "tag_name_template", GetTagNameTemplate());
        TJsonProcessor::Write(result, "tag_data_template", GetTagDataTemplate());
        TJsonProcessor::Write(result, "enabled", GetEnabled());
        return result;
    }

    class TFineDefaultTagConfig : public TFineTagConfig<TFineDefaultTagConfig> {
    public:
        class TSchemeSettings {
        public:
            static TString GetTagNameKey();
            static TString GetTagDataKey();
        };
    };

    class TFineChargeTagConfig : public TFineTagConfig<TFineChargeTagConfig> {
    public:
        class TSchemeSettings {
        public:
            static TString GetTagNameKey();
            static TString GetTagDataKey();
        };
    };

    class TFineMailTagConfig : public TFineTagConfig<TFineMailTagConfig> {
    public:
        class TSchemeSettings {
        public:
            static TString GetTagNameKey();
            static TString GetTagDataKey();
        };
    };

    class TFinePushTagConfig : public TFineTagConfig<TFinePushTagConfig> {
    public:
        class TSchemeSettings {
        public:
            static TString GetTagNameKey();
            static TString GetTagDataKey();
        };
    };

    class TFineRefundTagConfig : public TFineTagConfig<TFineRefundTagConfig> {
    public:
        class TSchemeSettings {
        public:
            static TString GetTagNameKey();
            static TString GetTagDataKey();
        };
    };
}
