#include "fine_config.h"

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/util/json_processing.h>

#include <util/generic/vector.h>
#include <util/string/join.h>
#include <util/string/split.h>

namespace {
    // originally similar is located in drive/backend/rt_background/document_builder/config.cpp
    static NDrive::TScheme GetTexBuilderConfigScheme() {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("folder").SetDefault(".");
        scheme.Add<TFSDuration>("convert_timeout").SetDefault(TDuration::Minutes(3));
        return scheme;
    }
}

namespace NDrive::NFine {
    const TString TFineFetchContextConfig::TViolationDetailedDocumentConfig::DefaultRobotUserId = "ec65f4f0-8fa8-4887-bfdc-ca01c9906696";  // robot-carsharing
    const TString TFineFetchContextConfig::TViolationDetailedDocumentConfig::DefaultAttachmentFilename = "documents.pdf";
    const TString TFineFetchContextConfig::TViolationDetailedDocumentConfig::DefaultTexTemplateName = "report_for_fine";
    const TString TFineFetchContextConfig::TViolationDetailedDocumentConfig::DefaultMDSBucketName = "carsharing-violations";
    const TString TFineFetchContextConfig::TViolationDetailedDocumentConfig::DefaultMDSFilePrefix = "session_detailed_documents";

    NDrive::TScheme TFineFetchContextConfig::TViolationDetailedDocumentConfig::GetScheme() {
        NDrive::TScheme scheme;

        scheme.Add<TFSBoolean>("use_cached_only", "Использовать только уже сгенерированные документы").SetDefault(true);

        scheme.Add<TFSString>("robot_user_id", "id робота для построения документов").SetDefault(DefaultRobotUserId);
        scheme.Add<TFSString>("attachment_filename", "Имя вложения в письме").SetDefault(DefaultAttachmentFilename);
        scheme.Add<TFSString>("mds_bucket_name", "Имя MDS бакета").SetDefault(DefaultMDSBucketName);
        scheme.Add<TFSString>("mds_file_prefix", "Префикс пути к файлу в MDS").SetDefault(DefaultMDSFilePrefix);
        scheme.Add<TFSString>("tex_template_name", "Имя TEX шаблона документа").SetDefault(DefaultTexTemplateName);

        scheme.Add<TFSStructure>("tex_builder_config", "Настройки построения документа с деталями о поездке").SetStructure(GetTexBuilderConfigScheme());

        return scheme;
    }

    bool TFineFetchContextConfig::TViolationDetailedDocumentConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TJsonProcessor::Read(data, "use_cached_only", UseCachedOnly)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "robot_user_id", RobotUserId)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "attachment_filename", AttachmentFilename)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "mds_bucket_name", MDSBucketName)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "mds_file_prefix", MDSFilePrefix)) {
            return false;
        }
        if (!TJsonProcessor::Read(data, "tex_template_name", TexTemplateName)) {
            return false;
        }
        if (!TexBuilderConfig.DeserializeFromJson(data["tex_builder_config"])) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue TFineFetchContextConfig::TViolationDetailedDocumentConfig::SerializeToJson() const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        TJsonProcessor::Write(result, "use_cached_only", UseCachedOnly);
        TJsonProcessor::Write(result, "robot_user_id", RobotUserId);
        TJsonProcessor::Write(result, "attachment_filename", AttachmentFilename);
        TJsonProcessor::Write(result, "mds_bucket_name", MDSBucketName);
        TJsonProcessor::Write(result, "mds_file_prefix", MDSFilePrefix);
        TJsonProcessor::Write(result, "tex_template_name", TexTemplateName);
        TJsonProcessor::WriteSerializable(result, "tex_builder_config", TexBuilderConfig);
        return result;
    }

    const TString TFineFetchContextConfig::DefaultRulingDateFormat = "%Y-%m-%d";
    const TString TFineFetchContextConfig::DefaultViolationTimeFormat = "%d.%m.%Y %H:%M";
    const TString TFineFetchContextConfig::DefaultViolationTimeZone = "Europe/Moscow";

    const TString TFineFetchContextConfig::PhotoBlockTitlePlaceHolder = "{fine.photo_block_title}";
    const TString TFineFetchContextConfig::DefaultPhotoBlockTitleTemplate = (
        "<span style='color:#949494'>" + TFineFetchContextConfig::PhotoBlockTitlePlaceHolder + "</span><br>"
    );
    const TVector<TString> TFineFetchContextConfig::DefaultPhotoBlockTitleVariants = { "Изображение с камеры", "Изображения с камеры" };

    const TString TFineFetchContextConfig::PhotoBlockImagePlaceHolder = "{fine.photo_url}";
    const TString TFineFetchContextConfig::DefaultPhotoBlockImageTemplate = (
        "<img src='" + TFineFetchContextConfig::PhotoBlockImagePlaceHolder + "' style='max-width:571px; padding-top: 7px;' alt='' border='0'><br>"
    );

    NDrive::TScheme TFineFetchContextConfig::GetScheme() {
        NDrive::TScheme scheme;

        scheme.Add<TFSString>("ruling_date_format", "Формат даты постановления").SetDefault(DefaultRulingDateFormat);
        scheme.Add<TFSString>("violation_time_format", "Формат времени нарушения").SetDefault(DefaultViolationTimeFormat);
        scheme.Add<TFSString>("violation_time_zone", "Таймзона времени нарушения (https://en.wikipedia.org/wiki/List_of_tz_database_time_zones)").SetDefault(DefaultViolationTimeZone);

        scheme.Add<TFSStructure>("violation_detailed_document_config", "Настройки построения документа с деталями о поездке").SetStructure(TViolationDetailedDocumentConfig::GetScheme());

        scheme.Add<TFSString>("photo_block_title_template", "Шаблон HTML-форматирования заголовка блока фото штрафов").SetDefault(DefaultPhotoBlockTitleTemplate);
        scheme.Add<TFSString>("photo_block_title_variants", "Заголовки блока фото штрафов в ед. ч. и мн. ч. через pipe (|)").SetDefault(JoinSeq("|", DefaultPhotoBlockTitleVariants));

        scheme.Add<TFSString>("photo_block_image_template", "Шаблон HTML-форматирования фото штрафа").SetDefault(DefaultPhotoBlockImageTemplate);

        return scheme;
    }

    bool TFineFetchContextConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (!TJsonProcessor::Read(data, "ruling_date_format", RulingDateFormat)) {
            return false;
        }

        if (!TJsonProcessor::Read(data, "violation_time_format", ViolationTimeFormat)) {
            return false;
        }

        if (data.Has("violation_time_zone")) {
            TString timeZoneName;
            if (!TJsonProcessor::Read(data, "violation_time_zone", timeZoneName)) {
                return false;
            }
            if (!NUtil::GetTimeZone(timeZoneName, ViolationTimeZone)) {
                return false;
            }
        }

        if (!ViolationDetailedDocumentConfig.DeserializeFromJson(data["violation_detailed_document_config"])) {
            return false;
        }

        if (!TJsonProcessor::Read(data, "photo_block_title_template", PhotoBlockTitleTemplate)) {
            return false;
        }
        if (!PhotoBlockTitleTemplate.Contains(PhotoBlockTitlePlaceHolder)) {
            return false;
        }

        if (data.Has("photo_block_title_variants")) {
            TString photoBlockTitleVariantsStr;
            if (!TJsonProcessor::Read(data, "photo_block_title_variants", photoBlockTitleVariantsStr)) {
                return false;
            }

            PhotoBlockTitleVariants = StringSplitter(photoBlockTitleVariantsStr).Split('|');
            if (PhotoBlockTitleVariants.size() != 2) {
                return false;
            }
        }

        if (!TJsonProcessor::Read(data, "photo_block_image_template", PhotoBlockImageTemplate)) {
            return false;
        }
        if (!PhotoBlockImageTemplate.Contains(PhotoBlockImagePlaceHolder)) {
            return false;
        }

        return true;
    }

    NJson::TJsonValue TFineFetchContextConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        TJsonProcessor::Write(result, "ruling_date_format", RulingDateFormat);
        TJsonProcessor::Write(result, "violation_time_format", ViolationTimeFormat);
        TJsonProcessor::Write(result, "violation_time_zone", TString(ViolationTimeZone.name()));
        TJsonProcessor::WriteSerializable(result, "violation_detailed_document_config", ViolationDetailedDocumentConfig);
        TJsonProcessor::Write(result, "photo_block_title_template", PhotoBlockTitleTemplate);
        TJsonProcessor::Write(result, "photo_block_title_variants", JoinSeq("|", PhotoBlockTitleVariants));
        TJsonProcessor::Write(result, "photo_block_image_template", PhotoBlockImageTemplate);
        return result;
    }
}
