#include "fine_context.h"

#include <drive/backend/fines/config.h>
#include <drive/backend/fines/autocode/entries.h>

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/doc_packages/manager.h>
#include <drive/backend/notifications/startrek/startrek.h>
#include <drive/backend/users/user.h>

#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/raw_text/datetime.h>
#include <drive/library/cpp/tex_builder/tex_builder.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <rtline/util/algorithm/ptr.h>

#include <util/generic/utility.h>
#include <util/string/builder.h>
#include <util/string/subst.h>

namespace NDrive::NFine {
    // fine fetchers

    TString TFineIdContextFetcher::GetTypeName() {
        return TypeNamePrefix + "id";
    }

    TFineIdContextFetcher::TRegistrator TFineIdContextFetcher::Registrator;

    bool TFineIdContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = context.GetEntry().GetId();
        return true;
    }

    TString TFineRulingNumberContextFetcher::GetTypeName() {
        return TypeNamePrefix + "ruling_number";
    }

    TFineRulingNumberContextFetcher::TRegistrator TFineRulingNumberContextFetcher::Registrator;

    bool TFineRulingNumberContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = context.GetEntry().GetRulingNumber();
        return true;
    }

    TString TFineSourceTypeContextFetcher::GetTypeName() {
        return TypeNamePrefix + "source_type";
    }

    TFineSourceTypeContextFetcher::TRegistrator TFineSourceTypeContextFetcher::Registrator;

    bool TFineSourceTypeContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = context.GetEntry().GetSourceType();
        return true;
    }

    TString TFineArticleContextFetcher::GetTypeName() {
        return TypeNamePrefix + "article";
    }

    TFineArticleContextFetcher::TRegistrator TFineArticleContextFetcher::Registrator;

    bool TFineArticleContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = context.GetEntry().GetArticleKoap();
        return true;
    }

    TString TFineArticleCodeContextFetcher::GetTypeName() {
        return TypeNamePrefix + "article_code";
    }

    TFineArticleCodeContextFetcher::TRegistrator TFineArticleCodeContextFetcher::Registrator;

    bool TFineArticleCodeContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = context.GetEntry().GetArticleCode();
        if (!result) {
            auto articleMatcherPtr = context.GetServer().GetDriveAPI()->GetFinesManager().GetFineArticleMatcherPtr(/* ensureActual = */ false);
            return !!articleMatcherPtr && articleMatcherPtr->DetermineArticle(context.GetEntry().GetArticleKoap(), result);
        }
        return true;
    }

    // fine violation session details fetchers

    TString TFineSessionIdContextFetcher::GetTypeName() {
        return TypeNamePrefix + "session_id";
    }

    TFineSessionIdContextFetcher::TRegistrator TFineSessionIdContextFetcher::Registrator;

    bool TFineSessionIdContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = context.GetEntry().GetSessionId();
        return true;
    }

    TString TFineUserIdContextFetcher::GetTypeName() {
        return TypeNamePrefix + "user_id";
    }

    TFineUserIdContextFetcher::TRegistrator TFineUserIdContextFetcher::Registrator;

    bool TFineUserIdContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = context.GetEntry().GetUserId();
        return true;
    }

    TString TFineFullUserNameContextFetcher::GetTypeName() {
        return TypeNamePrefix + "full_user_name";
    }

    TFineFullUserNameContextFetcher::TRegistrator TFineFullUserNameContextFetcher::Registrator;

    bool TFineFullUserNameContextFetcher::FetchUser(const IFineContextFetcher::TContextType& context, TDriveUserData& userData, TMessagesCollector& errors) const {
        CHECK_WITH_LOG(!!context.GetServer().GetDriveAPI());
        auto userDataPtr = context.GetServer().GetDriveAPI()->GetUsersData()->GetCachedObject(context.GetEntry().GetUserId());
        if (userDataPtr) {
            userData = std::move(*userDataPtr);
            return true;
        }
        errors.AddMessage(__LOCATION__, "Cannot restore user data for fine " + context.GetEntry().GetId());
        return false;
    }

    bool TFineFullUserNameContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        TDriveUserData userData;
        if (!FetchUser(context, userData, errors)) {
            return false;
        }
        result = userData.GetFullName();
        return true;
    }

    // fine violation details fetchers

    TString TFineRulingDateContextFetcher::GetTypeName() {
        return TypeNamePrefix + "ruling_date";
    }

    TFineRulingDateContextFetcher::TRegistrator TFineRulingDateContextFetcher::Registrator;

    bool TFineRulingDateContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        TInstant rulingDate = context.GetEntry().GetRulingDate();
        result = NUtil::FormatDatetime(rulingDate, context.GetConfig().GetRulingDateFormat());
        return true;
    }

    TString TFineLawnParkingDeadlineContextFetcher::GetTypeName() {
        return TypeNamePrefix + "lawn_parking_deadline";
    }

    TFineLawnParkingDeadlineContextFetcher::TRegistrator TFineLawnParkingDeadlineContextFetcher::Registrator;

    bool TFineLawnParkingDeadlineContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        TInstant rulingDate = context.GetEntry().GetRulingDate();
        TInstant deadline = rulingDate + TDuration::Days(5) + TDuration::Days(Max(0, 2 - static_cast<i32>(NUtil::GetWeekDay(rulingDate))));  // first working day past 5 added
        result = NUtil::FormatDatetime(deadline, context.GetConfig().GetRulingDateFormat());
        return true;
    }

    TString TFineViolationPlaceContextFetcher::GetTypeName() {
        return TypeNamePrefix + "violation_place";
    }

    TFineViolationPlaceContextFetcher::TRegistrator TFineViolationPlaceContextFetcher::Registrator;

    bool TFineViolationPlaceContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = context.GetEntry().GetViolationPlace();
        return true;
    }

    TString TFineViolationTimeContextFetcher::GetTypeName() {
        return TypeNamePrefix + "violation_time";
    }

    TFineViolationTimeContextFetcher::TRegistrator TFineViolationTimeContextFetcher::Registrator;

    bool TFineViolationTimeContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        TInstant absoluteTime = context.GetEntry().GetViolationTime();
        TInstant localTime = NUtil::ConvertTimeZone(absoluteTime, NUtil::GetUTCTimeZone(), context.GetConfig().GetViolationTimeZone());
        result = NUtil::FormatDatetime(localTime, context.GetConfig().GetViolationTimeFormat());
        return true;
    }

    TString TFineViolationPhotoBlockContextFetcher::GetTypeName() {
        return TypeNamePrefix + "violation_photo_block";
    }

    TFineViolationPhotoBlockContextFetcher::TRegistrator TFineViolationPhotoBlockContextFetcher::Registrator;

    bool TFineViolationPhotoBlockContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        TStringBuilder resultBuf;

        const auto& config = context.GetConfig();

        const auto& finePhotos = context.GetPrefetchedPhotos(errors);
        if (!finePhotos) {
            return false;
        }

        if (finePhotos->empty()) {  // no photo block
            result = "";
            return true;
        }

        size_t blockTitleIdx = static_cast<size_t>(finePhotos->size() > 1);
        TString blockTitle = config.GetPhotoBlockTitleVariants()[blockTitleIdx];

        resultBuf << SubstGlobalCopy(config.GetPhotoBlockTitleTemplate(), config.PhotoBlockTitlePlaceHolder, blockTitle);

        for (const auto& finePhoto : *finePhotos) {
            resultBuf << SubstGlobalCopy(config.GetPhotoBlockImageTemplate(), config.PhotoBlockImagePlaceHolder, finePhoto.GetUrl());
        }

        result = resultBuf;
        return true;
    }

    TString TFineViolationDetailedDocumentContextFetcher::GetTypeName() {
        return TypeNamePrefix + "violation_detailed_document";
    }

    TFineViolationDetailedDocumentContextFetcher::TRegistrator TFineViolationDetailedDocumentContextFetcher::Registrator;

    bool TFineViolationDetailedDocumentContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        const auto& documentConfig = context.GetConfig().GetViolationDetailedDocumentConfig();

        result = "";  // no attachments in case of error

        TString fileUrl;

        TFineViolationDetailedDocumentUrlContextFetcher fileUrlFetcher;
        if (!fileUrlFetcher.Fetch(context, fileUrl, errors)) {
            return false;
        }

        NJson::TJsonValue attachmentJson(NJson::JSON_MAP);
        attachmentJson["mime_type"] = "application/pdf";
        attachmentJson["filename"] = documentConfig.GetAttachmentFilename();
        attachmentJson["url"] = fileUrl;

        result = attachmentJson.GetStringRobust();
        return true;
    }

    TString TFineViolationDetailedDocumentUrlContextFetcher::GetTypeName() {
        return TypeNamePrefix + "violation_detailed_document_url";
    }

    TFineViolationDetailedDocumentUrlContextFetcher::TRegistrator TFineViolationDetailedDocumentUrlContextFetcher::Registrator;

    bool TFineViolationDetailedDocumentUrlContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        const auto& documentConfig = context.GetConfig().GetViolationDetailedDocumentConfig();

        result = "";  // no attachments in case of error

        TString cachedFileUrl;
        if (!context.GetEntry().GetCachedViolationDetailedDocumentUrl(cachedFileUrl, "")) {
            errors.AddMessage(__LOCATION__, "Cannot get cached violation detailed document from fine " + context.GetEntry().GetId());
            return false;
        }

        if (!cachedFileUrl && documentConfig.IsUseCachedOnly()) {
            errors.AddMessage(__LOCATION__, "Violation detailed document is not cached but cache only requested for fine " + context.GetEntry().GetId());
            return false;
        }

        if (!!cachedFileUrl) {
            result = cachedFileUrl;
        } else {
            TBlob document;
            if (!GetDocument(context, document, errors)) {  // note: method is synchronous and quite long time executing
                return false;
            }

            if (!UploadToMDS(context, document, result, errors)) {  // note: current implementation is synchronous
                return false;
            }
        }

        return true;
    }

    bool TFineViolationDetailedDocumentUrlContextFetcher::GetDocument(const IFineContextFetcher::TContextType& context, TBlob& document, TMessagesCollector& errors) const {
        const auto& documentConfig = context.GetConfig().GetViolationDetailedDocumentConfig();
        const auto* driveApiPtr = context.GetServer().GetDriveAPI();

        TTexServerDocBuilder texBuilder(documentConfig.GetTexBuilderConfig());
        if (!context.GetServer().GetDocumentsManager() || !context.GetServer().GetDocumentsManager()->InitializeTexBuilder(texBuilder)) {
            errors.AddMessage(__LOCATION__, "Cannot initialize TEX document builder");
            return false;
        }

        NJson::TJsonValue jsonDocument = NJson::JSON_MAP;
        jsonDocument.InsertValue("document_name", documentConfig.GetTexTemplateName());
        jsonDocument.InsertValue("session_id", context.GetEntry().GetSessionId());

        TUserPermissions::TPtr userPermissions = driveApiPtr->GetUserPermissions(documentConfig.GetRobotUserId(), TUserPermissionsFeatures());

        document = context.GetServer().GetDocumentsManager()->BuildDocument(jsonDocument, context.GetServer(), texBuilder, userPermissions, errors);

        if (document.Empty()) {
            errors.AddMessage(__LOCATION__, "Error building violation detailed document");
            return false;
        }

        return true;
    }

    bool TFineViolationDetailedDocumentUrlContextFetcher::UploadToMDS(const IFineContextFetcher::TContextType& context, const TBlob& document, TString& fileUrl, TMessagesCollector& errors) const {
        const auto& documentConfig = context.GetConfig().GetViolationDetailedDocumentConfig();
        const auto& mdsClient = context.GetServer().GetDriveAPI()->GetMDSClient();

        TString mdsBucketName = documentConfig.GetMDSBucketName();

        const TS3Client::TBucket* mdsBucketPtr = mdsClient.GetBucket(mdsBucketName);
        if (mdsBucketPtr == nullptr) {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "MDS bucket " << mdsBucketName << " does not exists");
            return false;
        }

        TString fileName;
        if (!context.GetEntry().GenerateViolationDetailedDocumentFilePath(documentConfig.GetMDSFilePrefix(), fileName)) {
            errors.AddMessage(__LOCATION__, "Error generating document file name: check fine session_id: " + context.GetEntry().GetId());
            return false;
        }

        TString documentData = TString(document.AsCharPtr(), document.Length());
        if (mdsBucketPtr->PutKey(fileName, documentData, errors) != HTTP_OK) {
            errors.AddMessage(__LOCATION__, TStringBuilder()
                                            << "Error uploading document into bucket " << mdsBucketName
                                            << " by path " << fileName << " : " << errors.GetStringReport());
            return false;
        }

        fileUrl = mdsClient.GetTmpFilePath(mdsBucketName, fileName);
        return true;
    }

    TString TFineChargeTagNameContextFetcher::GetTypeName() {
        return TypeNamePrefix + "dynamic_charge_tag_name";
    }

    TFineChargeTagNameContextFetcher::TRegistrator TFineChargeTagNameContextFetcher::Registrator;

    bool TFineChargeTagNameContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        TString articleCode;

        TFineArticleCodeContextFetcher articleCodeFetcher;  // backward compatibility for old fines
        if (!articleCodeFetcher.Fetch(context, articleCode, errors)) {
            return false;
        }

        result = TString{"fine_pdd_"} + articleCode;  // could be customized depend on article code
        return true;
    }

    TString TFineMailTagNameContextFetcher::GetTypeName() {
        return TypeNamePrefix + "dynamic_mail_tag_name";
    }

    TFineMailTagNameContextFetcher::TRegistrator TFineMailTagNameContextFetcher::Registrator;

    bool TFineMailTagNameContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        const auto& fine = context.GetEntry();

        TInstant violationTime = fine.GetViolationTime();
        if (!violationTime) {
            return false;
        }

        TString articleCode;

        TFineArticleCodeContextFetcher articleCodeFetcher;  // backward compatibility for old fines
        if (!articleCodeFetcher.Fetch(context, articleCode, errors)) {
            return false;
        }

        auto fineConstructorConfig = TFineConstructorConfig::Construct();
        if (!fineConstructorConfig) {
            return false;
        }

        bool appealPossible = true;
        {
            TDuration elapsedTime = ModelingNow() - violationTime;
            appealPossible = (elapsedTime < TDuration::Days(14));
        }

        bool hasDiscount = !!fine.GetDiscountDate();
        bool assumeRuleSecondViolation = false;
        {
            const auto& explicitDiscounts = fineConstructorConfig->GetExplicitDiscounts();
            if (auto discountInfoPtr = explicitDiscounts.FindPtr(articleCode)) {
                hasDiscount = discountInfoPtr->GetDiscountPercent() > 0;
                assumeRuleSecondViolation = discountInfoPtr->GetAssumeRuleSecondViolation();
            }
        }

        bool isInappropriateCityParking = false;
        {
            if (fineConstructorConfig->GetInappropriateCityParkingConfig().GetFineArticleCodes().contains(articleCode)) {
                isInappropriateCityParking = true;
            } else if (fineConstructorConfig->GetInappropriateLongTermParkingConfig().GetFineArticleCodes().contains(articleCode)) {
                isInappropriateCityParking = true;
            }
        }

        bool hasDocumentsToAttach = false;
        {
            TString violationDetailedDocumentUrl;
            hasDocumentsToAttach = fine.GetCachedViolationDetailedDocumentUrl(violationDetailedDocumentUrl, "") && !!violationDetailedDocumentUrl;
        }

        if (isInappropriateCityParking) {
            result = "email_fine_pdd_parking";
        } else if (appealPossible) {
            if (hasDiscount) {
                result = "email_fine_pdd";
            } else if (assumeRuleSecondViolation) {
                result = "email_fine_pdd_2nd_violation";
            } else {
                result = "email_fine_pdd_no_discount";
            }
        } else if (hasDocumentsToAttach) {
            if (hasDiscount) {
                result = "email_fine_pdd_3m";
            } else if (assumeRuleSecondViolation) {
                result = "email_fine_pdd_3m_2nd_violation";
            } else {
                result = "email_fine_pdd_3m_no_discount";
            }
        } else {
            if (hasDiscount) {
                result = "email_fine_pdd_2w";
            } else if (assumeRuleSecondViolation) {
                result = "email_fine_pdd_2w_2nd_violation";
            } else {
                result = "email_fine_pdd_2w_no_discount";
            }
        }

        return true;
    }

    // fine payment total fetchers

    TString TFinePaymentTotalContextFetcher::GetTypeName() {
        return TypeNamePrefix + "payment_total";
    }

    TFinePaymentTotalContextFetcher::TRegistrator TFinePaymentTotalContextFetcher::Registrator;

    bool TFinePaymentTotalContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = ToString(context.GetEntry().GetSumToPay());
        return true;
    }

    TString TFinePaymentCentsTotalContextFetcher::GetTypeName() {
        return TypeNamePrefix + "payment_cents_total";
    }

    TFinePaymentCentsTotalContextFetcher::TRegistrator TFinePaymentCentsTotalContextFetcher::Registrator;

    bool TFinePaymentCentsTotalContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = ToString(context.GetEntry().GetSumToPayCents());
        return true;
    }

    TString TFinePaymentTotalWithoutDiscountContextFetcher::GetTypeName() {
        return TypeNamePrefix + "payment_total_without_discount";
    }

    TFinePaymentTotalWithoutDiscountContextFetcher::TRegistrator TFinePaymentTotalWithoutDiscountContextFetcher::Registrator;

    bool TFinePaymentTotalWithoutDiscountContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        result = ToString(context.GetEntry().GetSumToPayWithoutDiscount());
        return true;
    }

    // fine charge context fetchers

    TString TFineChargeTagIdContextFetcher::GetTypeName() {
        return TypeNamePrefix + "charge_tag_id";
    }

    TFineChargeTagIdContextFetcher::TRegistrator TFineChargeTagIdContextFetcher::Registrator;

    bool TFineChargeTagIdContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& /* errors */) const {
        NJson::TJsonValue chargeTagIdJson = context.GetEntry().GetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::ChargeTagId);
        if (chargeTagIdJson.IsString() && !!chargeTagIdJson.GetString()) {
            result = chargeTagIdJson.GetString();
            return true;
        }
        return false;
    }

    // fine car context fetchers

    const TString IFineCarContextFetcher::TypeNamePrefix = IFineCarContextFetcher::TBase::TypeNamePrefix + "car.";

    bool IFineCarContextFetcher::FetchCar(const IFineContextFetcher::TContextType& context, TDriveCarInfo& carData, TMessagesCollector& errors) const {
        CHECK_WITH_LOG(!!context.GetServer().GetDriveAPI());
        auto gCars = context.GetServer().GetDriveAPI()->GetCarsData()->FetchInfo(context.GetEntry().GetCarId(), TInstant::Zero());
        auto carDataPtr = gCars.GetResultPtr(context.GetEntry().GetCarId());
        if (carDataPtr != nullptr) {
            carData = std::move(*carDataPtr);
            return true;
        }
        errors.AddMessage(__LOCATION__, "Cannot restore car data for fine " + context.GetEntry().GetId());
        return false;
    }

    TString TFineCarModelContextFetcher::GetTypeName() {
        return TypeNamePrefix + "model";
    }

    TFineCarModelContextFetcher::TRegistrator TFineCarModelContextFetcher::Registrator;

    bool TFineCarModelContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        TDriveCarInfo carData;
        if (!FetchCar(context, carData, errors)) {
            return false;
        }

        TString modelCode = carData.GetModel();

        auto modelFetchResult = context.GetServer().GetDriveAPI()->GetModelsData()->FetchInfo(modelCode, TInstant::Zero());
        auto modelPtr = modelFetchResult.GetResultPtr(modelCode);
        if (!modelPtr) {
            result = modelCode;
        } else {
            result = modelPtr->GetName();
        }

        return true;
    }

    TString TFineCarNumberContextFetcher::GetTypeName() {
        return TypeNamePrefix + "number";
    }

    TFineCarNumberContextFetcher::TRegistrator TFineCarNumberContextFetcher::Registrator;

    bool TFineCarNumberContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        TDriveCarInfo carData;
        if (!FetchCar(context, carData, errors)) {
            return false;
        }
        result = carData.GetNumber();
        return true;
    }

    TString TFineCarVinContextFetcher::GetTypeName() {
        return TypeNamePrefix + "vin";
    }

    TFineCarVinContextFetcher::TRegistrator TFineCarVinContextFetcher::Registrator;

    bool TFineCarVinContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        TDriveCarInfo carData;
        if (!FetchCar(context, carData, errors)) {
            return false;
        }
        result = carData.GetVin();
        return true;
    }

    // startrek integration

    TString TFineLastStartrekTicketKeyContextFetcher::GetTypeName() {
        return TypeNamePrefix + "last_startrek_ticket_key";
    }

    TFineLastStartrekTicketKeyContextFetcher::TRegistrator TFineLastStartrekTicketKeyContextFetcher::Registrator;

    bool TFineLastStartrekTicketKeyContextFetcher::Fetch(const IFineContextFetcher::TContextType& context, TString& result, TMessagesCollector& errors) const {
        TString rulingNumber = context.GetEntry().GetRulingNumber();

        auto notifyResultProvider = context.GetNotifyResultProvider();
        if (notifyResultProvider) {
            auto notifyResultPtr = notifyResultProvider->GetHandlingResult(rulingNumber);
            auto startrekNotifyResultPtr = std::dynamic_pointer_cast<TStartrekNotifierResult>(notifyResultPtr);
            if (startrekNotifyResultPtr) {
                auto& ticket = startrekNotifyResultPtr->GetTicket();
                result = ticket.GetKey();
                return !!result;
            }
        }

        auto startrekClientPtr = context.GetServer().GetStartrekClient();
        if (startrekClientPtr) {
            TStartrekClient::TTickets tickets;
            if (!startrekClientPtr->SearchIssue(rulingNumber, tickets, errors)) {
                return false;
            }

            TString queue = GetSpecificStartrekQueue();
            for (auto&& ticket : tickets) {
                if (!queue || queue == ticket.GetQueue()) {
                    result = ticket.GetKey();
                    return !!result;  // simply first found
                }
            }
        }

        return false;
    }

    TString TFineLastStartrekTicketKeyContextFetcher::GetSpecificStartrekQueue() const {
        return (Parameters.size() > 0) ? Parameters[0] : "";
    }
}
