#include "filters.h"

#include "article_matcher.h"
#include "util/generic/fwd.h"

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/util/instant_model.h>

#include <util/generic/algorithm.h>


namespace {
    void AppendCustomQueryCondition(NSQL::TQueryOptions& options, const TString& cond) {
        if (options.GetCustomCondition()) {
            options.MutableCustomCondition() += " AND (" + cond + ")";
        } else {
            options.AddCustomCondition(cond);
        }
    }
}

namespace NDrive::NFine {
    void TFineFilterGroup::Append(IFineFilter::TPtr filter) {
        Filters.push_back(filter);
    }

    bool TFineFilterGroup::Match(const TAutocodeFineEntry& entry) const {
        return AllOf(Filters, [&entry](IFineFilter::TPtr f) { return f->Match(entry); });
    }

    void TFineFilterGroup::PatchQuery(NSQL::TQueryOptions& options) const {
        for (auto&& f : Filters) {
            f->PatchQuery(options);
        }
    }

    bool TRejectFilter::Match(const TAutocodeFineEntry& /* entry */) const {
        return false;
    }

    void TRejectFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        AppendCustomQueryCondition(options, "False");
    }

    bool TFineChargeableFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetNeedsCharge();
    }

    void TFineChargeableFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("needs_charge", true);
    }

    bool TFineNotChargedFilter::Match(const TAutocodeFineEntry& entry) const {
        return !entry.GetChargedAt();
    }

    void TFineNotChargedFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("charged_at", TNull());
    }

    bool TFineChargedFilter::Match(const TAutocodeFineEntry& entry) const {
        return !!entry.GetChargedAt();
    }

    void TFineChargedFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("charged_at", NSQL::TNot<TNull>());
    }

    bool TFineChargeNotPassedFilter::Match(const TAutocodeFineEntry& entry) const {
        return !entry.GetChargePassedAt();
    }

    void TFineChargeNotPassedFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("charge_passed_at", TNull());
    }

    bool TFineCameraFixationFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetIsCameraFixation();
    }

    void TFineCameraFixationFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("is_camera_fixation", true);
    }

    bool TFineExplicitPhotoFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetHasPhoto();
    }

    void TFineExplicitPhotoFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("has_photo", true);
    }

    bool TFineAbsentDetailedViolationDocumentFilter::Match(const TAutocodeFineEntry& entry) const {
        TString fileUrl;
        return entry.GetCachedViolationDetailedDocumentUrl(fileUrl, "") && !fileUrl;
    }

    void TFineAbsentDetailedViolationDocumentFilter::PatchQuery(NSQL::TQueryOptions& /* options */) const {
    }

    bool TFineHasDetailedViolationDocumentFilter::Match(const TAutocodeFineEntry& entry) const {
        TString fileUrl;
        return entry.GetCachedViolationDetailedDocumentUrl(fileUrl, "") && !!fileUrl;
    }

    void TFineHasDetailedViolationDocumentFilter::PatchQuery(NSQL::TQueryOptions& /* options */) const {
    }

    bool TFineExplicitDecreeFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetHasDecree();
    }

    void TFineExplicitDecreeFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        AppendCustomQueryCondition(options, "meta_info::json->>'has_decree' = 'true'");
    }

    bool TFineUnknownIncludeToBillInfoFilter::Match(const TAutocodeFineEntry& entry) const {
        auto willBeIncludedToBillInfo = entry.GetMetaInfoProperty(TAutocodeFineEntry::EMetaInfoProperty::WillBeIncludedToBill);
        return !willBeIncludedToBillInfo.IsDefined();
    }

    void TFineUnknownIncludeToBillInfoFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        AppendCustomQueryCondition(options, "meta_info::json->>'will_be_included_to_bill' IS NULL");
    }

    TFineSumToPayCentsFilter::TFineSumToPayCentsFilter(const i64 maxAmount)
        : TFineSumToPayCentsFilter(0.0, maxAmount)
    {
    }

    TFineSumToPayCentsFilter::TFineSumToPayCentsFilter(const i64 minAmount, const i64 maxAmount)
        : MinAmount(minAmount)
        , MaxAmount(maxAmount)
    {
    }

    bool TFineSumToPayCentsFilter::Match(const TAutocodeFineEntry& entry) const {
        return MinAmount <= entry.GetSumToPayCents() && entry.GetSumToPayCents() <= MaxAmount;
    }

    void TFineSumToPayCentsFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        AppendCustomQueryCondition(options, ToString(MinAmount) + " <= sum_to_pay * 100");
        AppendCustomQueryCondition(options, "sum_to_pay * 100 <= " + ToString(MaxAmount));
    }

    TFineSerialIdFilter::TFineSerialIdFilter(const ui32 minSerialId, const ui32 maxSerialId)
        : MinSerialId(minSerialId)
        , MaxSerialId(maxSerialId)
    {
    }

    bool TFineSerialIdFilter::Match(const TAutocodeFineEntry& entry) const {
        return MinSerialId <= entry.GetSerialId() && entry.GetSerialId() <= MaxSerialId;
    }

    void TFineSerialIdFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("serial_id", MakeRange<ui64>(MinSerialId, MaxSerialId));
    }

    TSourceKeyFilter::TSourceKeyFilter(const i64 id)
        : SourceKey(id)
    {
    }

    bool TSourceKeyFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetAutocodeId() == SourceKey;
    }

    void TSourceKeyFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.AddGenericCondition("autocode_id", ToString(SourceKey));
    }

    TFineUserFilter::TFineUserFilter(const TString& userId)
        : UserId(userId)
    {
    }

    bool TFineUserFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetUserId() == UserId;
    }

    void TFineUserFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.AddGenericCondition("user_id", UserId);
    }

    TFineMultipleUsersFilter::TFineMultipleUsersFilter(const TSet<TString>& userIds, const bool contains)
        : UserIds(userIds)
        , Contains(contains)
    {
    }

    void TFineMultipleUsersFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        if (Contains) {
            options.SetGenericCondition("user_id", UserIds);
        } else {
            options.SetGenericCondition("user_id", NSQL::Not(UserIds));
        }
    }

    bool TFineMultipleUsersFilter::Match(const TAutocodeFineEntry& entry) const {
        return Contains == UserIds.contains(entry.GetUserId());
    }

    TFineCarFilter::TFineCarFilter(const TString& carId)
        : CarId(carId)
    {
    }

    bool TFineCarFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetCarId() == CarId;
    }

    void TFineCarFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.AddGenericCondition("car_id", CarId);
    }

    TFineMultipleSessionsFilter::TFineMultipleSessionsFilter(const TSet<TString>& sessionIds)
        : SessionIds(sessionIds)
    {
    }

    bool TFineMultipleSessionsFilter::Match(const TAutocodeFineEntry& entry) const {
        if (entry.GetSessionId() && SessionIds.contains(entry.GetSessionId())) {
            return true;
        }
        if (entry.GetOrderId() && SessionIds.contains(entry.GetOrderId())) {
            return true;
        }
        return false;
    }

    void TFineMultipleSessionsFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.MutableSubOrOptions().emplace_back(NSQL::TQueryOptions().SetGenericCondition("session_id", SessionIds));
        options.MutableSubOrOptions().emplace_back(NSQL::TQueryOptions().SetGenericCondition("order_id", SessionIds));
    }

    TFineMultipleCarsFilter::TFineMultipleCarsFilter(const TSet<TString>& carIds)
        : CarIds(carIds)
    {
    }

    bool TFineMultipleCarsFilter::Match(const TAutocodeFineEntry& entry) const {
        return CarIds.contains(entry.GetCarId());
    }

    void TFineMultipleCarsFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("car_id", CarIds);
    }

    TFineSourceTypeFilter::TFineSourceTypeFilter(const ESourceType& sourceType)
        : SourceType(ToString(sourceType))
    {
    }

    TFineIdFilter::TFineIdFilter(const TString& fineId)
        : FineId(fineId)
    {
    }

    bool TFineIdFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetId() == FineId;
    }

    void TFineIdFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.AddGenericCondition("id", FineId);
    }

    bool TMultipleFineIdsFilter::Match(const TAutocodeFineEntry& entry) const {
        return FineIds.contains(entry.GetId());
    }

    void TMultipleFineIdsFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("id", FineIds);
    }

    TFineRulingNumberFilter::TFineRulingNumberFilter(const TString& rulingNumber)
        : RulingNumber(rulingNumber)
    {
    }

    bool TFineRulingNumberFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetRulingNumber() == RulingNumber;
    }

    void TFineRulingNumberFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.AddGenericCondition("ruling_number", RulingNumber);
    }

    TFineMultipleRulingNumbersFilter::TFineMultipleRulingNumbersFilter(const TSet<TString>& rulingNumbers)
        : RulingNumbers(rulingNumbers)
    {
    }

    bool TFineMultipleRulingNumbersFilter::Match(const TAutocodeFineEntry& entry) const {
        return RulingNumbers.contains(entry.GetRulingNumber());
    }

    void TFineMultipleRulingNumbersFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("ruling_number", RulingNumbers);
    }

    bool TFineSourceTypeFilter::Match(const TAutocodeFineEntry& entry) const {
        return entry.GetSourceType() == SourceType;
    }

    void TFineSourceTypeFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.AddGenericCondition("source_type", SourceType);
    }

    TFineMultipleSourcesTypeFilter::TFineMultipleSourcesTypeFilter(const TSet<ESourceType>& sourceTypes) {
        for (const auto& sourceType : sourceTypes) {
            SourceTypes.emplace(ToString(sourceType));
        }
    }

    bool TFineMultipleSourcesTypeFilter::Match(const TAutocodeFineEntry& entry) const {
        return SourceTypes.contains(entry.GetSourceType());
    }

    void TFineMultipleSourcesTypeFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        options.SetGenericCondition("source_type", SourceTypes);
    }

    TFineDateTimeFilter::TFineDateTimeFilter(const TInstant since, const TInstant until)
        : Since(since), Until(until)
    {
    }

    bool TFineDateTimeFilter::Match(const TAutocodeFineEntry& entry) const {
        const auto timestamp = GetActualTimestamp(entry);
        if (!timestamp) {
            return false;
        }
        if (!!Since && timestamp < Since) {
            return false;
        }
        if (!!Until && timestamp >= Until) {
            return false;
        }
        return true;
    }

    void TFineDateTimeFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        TRange<ui64> range(Since.Seconds());
        if (Until) {
            range.To = Until.Seconds();
        }
        options.SetGenericCondition("EXTRACT(EPOCH FROM " + GetFieldTs() + ")", range);
    }

    bool IDateTimeFilterConfig::operator !() const {
        return !HasRestrictions();
    }

    bool IDateTimeFilterConfig::HasRestrictions() const {
        return ((!!Since && Since != TInstant::Max()) ||
                (!!Until && Until != TInstant::Max()) ||
                (!!For && For != TDuration::Max()) ||
                (!!To && To != TDuration::Max()));
    }

    NDrive::TScheme IDateTimeFilterConfig::GetScheme() {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("since", "Не раньше чем");
        scheme.Add<TFSString>("until", "Не позже чем");
        scheme.Add<TFSDuration>("for", "Со стольки с текущего момента");
        scheme.Add<TFSDuration>("to", "До стольки с текущего момента");
        return scheme;
    }

    IFineFilter::TPtr IDateTimeFilterConfig::GetFilter() const {
        TInstant since, until;

        if (!!Since && Since != TInstant::Max()) {
            since = Since;
        } else if (!!For && For != TDuration::Max()) {
            since = ModelingNow() - For;
        } else {
            since = TInstant::Zero();
        }

        if (!!Until && Until != TInstant::Max()) {
            until = Until;
        } else if (!!To && To != TDuration::Max()) {
            until = ModelingNow() - To;
        } else {
            until = TInstant::Max();
        }

        return DoGetFilter(since, until);
    }

    bool IDateTimeFilterConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
        if (data["since"].IsDefined() && !!data["since"].GetStringRobust() && !TJsonProcessor::Read(data, "since", Since)) {
            return false;
        }
        if (data["until"].IsDefined() && !!data["until"].GetStringRobust() && !TJsonProcessor::Read(data, "until", Until)) {
            return false;
        }
        if (data["for"].IsDefined() && !!data["for"].GetStringRobust() && !TJsonProcessor::Read(data, "for", For)) {
            return false;
        }
        if (data["to"].IsDefined() && !!data["to"].GetStringRobust() && !TJsonProcessor::Read(data, "to", To)) {
            return false;
        }
        return true;
    }

    NJson::TJsonValue IDateTimeFilterConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        if (!!Since && Since != TInstant::Max()) {
            TJsonProcessor::Write(result, "since", Since.ToString());
        }
        if (!!Until && Until != TInstant::Max()) {
            TJsonProcessor::Write(result, "until", Until.ToString());
        }
        if (!!For && For != TDuration::Max()) {
            TJsonProcessor::WriteDurationString(result, "for", For);
        }
        if (!!To && To != TDuration::Max()) {
            TJsonProcessor::WriteDurationString(result, "to", To);
        }
        return result;
    }

    TInstant TFineRulingDateFilter::GetActualTimestamp(const TAutocodeFineEntry& entry) const {
        return entry.GetRulingDate();
    }

    TString TFineRulingDateFilter::GetFieldTs() const {
        return "ruling_date";
    }

    TInstant TFineViolationTimeFilter::GetActualTimestamp(const TAutocodeFineEntry& entry) const {
        return entry.GetViolationTime();
    }

    TString TFineViolationTimeFilter::GetFieldTs() const {
        return "violation_time";
    }

    TInstant TFineReceiveTimeFilter::GetActualTimestamp(const TAutocodeFineEntry& entry) const {
        return entry.GetFineInformationReceivedAt();
    }

    TString TFineReceiveTimeFilter::GetFieldTs() const {
        return "fine_information_received_at";
    }

    TInstant TFineChargeTimeFilter::GetActualTimestamp(const TAutocodeFineEntry& entry) const {
        return entry.GetChargedAt();
    }

    TString TFineChargeTimeFilter::GetFieldTs() const {
        return "charged_at";
    }

    TFineUnknownArticleFilter::TFineUnknownArticleFilter(TAtomicSharedPtr<TFineArticleMatcher> matcherPtr)
        : MatcherPtr(matcherPtr)
    {
        CHECK_WITH_LOG(!!matcherPtr);
    }

    bool TFineUnknownArticleFilter::Match(const TAutocodeFineEntry& entry) const {
        // article code can be suggested or set explicitly, so there are two cases:
        //   1. unknown article koap; 2. unrecognizable article code
        const bool isArticleUnknown = !entry.GetArticleKoap();
        if (isArticleUnknown) {
            return true;
        }
        TString articleCode = entry.GetArticleCode();
        const bool isArticleUnrecognizable = !articleCode && !MatcherPtr->DetermineArticle(entry.GetArticleKoap(), articleCode);
        return isArticleUnrecognizable;
    }

    void TFineUnknownArticleFilter::PatchQuery(NSQL::TQueryOptions& /* options */) const {
    }

    TFineArticleFilter::TFineArticleFilter(TAtomicSharedPtr<TFineArticleMatcher> matcherPtr, const TSet<TString>& articles, const bool doExclude)
        : MatcherPtr(matcherPtr)
        , Articles(articles)
        , DoExclude(doExclude)
    {
        CHECK_WITH_LOG(!!matcherPtr);
    }

    bool TFineArticleFilter::Match(const TAutocodeFineEntry& entry) const {
        TString articleCode = entry.GetArticleCode();
        if (!articleCode && !MatcherPtr->DetermineArticle(entry.GetArticleKoap(), articleCode)) {
            ERROR_LOG << "Unknown fine article: " << entry.GetRulingNumber() << "; car id: " << entry.GetCarId() << Endl;
            return false;
        }
        return DoExclude ^ Articles.contains(articleCode);
    }

    void TFineArticleFilter::PatchQuery(NSQL::TQueryOptions& options) const {
        AppendCustomQueryCondition(options, "meta_info::json->>'article_code' IS NOT NULL");
    }
}
