#pragma once

#include <drive/backend/fines/autocode/entries.h>

#include <rtline/library/storage/sql/query.h>


namespace NDrive {
    class TScheme;
}

namespace NDrive::NFine {
    class TFineArticleMatcher;

    class IFineFilter {
    public:
        using TPtr = TAtomicSharedPtr<IFineFilter>;

        virtual bool Match(const TAutocodeFineEntry& entry) const = 0;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const = 0;
        virtual ~IFineFilter() = default;
    };

    class TFineFilterGroup {
    public:
        TFineFilterGroup() = default;
        TFineFilterGroup(const TFineFilterGroup&) = default;
        TFineFilterGroup(std::initializer_list<IFineFilter::TPtr> il)
            : Filters(il)
        {
        }

        template<typename TIter>
        TFineFilterGroup(TIter first, TIter last)
            : Filters(first, last)
        {
        }

        void Append(IFineFilter::TPtr filter);
        bool Match(const TAutocodeFineEntry& entry) const;
        void PatchQuery(NSQL::TQueryOptions& options) const;

    private:
        TVector<IFineFilter::TPtr> Filters;
    };

    class TRejectFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineChargeableFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineNotChargedFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineChargedFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineChargeNotPassedFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineCameraFixationFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineExplicitPhotoFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineAbsentDetailedViolationDocumentFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineHasDetailedViolationDocumentFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineExplicitDecreeFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineUnknownIncludeToBillInfoFilter : public IFineFilter {
    public:
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;
    };

    class TFineSumToPayCentsFilter : public IFineFilter {
    public:
        explicit TFineSumToPayCentsFilter(const i64 maxAmount);
        TFineSumToPayCentsFilter(const i64 minAmount, const i64 maxAmount);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const i64 MinAmount;
        const i64 MaxAmount;
    };

    class TFineSerialIdFilter : public IFineFilter {
    public:
        explicit TFineSerialIdFilter(const ui32 minSerialId, const ui32 maxSerialId = std::numeric_limits<ui32>::max());
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const ui32 MinSerialId;
        const ui32 MaxSerialId;
    };

    class TSourceKeyFilter : public IFineFilter {
    public:
        explicit TSourceKeyFilter(const i64 id);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const i64 SourceKey;
    };

    class TFineUserFilter : public IFineFilter {
    public:
        explicit TFineUserFilter(const TString& userId);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TString UserId;
    };

    class TFineMultipleUsersFilter : public IFineFilter {
    public:
        explicit TFineMultipleUsersFilter(const TSet<TString>& userIds, const bool contains = true);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TSet<TString> UserIds;
        const bool Contains;
    };

    class TFineCarFilter : public IFineFilter {
    public:
        explicit TFineCarFilter(const TString& carId);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TString CarId;
    };

    class TFineMultipleSessionsFilter : public IFineFilter {
    public:
        explicit TFineMultipleSessionsFilter(const TSet<TString>& sessionIds);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TSet<TString> SessionIds;
    };

    class TFineMultipleCarsFilter : public IFineFilter {
    public:
        explicit TFineMultipleCarsFilter(const TSet<TString>& carIds);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TSet<TString> CarIds;
    };

    class TFineIdFilter : public IFineFilter {
    public:
        explicit TFineIdFilter(const TString& fineId);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TString FineId;
    };

    class TMultipleFineIdsFilter : public IFineFilter {
    public:
        template <typename C>
        explicit TMultipleFineIdsFilter(const C& fineIds)
            : FineIds(std::begin(fineIds), std::end(fineIds))
        {
        }

        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TSet<TString> FineIds;
    };

    class TFineRulingNumberFilter : public IFineFilter {
    public:
        explicit TFineRulingNumberFilter(const TString& rulingNumber);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TString RulingNumber;
    };

    class TFineMultipleRulingNumbersFilter : public IFineFilter {
    public:
        explicit TFineMultipleRulingNumbersFilter(const TSet<TString>& rulingNumbers);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TSet<TString> RulingNumbers;
    };

    class TFineSourceTypeFilter : public IFineFilter {
    public:
        explicit TFineSourceTypeFilter(const ESourceType& sourceType);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        const TString SourceType;
    };

    class TFineMultipleSourcesTypeFilter : public IFineFilter {
    public:
        explicit TFineMultipleSourcesTypeFilter(const TSet<ESourceType>& sourceTypes);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        TSet<TString> SourceTypes;
    };

    class TFineDateTimeFilter : public IFineFilter {
    public:
        TFineDateTimeFilter(const TInstant since = TInstant::Zero(), const TInstant until = TInstant::Zero());
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    protected:
        virtual TInstant GetActualTimestamp(const TAutocodeFineEntry& entry) const = 0;
        virtual TString GetFieldTs() const = 0;

        const TInstant Since;
        const TInstant Until;
    };

    class IDateTimeFilterConfig {
        R_FIELD(TInstant, Since);
        R_FIELD(TInstant, Until);
        R_FIELD(TDuration, For, TDuration::Max());
        R_FIELD(TDuration, To, TDuration::Zero());

    public:
        virtual ~IDateTimeFilterConfig() = default;

        virtual IFineFilter::TPtr GetFilter() const final;

        static NDrive::TScheme GetScheme();

        bool operator !() const;
        bool HasRestrictions() const;

        bool DeserializeFromJson(const NJson::TJsonValue& data);
        NJson::TJsonValue SerializeToJson() const;

    protected:
        virtual IFineFilter::TPtr DoGetFilter(const TInstant since, const TInstant until) const = 0;
    };

    template <typename T>
    class TDateFilterConfig : public IDateTimeFilterConfig {
        virtual IFineFilter::TPtr DoGetFilter(const TInstant since, const TInstant until) const override {
            return MakeAtomicShared<T>(since, until);
        }
    };

    class TFineRulingDateFilter : public TFineDateTimeFilter {
        using TBase = TFineDateTimeFilter;

    public:
        using TBase::TBase;

        virtual TInstant GetActualTimestamp(const TAutocodeFineEntry& entry) const override;
        virtual TString GetFieldTs() const override;
    };

    using TRulingDateFilterConfig = TDateFilterConfig<TFineRulingDateFilter>;

    class TFineViolationTimeFilter : public TFineDateTimeFilter {
        using TBase = TFineDateTimeFilter;

    public:
        using TBase::TBase;

        virtual TInstant GetActualTimestamp(const TAutocodeFineEntry& entry) const override;
        virtual TString GetFieldTs() const override;
    };

    using TViolationTimeFilterConfig = TDateFilterConfig<TFineViolationTimeFilter>;

    class TFineReceiveTimeFilter : public TFineDateTimeFilter {
        using TBase = TFineDateTimeFilter;

    public:
        using TBase::TBase;

        virtual TInstant GetActualTimestamp(const TAutocodeFineEntry& entry) const override;
        virtual TString GetFieldTs() const override;
    };

    using TFineReceiveTimeFilterConfig = TDateFilterConfig<TFineReceiveTimeFilter>;

    class TFineChargeTimeFilter : public TFineDateTimeFilter {
        using TBase = TFineDateTimeFilter;

    public:
        using TBase::TBase;

        virtual TInstant GetActualTimestamp(const TAutocodeFineEntry& entry) const override;
        virtual TString GetFieldTs() const override;
    };

    using TFineChargeTimeFilterConfig = TDateFilterConfig<TFineChargeTimeFilter>;

    class TFineUnknownArticleFilter : public IFineFilter {
    public:
        TFineUnknownArticleFilter(TAtomicSharedPtr<TFineArticleMatcher> matcherPtr);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& /* options */) const override;

    private:
        TAtomicSharedPtr<TFineArticleMatcher> MatcherPtr;
    };

    class TFineArticleFilter : public IFineFilter {
    public:
        TFineArticleFilter(TAtomicSharedPtr<TFineArticleMatcher> matcherPtr, const TSet<TString>& articles, const bool doExclude = false);
        virtual bool Match(const TAutocodeFineEntry& entry) const override;
        virtual void PatchQuery(NSQL::TQueryOptions& options) const override;

    private:
        TAtomicSharedPtr<TFineArticleMatcher> MatcherPtr;
        const TSet<TString> Articles;
        const bool DoExclude;
    };
}
